import React, { useState, useEffect } from 'react';
import { 
  Plus, 
  Edit, 
  Trash2, 
  Eye, 
  EyeOff, 
  Shield, 
  UserPlus, 
  Search,
  Filter,
  MoreVertical,
  CheckCircle,
  XCircle,
  X,
  Users,
  Settings,
  BarChart3,
  CreditCard,
  Bell,
  FileText
} from 'lucide-react';
import { Alert, AlertColor, Snackbar } from '@mui/material';
import { NavLink, Outlet } from 'react-router-dom';

interface Admin {
  id: string;
  firstName: string;
  lastName: string;
  email: string;
  username: string;
  role: 'MotherAdmin' | 'Admin' | 'SubAdmin' | 'Master' | 'Agent' | 'SubAgent' | 'User';
  password: string;
  status: 'active' | 'inactive';
  createdAt: string;
  lastLogin?: string;
}

interface SnackbarState {
  open: boolean;
  message: string;
  severity: AlertColor;
}

interface DeleteConfirmState {
  open: boolean;
  adminId: string | null;
  adminName: string;
}

interface TabItem {
  id: string;
  label: string;
  icon: React.ReactNode;
  component: React.ReactNode;
}

interface FormData {
  firstName: string;
  lastName: string;
  email: string;
  username: string;
  role: 'MotherAdmin' | 'Admin' | 'SubAdmin' | 'Master' | 'Agent' | 'SubAgent' | 'User';
  password: string;
  confirmPassword: string;
}

interface FormErrors {
  [key: string]: string;
}

const ROLE_HIERARCHY = {
  'MotherAdmin': { level: 0, color: 'bg-black', description: 'Supreme admin access' },
  'Admin': { level: 1, color: 'bg-red-500', description: 'Full system access' },
  'SubAdmin': { level: 2, color: 'bg-orange-500', description: 'Limited admin access' },
  'Master': { level: 3, color: 'bg-blue-500', description: 'Master level control' },
  'Agent': { level: 4, color: 'bg-green-500', description: 'Agent operations' },
  'SubAgent': { level: 5, color: 'bg-purple-500', description: 'Basic operations' },
  'User': { level: 6, color: 'bg-gray-500', description: 'Standard user access' }
};

// Toggle Switch Component
const ToggleSwitch = ({ 
  checked, 
  onChange, 
  disabled = false 
}: { 
  checked: boolean; 
  onChange: (checked: boolean) => void; 
  disabled?: boolean;
}) => {
  return (
    <button
      type="button"
      role="switch"
      aria-checked={checked}
      disabled={disabled}
      onClick={() => onChange(!checked)}
      className={`
        relative inline-flex h-5 w-9 shrink-0 cursor-pointer rounded-full border-2 border-transparent 
        transition-colors duration-200 ease-in-out focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2
        ${checked 
          ? 'bg-green-600 hover:bg-green-700' 
          : 'bg-gray-200 hover:bg-gray-300'
        }
        ${disabled ? 'opacity-50 cursor-not-allowed' : ''}
      `}
    >
      <span
        aria-hidden="true"
        className={`
          pointer-events-none inline-block h-4 w-4 transform rounded-full bg-white shadow-lg ring-0 
          transition duration-200 ease-in-out
          ${checked ? 'translate-x-4' : 'translate-x-0'}
        `}
      />
    </button>
  );
};

// User Management Component (current functionality)
const UserManagementTab = () => {
  const [admins, setAdmins] = useState<Admin[]>([]);
  const [showAddForm, setShowAddForm] = useState(false);
  const [editingAdmin, setEditingAdmin] = useState<Admin | null>(null);
  const [searchTerm, setSearchTerm] = useState('');
  const [roleFilter, setRoleFilter] = useState<string>('all');
  const [statusFilter, setStatusFilter] = useState<string>('all');
  const [showPassword, setShowPassword] = useState(false);
  const [showConfirmPassword, setShowConfirmPassword] = useState(false);
  const [snackbar, setSnackbar] = useState<SnackbarState>({ 
    open: false, 
    message: '', 
    severity: 'success' 
  });

  const [deleteConfirm, setDeleteConfirm] = useState<DeleteConfirmState>({
    open: false,
    adminId: null,
    adminName: ''
  });

  const [formData, setFormData] = useState<FormData>({
    firstName: '',
    lastName: '',
    email: '',
    username: '',
    role: 'User',
    password: '',
    confirmPassword: ''
  });

  const [errors, setErrors] = useState<FormErrors>({});

  // Initialize with some sample data
  useEffect(() => {
    const sampleAdmins: Admin[] = [
      {
        id: '1',
        firstName: 'John',
        lastName: 'Smith',
        email: 'john.smith@betlover365.com',
        username: 'jsmith',
        role: 'Admin',
        password: 'Admin@123',
        status: 'active',
        createdAt: '2024-01-15',
        lastLogin: '2024-01-20'
      },
      {
        id: '2',
        firstName: 'Sarah',
        lastName: 'Johnson',
        email: 'sarah.johnson@betlover365.com',
        username: 'sjohnson',
        role: 'SubAdmin',
        password: 'SubAdmin@123',
        status: 'active',
        createdAt: '2024-01-16',
        lastLogin: '2024-01-19'
      },
      {
        id: '3',
        firstName: 'Mike',
        lastName: 'Davis',
        email: 'mike.davis@betlover365.com',
        username: 'mdavis',
        role: 'User',
        password: 'User@123',
        status: 'active',
        createdAt: '2024-01-17',
        lastLogin: '2024-01-18'
      }
    ];
    
    // Load from localStorage or use sample data
    const storedAdmins = localStorage.getItem('adminUsers');
    console.log('Loading from localStorage:', storedAdmins); // Debug log
    if (storedAdmins) {
      const parsedAdmins = JSON.parse(storedAdmins);
      console.log('Parsed admins:', parsedAdmins); // Debug log
      setAdmins(parsedAdmins);
    } else {
      console.log('No stored admins, using sample data'); // Debug log
      setAdmins(sampleAdmins);
      localStorage.setItem('adminUsers', JSON.stringify(sampleAdmins));
    }
  }, []);

  // Save admins to localStorage whenever admins change
  useEffect(() => {
    if (admins.length > 0) {
      console.log('Saving to localStorage:', admins); // Debug log
      localStorage.setItem('adminUsers', JSON.stringify(admins));
    }
  }, [admins]);

  // Handle ESC key to close modal
  useEffect(() => {
    const handleEscKey = (event: KeyboardEvent) => {
      if (event.key === 'Escape' && showAddForm) {
        resetForm();
      }
    };

    document.addEventListener('keydown', handleEscKey);
    return () => {
      document.removeEventListener('keydown', handleEscKey);
    };
  }, [showAddForm]);

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value } = e.target;
    setFormData(prev => ({ ...prev, [name]: value }));
    
    // Clear error when user starts typing
    if (errors[name]) {
      setErrors(prev => ({ ...prev, [name]: '' }));
    }
  };

  const validateForm = (): boolean => {
    const newErrors: FormErrors = {};

    if (!formData.firstName.trim()) newErrors.firstName = 'First name is required';
    if (!formData.lastName.trim()) newErrors.lastName = 'Last name is required';
    if (!formData.email.trim()) newErrors.email = 'Email is required';
    else if (!/\S+@\S+\.\S+/.test(formData.email)) newErrors.email = 'Invalid email format';
    if (!formData.username.trim()) newErrors.username = 'Username is required';
    if (!formData.password) newErrors.password = 'Password is required';
    else if (formData.password.length < 6) newErrors.password = 'Password must be at least 6 characters';
    if (formData.password !== formData.confirmPassword) {
      newErrors.confirmPassword = 'Passwords do not match';
    }

    // Check for duplicate email or username
    const isDuplicate = admins.some(admin => 
      (admin.email === formData.email || admin.username === formData.username) &&
      admin.id !== editingAdmin?.id
    );
    
    if (isDuplicate) {
      newErrors.email = 'Email or username already exists';
      newErrors.username = 'Email or username already exists';
    }

    setErrors(newErrors);
    return Object.keys(newErrors).length === 0;
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!validateForm()) return;

    const newAdmin: Admin = {
      id: editingAdmin?.id || Date.now().toString(),
      ...formData,
      status: 'active',
      createdAt: editingAdmin?.createdAt || new Date().toISOString().split('T')[0],
      lastLogin: editingAdmin?.lastLogin
    };

    if (editingAdmin) {
      console.log('Updating admin:', newAdmin); // Debug log
      setAdmins(prev => {
        const updated = prev.map(admin => admin.id === editingAdmin.id ? newAdmin : admin);
        console.log('Updated admins list:', updated); // Debug log
        return updated;
      });
      setSnackbar({
        open: true,
        message: 'Client updated successfully!',
        severity: 'success'
      });
    } else {
      console.log('Adding new admin:', newAdmin); // Debug log
      setAdmins(prev => {
        const updated = [...prev, newAdmin];
        console.log('New admins list:', updated); // Debug log
        return updated;
      });
      setSnackbar({
        open: true,
        message: 'Client added successfully!',
        severity: 'success'
      });
    }

    // Force immediate save to localStorage
    setTimeout(() => {
      const currentAdmins = JSON.parse(localStorage.getItem('adminUsers') || '[]');
      console.log('Current localStorage adminUsers:', currentAdmins); // Debug log
    }, 100);

    resetForm();
  };

  const resetForm = () => {
    setFormData({
      firstName: '',
      lastName: '',
      email: '',
      username: '',
      role: 'User',
      password: '',
      confirmPassword: ''
    });
    setErrors({});
    setShowAddForm(false);
    setEditingAdmin(null);
    setShowPassword(false);
    setShowConfirmPassword(false);
  };

  const handleEdit = (admin: Admin) => {
    setFormData({
      firstName: admin.firstName,
      lastName: admin.lastName,
      email: admin.email,
      username: admin.username,
      role: admin.role,
      password: admin.password,
      confirmPassword: admin.password
    });
    setEditingAdmin(admin);
    setShowAddForm(true);
  };

  const handleDelete = (adminId: string) => {
    const admin = admins.find(a => a.id === adminId);
    if (admin) {
      setDeleteConfirm({
        open: true,
        adminId,
        adminName: `${admin.firstName} ${admin.lastName}`
      });
    }
  };

  const confirmDelete = () => {
    if (deleteConfirm.adminId) {
      setAdmins(prev => prev.filter(admin => admin.id !== deleteConfirm.adminId));
      setSnackbar({
        open: true,
        message: 'Client deleted successfully!',
        severity: 'success'
      });
      setDeleteConfirm({ open: false, adminId: null, adminName: '' });
    }
  };

  const cancelDelete = () => {
    setDeleteConfirm({ open: false, adminId: null, adminName: '' });
  };

  const handleBulkStatusChange = (status: 'active' | 'inactive') => {
    const filteredAdminIds = filteredAdmins.map(admin => admin.id);
    setAdmins(prev => prev.map(admin => 
      filteredAdminIds.includes(admin.id) 
        ? { ...admin, status }
        : admin
    ));
    setSnackbar({
      open: true,
      message: `All visible clients marked as ${status}!`,
      severity: 'success'
    });
  };

  const handleIndividualStatusChange = (adminId: string, status: 'active' | 'inactive') => {
    setAdmins(prev => prev.map(admin => 
      admin.id === adminId 
        ? { ...admin, status }
        : admin
    ));
    setSnackbar({
      open: true,
      message: `Client marked as ${status}!`,
      severity: 'success'
    });
  };

  const toggleStatus = (adminId: string) => {
    setAdmins(prev => prev.map(admin => 
      admin.id === adminId 
        ? { ...admin, status: admin.status === 'active' ? 'inactive' : 'active' }
        : admin
    ));
  };

  const handleCloseSnackbar = () => {
    setSnackbar(prev => ({ ...prev, open: false }));
  };

  // Get current user role and implement hierarchy-based access
  const userRole = localStorage.getItem('userRole');
  const currentUserLevel = ROLE_HIERARCHY[userRole as keyof typeof ROLE_HIERARCHY]?.level ?? 999;
  
  // Function to get roles that current user can manage (only levels below current user)
  const getManageableRoles = () => {
    return Object.keys(ROLE_HIERARCHY).filter(role => 
      ROLE_HIERARCHY[role as keyof typeof ROLE_HIERARCHY].level > currentUserLevel
    );
  };

  // Function to filter visible admins based on hierarchy
  const getVisibleAdmins = () => {
    const manageableRoles = getManageableRoles();
    return admins.filter(admin => manageableRoles.includes(admin.role));
  };

  // Get filtered admins based on hierarchy and search criteria
  const getFilteredAdmins = () => {
    const manageableRoles = getManageableRoles();
    const visibleAdmins = admins.filter(admin => manageableRoles.includes(admin.role));
    
    return visibleAdmins.filter(admin => {
      const matchesSearch = admin.firstName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                           admin.lastName.toLowerCase().includes(searchTerm.toLowerCase()) ||
                           admin.email.toLowerCase().includes(searchTerm.toLowerCase()) ||
                           admin.username.toLowerCase().includes(searchTerm.toLowerCase());
      
      const matchesRole = roleFilter === 'all' || admin.role === roleFilter;
      const matchesStatus = statusFilter === 'all' || admin.status === statusFilter;
      
      return matchesSearch && matchesRole && matchesStatus;
    });
  };

  const filteredAdmins = getFilteredAdmins();

  // Check if user has no manageable roles (like User role)
  if (getManageableRoles().length === 0) {
    return (
      <div className="ml-0 lg:ml-[256px] p-2 max-w-7xl mx-auto">
        <div className="flex items-center justify-center h-64">
          <div className="text-center">
            <Shield className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No Management Access</h3>
            <p className="text-gray-500">Your role ({userRole}) does not have permission to manage other users.</p>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="ml-0 lg:ml-[256px] p-2 max-w-9xl mx-auto">
      {/* Header */}
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-white flex items-center">
          <Shield className="w-8 h-8 mr-3 text-white" />
          Client Management
        </h1>
      </div>

      {/* Controls */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200 p-6 mb-6">
        <div className="flex flex-col lg:flex-row lg:items-center lg:justify-between gap-4">
          <div className="flex flex-col sm:flex-row gap-4">
            {/* Search */}
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <input
                type="text"
                placeholder="Search clients..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
            </div>

            {/* Role Filter */}
            <div className="relative">
              <Filter className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <select
                value={roleFilter}
                onChange={(e) => setRoleFilter(e.target.value)}
                className="pl-10 pr-8 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent appearance-none bg-white"
              >
                <option value="all">All Roles</option>
                {getManageableRoles().map(role => (
                  <option key={role} value={role}>{role}</option>
                ))}
              </select>
            </div>

            {/* Status Filter */}
            <div className="relative">
              <CheckCircle className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <select
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
                className="pl-10 pr-8 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent appearance-none bg-white"
              >
                <option value="all">All Status</option>
                <option value="active">Active</option>
                <option value="inactive">Inactive</option>
                <option value="pending">Pending</option>
              </select>
            </div>
          </div>

          {/* Add Button */}
          <button
            onClick={() => setShowAddForm(true)}
            className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
          >
            <UserPlus className="w-5 h-5 mr-2" />
            Add Client
          </button>
        </div>
      </div>

      {/* Modal Overlay */}
      {showAddForm && (
        <div 
          className="fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center p-4"
          style={{ zIndex: 9999 }}
          onClick={resetForm}
        >
          <div 
            className="bg-white rounded-xl shadow-2xl border border-gray-200 w-full max-w-2xl max-h-[90vh] overflow-y-auto"
            onClick={(e) => e.stopPropagation()}
          >
            {/* Modal Header */}
            <div className="flex items-center justify-between p-6 border-b border-gray-200 bg-white">
              <h2 className="text-xl font-semibold text-gray-900">
                {editingAdmin ? 'Edit Client' : 'Add New Client'}
              </h2>
              <button
                onClick={resetForm}
                className="text-gray-400 hover:text-gray-600 p-1 rounded-full hover:bg-gray-100 transition-colors"
              >
                <X className="w-6 h-6" />
              </button>
            </div>
            
            {/* Modal Body */}
            <div className="p-6">
              <form onSubmit={handleSubmit} className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {/* First Name */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  First Name *
                </label>
                <input
                  type="text"
                  name="firstName"
                  value={formData.firstName}
                  onChange={handleInputChange}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Enter first name"
                />
                {errors.firstName && (
                  <p className="text-red-500 text-sm mt-1">{errors.firstName}</p>
                )}
              </div>

              {/* Last Name */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Last Name *
                </label>
                <input
                  type="text"
                  name="lastName"
                  value={formData.lastName}
                  onChange={handleInputChange}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Enter last name"
                />
                {errors.lastName && (
                  <p className="text-red-500 text-sm mt-1">{errors.lastName}</p>
                )}
              </div>

              {/* Email */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Email *
                </label>
                <input
                  type="email"
                  name="email"
                  value={formData.email}
                  onChange={handleInputChange}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Enter email address"
                />
                {errors.email && (
                  <p className="text-red-500 text-sm mt-1">{errors.email}</p>
                )}
              </div>

              {/* Username */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Username *
                </label>
                <input
                  type="text"
                  name="username"
                  value={formData.username}
                  onChange={handleInputChange}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                  placeholder="Enter username"
                />
                {errors.username && (
                  <p className="text-red-500 text-sm mt-1">{errors.username}</p>
                )}
              </div>

              {/* Role */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Role *
                </label>
                <select
                  name="role"
                  value={formData.role}
                  onChange={handleInputChange}
                  className="w-full px-3 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                >
                  {getManageableRoles().map(role => {
                    const info = ROLE_HIERARCHY[role as keyof typeof ROLE_HIERARCHY];
                    return (
                      <option key={role} value={role}>
                        {role} - {info.description}
                      </option>
                    );
                  })}
                </select>
              </div>

              {/* Password */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Password *
                </label>
                <div className="relative">
                  <input
                    type={showPassword ? 'text' : 'password'}
                    name="password"
                    value={formData.password}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 pr-10 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    placeholder="Enter password"
                  />
                  <button
                    type="button"
                    onClick={() => setShowPassword(!showPassword)}
                    className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600"
                  >
                    {showPassword ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
                  </button>
                </div>
                {errors.password && (
                  <p className="text-red-500 text-sm mt-1">{errors.password}</p>
                )}
              </div>

              {/* Confirm Password */}
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Confirm Password *
                </label>
                <div className="relative">
                  <input
                    type={showConfirmPassword ? 'text' : 'password'}
                    name="confirmPassword"
                    value={formData.confirmPassword}
                    onChange={handleInputChange}
                    className="w-full px-3 py-2 pr-10 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                    placeholder="Confirm password"
                  />
                  <button
                    type="button"
                    onClick={() => setShowConfirmPassword(!showConfirmPassword)}
                    className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600"
                  >
                    {showConfirmPassword ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
                  </button>
                </div>
                {errors.confirmPassword && (
                  <p className="text-red-500 text-sm mt-1">{errors.confirmPassword}</p>
                )}
              </div>
            </div>

                <div className="flex justify-end space-x-3 pt-4">
                  <button
                    type="button"
                    onClick={resetForm}
                    className="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition-colors"
                  >
                    Cancel
                  </button>
                  <button
                    type="submit"
                    className="px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                  >
                    {editingAdmin ? 'Update Client' : 'Add Client'}
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      )}

      {/* Admins Table */}
      <div className="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
        <div className="overflow-x-auto">
          <table className="w-full">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Client Details
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Role
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Status
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Last Login
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Actions
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {filteredAdmins.map((admin) => (
                <tr key={admin.id} className="hover:bg-gray-50">
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div>
                      <div className="text-sm font-medium text-gray-900">
                        {admin.firstName} {admin.lastName}
                      </div>
                      <div className="text-sm text-gray-500">{admin.email}</div>
                      <div className="text-xs text-gray-400">@{admin.username}</div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium text-white ${ROLE_HIERARCHY[admin.role].color}`}>
                      {admin.role}
                    </span>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <button
                      onClick={() => toggleStatus(admin.id)}
                      className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                        admin.status === 'active' 
                          ? 'bg-green-100 text-green-800' 
                          : 'bg-red-100 text-red-800'
                      }`}
                    >
                      {admin.status === 'active' ? (
                        <CheckCircle className="w-3 h-3 mr-1" />
                      ) : (
                        <XCircle className="w-3 h-3 mr-1" />
                      )}
                      {admin.status}
                    </button>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                    {admin.lastLogin || 'Never'}
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                    <div className="flex items-center space-x-3">
                      <button
                        onClick={() => handleEdit(admin)}
                        className="text-blue-600 hover:text-blue-900 p-1 rounded-full hover:bg-blue-50"
                        title="Edit client"
                      >
                        <Edit className="w-4 h-4" />
                      </button>
                      
                      <div className="flex items-center space-x-2">
                        <span className="text-xs text-gray-500">
                          {admin.status === 'active' ? 'Active' : 'Inactive'}
                        </span>
                        <ToggleSwitch
                          checked={admin.status === 'active'}
                          onChange={(checked) => handleIndividualStatusChange(admin.id, checked ? 'active' : 'inactive')}
                        />
                      </div>
                      
                      <button
                        onClick={() => handleDelete(admin.id)}
                        className="text-red-600 hover:text-red-900 p-1 rounded-full hover:bg-red-50"
                        title="Delete client"
                      >
                        <Trash2 className="w-4 h-4" />
                      </button>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>

        {filteredAdmins.length === 0 && (
          <div className="text-center py-12">
            <Shield className="w-12 h-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">No clients found</h3>
            <p className="text-gray-500">Try adjusting your search or filter criteria.</p>
          </div>
        )}
      </div>

      {/* Delete Confirmation Modal */}
      {deleteConfirm.open && (
        <div 
          className="fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center p-4"
          style={{ zIndex: 10000 }}
          onClick={cancelDelete}
        >
          <div 
            className="bg-white rounded-xl shadow-2xl border border-gray-200 w-full max-w-md"
            onClick={(e) => e.stopPropagation()}
          >
            {/* Modal Header */}
            <div className="flex items-center justify-between p-6 border-b border-gray-200">
              <div className="flex items-center">
                <div className="flex-shrink-0 w-10 h-10 mx-auto bg-red-100 rounded-full flex items-center justify-center">
                  <Trash2 className="w-6 h-6 text-red-600" />
                </div>
                <div className="ml-4">
                  <h3 className="text-lg font-medium text-gray-900">
                    Delete Client
                  </h3>
                </div>
              </div>
              <button
                onClick={cancelDelete}
                className="text-gray-400 hover:text-gray-600 p-1 rounded-full hover:bg-gray-100 transition-colors"
              >
                <X className="w-6 h-6" />
              </button>
            </div>
            
            {/* Modal Body */}
            <div className="p-6">
              <div className="text-sm text-gray-500 mb-4">
                Are you sure you want to delete <strong className="text-gray-900">{deleteConfirm.adminName}</strong>? 
                This action cannot be undone and will permanently remove this client from the system.
              </div>
              
              <div className="bg-red-50 border border-red-200 rounded-lg p-4 mb-4">
                <div className="flex">
                  <div className="flex-shrink-0">
                    <XCircle className="h-5 w-5 text-red-400" />
                  </div>
                  <div className="ml-3">
                    <h3 className="text-sm font-medium text-red-800">
                      Warning
                    </h3>
                    <div className="mt-2 text-sm text-red-700">
                      <p>
                        This will permanently delete the client and all associated data. 
                        This action cannot be reversed.
                      </p>
                    </div>
                  </div>
                </div>
              </div>
            </div>
            
            {/* Modal Footer */}
            <div className="flex justify-end space-x-3 p-6 border-t border-gray-200 bg-gray-50 rounded-b-xl">
              <button
                onClick={cancelDelete}
                className="px-4 py-2 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-100 transition-colors font-medium"
              >
                Cancel
              </button>
              <button
                onClick={confirmDelete}
                className="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition-colors font-medium flex items-center"
              >
                <Trash2 className="w-4 h-4 mr-2" />
                Delete Client
              </button>
            </div>
          </div>
        </div>
      )}

      {/* Snackbar */}
      <Snackbar open={snackbar.open} autoHideDuration={3000} onClose={handleCloseSnackbar}>
        <Alert onClose={handleCloseSnackbar} severity={snackbar.severity} sx={{ width: '100%' }}>
          {snackbar.message}
        </Alert>
      </Snackbar>
    </div>
  );
};

// Dashboard Tab Component
const DashboardTab = () => {
  return (
    <div className="p-8 text-center">
      <BarChart3 className="w-16 h-16 text-blue-500 mx-auto mb-4" />
      <h3 className="text-xl font-semibold text-gray-800 mb-2">Dashboard Overview</h3>
      <p className="text-gray-600">Dashboard content will be implemented here.</p>
    </div>
  );
};

// Settings Tab Component
const SettingsTab = () => {
  return (
    <div className="p-8 text-center">
      <Settings className="w-16 h-16 text-blue-500 mx-auto mb-4" />
      <h3 className="text-xl font-semibold text-gray-800 mb-2">Settings</h3>
      <p className="text-gray-600">Settings configuration will be implemented here.</p>
    </div>
  );
};

// My Account Tab Component
const MyAccountTab = () => {
  return (
    <div className="p-8 text-center">
      <Shield className="w-16 h-16 text-blue-500 mx-auto mb-4" />
      <h3 className="text-xl font-semibold text-gray-800 mb-2">My Account</h3>
      <p className="text-gray-600">Account settings and profile management.</p>
    </div>
  );
};

// BetList Tab Component
const BetListTab = () => {
  return (
    <div className="p-8 text-center">
      <FileText className="w-16 h-16 text-blue-500 mx-auto mb-4" />
      <h3 className="text-xl font-semibold text-gray-800 mb-2">BetList</h3>
      <p className="text-gray-600">Manage and view betting lists.</p>
    </div>
  );
};

// BetListLive Tab Component
const BetListLiveTab = () => {
  return (
    <div className="p-8 text-center">
      <BarChart3 className="w-16 h-16 text-blue-500 mx-auto mb-4" />
      <h3 className="text-xl font-semibold text-gray-800 mb-2">BetListLive</h3>
      <p className="text-gray-600">Live betting lists and real-time updates.</p>
    </div>
  );
};

// Banking Tab Component
const BankingTab = () => {
  return (
    <div className="p-8 text-center">
      <CreditCard className="w-16 h-16 text-blue-500 mx-auto mb-4" />
      <h3 className="text-xl font-semibold text-gray-800 mb-2">Banking</h3>
      <p className="text-gray-600">Financial operations and banking management.</p>
    </div>
  );
};

// Casino Tab Component
const CasinoTab = () => {
  return (
    <div className="p-8 text-center">
      <Plus className="w-16 h-16 text-blue-500 mx-auto mb-4" />
      <h3 className="text-xl font-semibold text-gray-800 mb-2">Casino</h3>
      <p className="text-gray-600">Casino games and management.</p>
    </div>
  );
};

// Risk Management Tab Component
const RiskManagementTab = () => {
  return (
    <div className="p-8 text-center">
      <Shield className="w-16 h-16 text-blue-500 mx-auto mb-4" />
      <h3 className="text-xl font-semibold text-gray-800 mb-2">Risk Management</h3>
      <p className="text-gray-600">Risk assessment and management tools.</p>
    </div>
  );
};

// Import Tab Component
const ImportTab = () => {
  return (
    <div className="p-8 text-center">
      <Plus className="w-16 h-16 text-blue-500 mx-auto mb-4" />
      <h3 className="text-xl font-semibold text-gray-800 mb-2">Import</h3>
      <p className="text-gray-600">Data import and management tools.</p>
    </div>
  );
};

// Message Tab Component
const MessageTab = () => {
  return (
    <div className="p-8 text-center">
      <Bell className="w-16 h-16 text-blue-500 mx-auto mb-4" />
      <h3 className="text-xl font-semibold text-gray-800 mb-2">Message</h3>
      <p className="text-gray-600">Message center and notifications.</p>
    </div>
  );
};

// Game Center Tab Component
const GameCenterTab = () => {
  return (
    <div className="p-8 text-center">
      <Plus className="w-16 h-16 text-blue-500 mx-auto mb-4" />
      <h3 className="text-xl font-semibold text-gray-800 mb-2">Game Center</h3>
      <p className="text-gray-600">Game management and center operations.</p>
    </div>
  );
};

// Self Department Tab Component
const SelfDepartmentTab = () => {
  return (
    <div className="p-8 text-center">
      <Users className="w-16 h-16 text-blue-500 mx-auto mb-4" />
      <h3 className="text-xl font-semibold text-gray-800 mb-2">Self Department</h3>
      <p className="text-gray-600">Department management and operations.</p>
    </div>
  );
};

const tabs = [
  { id: 'dashboard', label: 'Dashboard', icon: <BarChart3 className="w-4 h-4" /> },
  { id: 'user', label: 'User', icon: <Users className="w-4 h-4" /> },
  // { id: 'setting', label: 'Setting', icon: <Settings className="w-4 h-4" /> },
  { id: 'myaccount', label: 'My Account', icon: <Shield className="w-4 h-4" /> },
  // { id: 'betlist', label: 'BetList', icon: <FileText className="w-4 h-4" /> },
  // { id: 'betlistlive', label: 'BetListLive', icon: <BarChart3 className="w-4 h-4" /> },
  { id: 'banking', label: 'Banking', icon: <CreditCard className="w-4 h-4" /> },
  // { id: 'casino', label: 'Casino', icon: <Plus className="w-4 h-4" /> },
  // { id: 'riskmanagement', label: 'Risk Management', icon: <Shield className="w-4 h-4" /> },
  // { id: 'import', label: 'Import', icon: <Plus className="w-4 h-4" /> },
  // { id: 'message', label: 'Message', icon: <Bell className="w-4 h-4" /> },
  // { id: 'gamecenter', label: 'Game Center', icon: <Plus className="w-4 h-4" /> },
  // { id: 'selfdepartment', label: 'Self department', icon: <Users className="w-4 h-4" /> },
];



const AdminManagementLayout = () => {
  return (
    <div className="ml-0 lg:ml-[258px] w-full md:w-[calc(100%-258px)] mt-2">
      <style>
        {`
          .scrollbar-hide {
            -ms-overflow-style: none;
            scrollbar-width: none;
          }
          .scrollbar-hide::-webkit-scrollbar {
            display: none;
          }
        `}
      </style>
      {/* Tab Navigation */}
      <div className="bg-gradient-to-r from-blue-600 via-blue-700 to-blue-800 shadow-lg border-b-2 border-blue-900">
        <div className="max-w-full">
          <nav className="flex overflow-x-auto scrollbar-hide">
            {tabs.map((tab) => (
              <NavLink
                key={tab.id}
                to={tab.id}
                className={({ isActive }) =>
                  `flex items-center space-x-2 px-4 py-3 text-sm font-medium whitespace-nowrap transition-all duration-200 min-w-fit ` +
                  (isActive
                    ? 'bg-white text-blue-700 border-b-4 border-blue-700 shadow-inner font-semibold'
                    : 'text-white hover:text-blue-100 hover:bg-blue-500/50') +
                  ' border-r border-blue-500/30 last:border-r-0'
                }
                end
              >
                {tab.icon}
                <span className="text-xs">{tab.label}</span>
              </NavLink>
            ))}
          </nav>
        </div>
      </div>
      {/* Tab Content */}
      <div className="bg-gray-100 min-h-screen">
        <div className="max-w-full p-0">
          <Outlet />
        </div>
      </div>
    </div>
  );
};

export default AdminManagementLayout; 