import React, { useState, useEffect, DragEvent } from "react";
import axios from "axios";
import { 
  Box, 
  Typography, 
  Card, 
  CardContent, 
  CardHeader, 
  Grid, 
  Button, 
  Input, 
  TextField,
  FormLabel, 
  Snackbar, 
  Alert, 
  IconButton, 
  useTheme,
  useMediaQuery,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Skeleton,
  Chip,
  Stack,
    Tooltip,          
} from "@mui/material";
import { 
  CloudUpload as CloudUploadIcon, 
  Close as CloseIcon, 
  ImageSearch as ImageSearchIcon,
  Delete as DeleteIcon,
  AddPhotoAlternate as AddPhotoAlternateIcon
} from "@mui/icons-material";

interface BannerImage {
  id: number;
  type: string;
  filename: string; // server key: "filename"
  base64: string;
}

const BASE_URL = "https://api.air1bet.com/api/banners";

const Banner: React.FC = () => {
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('sm'));

  const [banners, setBanners] = useState<BannerImage[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [selectedBanner, setSelectedBanner] = useState<{
    type: string;
    file: File | null;
    base64Preview?: string;
  } | null>(null);

  // NEW: Add panel toggle
  const [showAddPanel, setShowAddPanel] = useState(false);

  // NEW: create/add banner state
  const [newBannerType, setNewBannerType] = useState<string>("");
  const [newFiles, setNewFiles] = useState<File[]>([]);
  const [newFilePreviews, setNewFilePreviews] = useState<string[]>([]);
  const [isAdding, setIsAdding] = useState(false);
  const [isDragging, setIsDragging] = useState(false);

  // NEW: delete confirmation
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [bannerToDelete, setBannerToDelete] = useState<BannerImage | null>(null);
  const [isDeleting, setIsDeleting] = useState(false);

  const [snackbar, setSnackbar] = useState<{
    open: boolean;
    message: string;
    severity: 'success' | 'error';
  }>({
    open: false,
    message: '',
    severity: 'success'
  });
  const [previewDialogOpen, setPreviewDialogOpen] = useState(false);
  const [previewImage, setPreviewImage] = useState<string>('');

  useEffect(() => {
    fetchBanners();
  }, []);

  const fetchBanners = async () => {
    setIsLoading(true);
    try {
      const response = await axios.get(`${BASE_URL}/getAllImagesBinary`, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });
      if (response.data.status === 'success') {
        setBanners(response.data.images);
      } else {
        handleSnackbarOpen('Failed to fetch banners', 'error');
      }
    } catch (error) {
      handleSnackbarOpen('Failed to fetch banners', 'error');
    } finally {
      setIsLoading(false);
    }
  };

  const handleSnackbarOpen = (message: string, severity: 'success' | 'error' = 'success') => {
    setSnackbar({ open: true, message, severity });
  };

  const handleSnackbarClose = (event?: React.SyntheticEvent | Event, reason?: string) => {
    if (reason === 'clickaway') return;
    setSnackbar({ ...snackbar, open: false });
  };

  const handleBannerSelect = (banner: BannerImage) => {
    setSelectedBanner({ 
      type: banner.type, 
      file: null, 
      base64Preview: `data:image/png;base64,${banner.base64}` 
    });
  };

  const handleFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    if (event.target.files && event.target.files.length > 0) {
      const file = event.target.files[0];
      const reader = new FileReader();
      reader.onloadend = () => {
        setSelectedBanner(prev => ({
          type: prev?.type || '',
          file: file,
          base64Preview: reader.result as string
        }));
      };
      reader.readAsDataURL(file);
    }
  };

  const updateBanner = async () => {
    if (!selectedBanner || !selectedBanner.file) {
      handleSnackbarOpen('Please select a banner and an image', 'error');
      return;
    }

    const formData = new FormData();
    formData.append('type', selectedBanner.type);
    formData.append('image', selectedBanner.file);

    try {
      setIsLoading(true);
      await axios.post(`${BASE_URL}/updateBanner`, formData, {
        headers: { 
          'Content-Type': 'multipart/form-data',
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });

      handleSnackbarOpen(`Banner ${selectedBanner.type} updated successfully`);
      await fetchBanners();
      setSelectedBanner(null);
    } catch (error) {
      handleSnackbarOpen('Failed to update banner', 'error');
    } finally {
      setIsLoading(false);
    }
  };

  // ---------- Add banner (type + multiple photos) ----------
  const onNewFilesPicked = (filesLike: FileList | File[]) => {
    const files = Array.from(filesLike);
    const valid = files.filter(f => /image\/(png|jpe?g|webp)/i.test(f.type));
    const tooBig = files.filter(f => f.size > 8 * 1024 * 1024);
    if (tooBig.length) {
      handleSnackbarOpen(`Some files exceed 8MB and were skipped: ${tooBig.map(f => f.name).join(', ')}`, 'error');
    }
    const kept = valid.filter(f => f.size <= 8 * 1024 * 1024);

    // merge & dedupe by name+size
    const merged = [...newFiles, ...kept].filter((f, idx, arr) =>
      arr.findIndex(x => x.name === f.name && x.size === f.size) === idx
    );

    setNewFiles(merged);

    // previews
    const readers = kept.map(
      f =>
        new Promise<string>((resolve) => {
          const r = new FileReader();
          r.onload = () => resolve(r.result as string);
          r.readAsDataURL(f);
        })
    );
    Promise.all(readers).then(imgs => setNewFilePreviews(prev => [...prev, ...imgs]));
  };

  const handleNewInputChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    if (e.target.files?.length) onNewFilesPicked(e.target.files);
  };

  const handleDrop = (e: DragEvent<HTMLDivElement>) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(false);
    if (e.dataTransfer.files?.length) onNewFilesPicked(e.dataTransfer.files);
  };

  const addBanners = async () => {
    if (!newBannerType.trim()) {
      handleSnackbarOpen('Please enter a banner type', 'error');
      return;
    }
    if (newFiles.length === 0) {
      handleSnackbarOpen('Please add at least one image', 'error');
      return;
    }

    try {
      setIsAdding(true);

      for (const file of newFiles) {
        const fd = new FormData();
        fd.append('type', newBannerType.trim());
        fd.append('image', file);

        const resp = await axios.post(`${BASE_URL}/createBanner`, fd, {
          headers: {
            'Content-Type': 'multipart/form-data',
            'Authorization': `Bearer ${localStorage.getItem('token')}`
          }
        });

        if (resp.data?.status !== 'success') {
          throw new Error('Create banner failed');
        }
      }

      handleSnackbarOpen(`Added ${newFiles.length} image(s) to type "${newBannerType}"`);
      setNewBannerType("");
      setNewFiles([]);
      setNewFilePreviews([]);
      await fetchBanners();
      setShowAddPanel(false);
    } catch (err: any) {
      const msg = err?.response?.data?.message || 'Failed to add banner(s)';
      handleSnackbarOpen(msg, 'error');
    } finally {
      setIsAdding(false);
    }
  };

  const removeNewFileAt = (idx: number) => {
    setNewFiles(prev => prev.filter((_, i) => i !== idx));
    setNewFilePreviews(prev => prev.filter((_, i) => i !== idx));
  };

  const clearNewForm = () => {
    setNewBannerType("");
    setNewFiles([]);
    setNewFilePreviews([]);
  };
  // -------------------------------------------------------------

  const openPreviewDialog = (base64: string) => {
    setPreviewImage(`data:image/png;base64,${base64.replace(/^data:image\/\w+;base64,/, '')}`);
    setPreviewDialogOpen(true);
  };

  const renderBannerSkeletons = () => {
    return Array.from(new Array(6)).map((_, index) => (
      <Grid item xs={12} sm={6} md={4} key={index}>
        <Box>
          <Skeleton variant="rectangular" width="100%" height={200} sx={{ borderRadius: 2 }} />
          <Skeleton width="60%" sx={{ mx: 'auto', mt: 1 }} />
        </Box>
      </Grid>
    ));
  };

  // Delete flow
  const askDeleteBanner = (banner: BannerImage) => {
    setBannerToDelete(banner);
    setDeleteDialogOpen(true);
  };

  const deleteBanner = async () => {
    if (!bannerToDelete) return;
    try {
      setIsDeleting(true);
      await axios.delete(`${BASE_URL}/deleteBanner/${bannerToDelete.id}`, {
        headers: { 'Authorization': `Bearer ${localStorage.getItem('token')}` }
      });
      handleSnackbarOpen('Banner deleted successfully');
      setDeleteDialogOpen(false);
      setBannerToDelete(null);
      await fetchBanners();
      // If the deleted banner was selected for update, clear selection
      if (selectedBanner?.type === bannerToDelete.type) {
        setSelectedBanner(null);
      }
    } catch (err: any) {
      const msg = err?.response?.data?.message || 'Failed to delete banner';
      handleSnackbarOpen(msg, 'error');
    } finally {
      setIsDeleting(false);
    }
  };

  // Get unique existing types to hint to user
  const existingTypes = Array.from(new Set(banners.map(b => b.type))).sort();

  return (
    <div className="ml-0 lg:ml-[256px] p-2">
      <Box 
        sx={{ 
          width: '100%', 
          maxWidth: 1200, 
          margin: 'auto', 
          padding: 2,
          backgroundColor: theme.palette.background.default
        }}
      >
        {/* Top actions */}
        <Stack direction="row" justifyContent="flex-end" sx={{ mb: 2 }}>
          <Button
            variant="contained"
            startIcon={<AddPhotoAlternateIcon />}
            onClick={() => setShowAddPanel(v => !v)}
          >
            {showAddPanel ? 'Close Add Image' : 'Add Image'}
          </Button>
        </Stack>

        {/* ADD NEW BANNER (collapsible) */}
        {showAddPanel && (
          <Card elevation={6} sx={{ mb: 3, borderRadius: 3, overflow: 'hidden' }}>
            <CardHeader 
              title="Add New Banner"
              titleTypographyProps={{ variant: 'h5', color: 'primary' }}
              subheader={
                existingTypes.length ? (
                  <Box sx={{ mt: 1 }}>
                    <Typography variant="body2" color="text.secondary">
                      Existing types:
                    </Typography>
                    <Box sx={{ mt: 1, display: 'flex', gap: 1, flexWrap: 'wrap' }}>
                      {existingTypes.map(t => (
                        <Chip key={t} label={t} size="small" />
                      ))}
                    </Box>
                  </Box>
                ) : null
              }
              sx={{ 
                backgroundColor: theme.palette.background.paper,
                borderBottom: `1px solid ${theme.palette.divider}`
              }}
            />
            <CardContent>
              <Grid container spacing={2}>
                <Grid item xs={12} md={4}>
                  <TextField
                    fullWidth
                    label="Banner Type"
                    value={newBannerType}
                    onChange={(e) => setNewBannerType(e.target.value)}
                    placeholder="e.g., home_top, promo, footer_ads"
                  />
                  <Typography variant="caption" color="text.secondary">
                    Use a unique type to create a new banner category.
                  </Typography>
                </Grid>
                <Grid item xs={12} md={8}>
                  <Box
                    onDragEnter={(e) => { e.preventDefault(); setIsDragging(true); }}
                    onDragOver={(e) => { e.preventDefault(); }}
                    onDragLeave={(e) => { e.preventDefault(); setIsDragging(false); }}
                    onDrop={handleDrop}
                    sx={{
                      border: '2px dashed',
                      borderColor: isDragging ? 'primary.main' : 'grey.400',
                      borderRadius: 2,
                      p: 2,
                      textAlign: 'center',
                      transition: '0.2s',
                      bgcolor: isDragging ? 'action.hover' : 'transparent',
                    }}
                  >
                    <CloudUploadIcon />
                    <Typography variant="body2" sx={{ mt: 1 }}>
                      Drag & drop images here, or click to select
                    </Typography>

                    <Input 
                      type="file" 
                      inputProps={{ 
                        accept: 'image/png,image/jpeg,image/jpg,image/webp',
                        multiple: true,
                        style: { display: 'none' }
                      }}
                      onChange={handleNewInputChange}
                      id="new-banner-file-input"
                    />
                    <label htmlFor="new-banner-file-input">
                      <Button component="span" variant="outlined" sx={{ mt: 1 }}>
                        Choose Images
                      </Button>
                    </label>
                    <Typography variant="caption" color="text.secondary" display="block" sx={{ mt: 1 }}>
                      PNG/JPG/WEBP up to 8MB each.
                    </Typography>
                  </Box>
                </Grid>

                {newFilePreviews.length > 0 && (
                  <Grid item xs={12}>
                    <Grid container spacing={2}>
                      {newFilePreviews.map((src, idx) => (
                        <Grid item xs={6} sm={4} md={3} key={idx}>
                          <Box sx={{ position: 'relative' }}>
                            <img
                              src={src}
                              alt={`new-${idx}`}
                              style={{
                                width: '100%',
                                height: 140,
                                objectFit: 'cover',
                                borderRadius: 8,
                                boxShadow: theme.shadows[2]
                              }}
                            />
                            <IconButton
                              size="small"
                              onClick={() => removeNewFileAt(idx)}
                              sx={{
                                position: 'absolute',
                                top: 6,
                                right: 6,
                                bgcolor: 'rgba(0,0,0,0.5)',
                                color: '#fff',
                                '&:hover': { bgcolor: 'rgba(0,0,0,0.7)' }
                              }}
                              aria-label="remove image"
                            >
                              <DeleteIcon fontSize="small" />
                            </IconButton>
                          </Box>
                          <Typography variant="caption" sx={{ display: 'block', mt: 0.5 }} noWrap>
                            {newFiles[idx]?.name}
                          </Typography>
                        </Grid>
                      ))}
                    </Grid>
                  </Grid>
                )}

                <Grid item xs={12} md={6}>
                  <Button 
                    onClick={addBanners}
                    disabled={isAdding}
                    variant="contained"
                    color="secondary"
                    fullWidth
                    sx={{ py: 1.5, borderRadius: 2 }}
                    startIcon={<CloudUploadIcon />}
                  >
                    {isAdding ? 'Uploading...' : 'Upload New Banner(s)'}
                  </Button>
                </Grid>
                <Grid item xs={12} md={6}>
                  <Button 
                    onClick={clearNewForm}
                    disabled={isAdding || (newFiles.length === 0 && !newBannerType)}
                    variant="outlined"
                    fullWidth
                    sx={{ py: 1.5, borderRadius: 2 }}
                  >
                    Clear
                  </Button>
                </Grid>
              </Grid>
            </CardContent>
          </Card>
        )}

        {/* EXISTING BANNERS LIST + UPDATE */}
        <Card 
          elevation={6} 
          sx={{ 
            borderRadius: 3,
            transition: 'transform 0.3s',
            '&:hover': { transform: 'scale(1.01)' }
          }}
        >
          <CardHeader 
            title="Banner Management" 
            titleTypographyProps={{ variant: 'h4', color: 'primary', align: 'center' }}
            sx={{ 
              backgroundColor: theme.palette.background.paper,
              borderBottom: `1px solid ${theme.palette.divider}`
            }}
          />
          <CardContent>
            <Grid container spacing={3}>
              {isLoading 
                ? renderBannerSkeletons()
                : banners.map(banner => (
             <Grid item xs={12} sm={6} md={4} key={banner.id}>
  <Box
    sx={{
      border: '2px solid',
      borderColor: selectedBanner?.type === banner.type ? 'primary.main' : 'grey.300',
      borderRadius: 2,
      p: 1,
      cursor: 'pointer',
      backgroundColor: selectedBanner?.type === banner.type ? 'primary.light' : 'transparent',
      transition: 'all 0.3s ease',
      position: 'relative',     // <-- important
      '&:hover': { boxShadow: theme.shadows[4], transform: 'scale(1.05)' }
    }}
  >
    <Box onClick={() => handleBannerSelect(banner)} sx={{ position: 'relative' }}>
      {/* DELETE BUTTON (overlay, top-left) */}
      <Tooltip title="Delete image">
        <IconButton
          onClick={(e) => { e.stopPropagation(); askDeleteBanner(banner); }}
          sx={{
            position: 'absolute',
            top: 8,
            left: 8,
            zIndex: 2,                              // <-- ensure on top
            bgcolor: 'rgba(255,255,255,0.95)',
            color: 'error.main',                    // red icon
            boxShadow: theme.shadows[2],
            '&:hover': { bgcolor: 'rgba(255,255,255,1)' }
          }}
          aria-label="delete banner"
          size="small"
        >
          <DeleteIcon fontSize="small" />
        </IconButton>
      </Tooltip>

      {/* PREVIEW BUTTON (overlay, top-right) */}
      <Tooltip title="Preview">
        <IconButton
          onClick={(e) => { e.stopPropagation(); openPreviewDialog(banner.base64); }}
          sx={{
            position: 'absolute',
            top: 8,
            right: 8,
            zIndex: 2,
            bgcolor: 'rgba(255,255,255,0.95)',
            color: 'inherit',
            boxShadow: theme.shadows[2],
            '&:hover': { bgcolor: 'rgba(255,255,255,1)' }
          }}
          aria-label="preview banner"
          size="small"
        >
          <ImageSearchIcon fontSize="small" />
        </IconButton>
      </Tooltip>

      {/* IMAGE */}
      <img
        src={`data:image/png;base64,${banner.base64}`}
        alt={banner.type}
        style={{
          width: '100%',
          height: 200,
          objectFit: 'cover',
          borderRadius: 8,
          display: 'block'
        }}
      />
    </Box>

    <Typography variant="subtitle2" sx={{ mt: 1, textAlign: 'center' }}>
      {banner.type}
    </Typography>
  </Box>
</Grid>

                  ))
              }
            </Grid>

            {!isLoading && selectedBanner && (
              <Box sx={{ mt: 4, textAlign: 'center' }}>
                <FormLabel sx={{ display: 'block', mb: 2, fontWeight: 'bold' }}>
                  Selected Banner Type: {selectedBanner.type}
                </FormLabel>
                <Input 
                  type="file" 
                  inputProps={{ accept: '.png,.jpg,.jpeg,.webp', style: { display: 'none' } }}
                  onChange={handleFileChange}
                  id="banner-file-input"
                />
                <label htmlFor="banner-file-input">
                  <Button 
                    component="span"
                    variant="contained"
                    color="primary"
                    startIcon={<CloudUploadIcon />}
                    sx={{ mb: 2 }}
                  >
                    Choose File
                  </Button>
                </label>

                {selectedBanner.base64Preview && (
                  <Box sx={{ mb: 2, display: 'flex', justifyContent: 'center', alignItems: 'center' }}>
                    <img 
                      src={selectedBanner.base64Preview} 
                      alt="Preview" 
                      style={{ maxWidth: '100%', maxHeight: 300, objectFit: 'contain', borderRadius: 8, boxShadow: theme.shadows[3] }}
                    />
                  </Box>
                )}
                <Button 
                  onClick={updateBanner} 
                  disabled={!selectedBanner.file || isLoading}
                  variant="contained"
                  color="secondary"
                  fullWidth
                  sx={{ mt: 2, py: 1.5, borderRadius: 2 }}
                >
                  {isLoading ? 'Updating...' : 'Update Banner'}
                </Button>
              </Box>
            )}
          </CardContent>
        </Card>

        {/* SNACKBAR */}
        <Snackbar
          open={snackbar.open}
          autoHideDuration={6000}
          onClose={handleSnackbarClose}
          anchorOrigin={{ vertical: 'bottom', horizontal: 'right' }}
        >
          <Alert 
            onClose={handleSnackbarClose} 
            severity={snackbar.severity} 
            sx={{ width: '100%' }}
            action={
              <IconButton size="small" aria-label="close" color="inherit" onClick={handleSnackbarClose}>
                <CloseIcon fontSize="small" />
              </IconButton>
            }
          >
            {snackbar.message}
          </Alert>
        </Snackbar>

        {/* PREVIEW DIALOG */}
        <Dialog open={previewDialogOpen} onClose={() => setPreviewDialogOpen(false)} maxWidth="md" fullWidth>
          <DialogTitle>Banner Preview</DialogTitle>
          <DialogContent>
            <img 
              src={previewImage} 
              alt="Full Preview" 
              style={{ width: '100%', maxHeight: '70vh', objectFit: 'contain' }}
            />
          </DialogContent>
          <DialogActions>
            <Button onClick={() => setPreviewDialogOpen(false)} color="primary">Close</Button>
          </DialogActions>
        </Dialog>

        {/* DELETE CONFIRMATION */}
        <Dialog open={deleteDialogOpen} onClose={() => setDeleteDialogOpen(false)}>
          <DialogTitle>Delete Banner</DialogTitle>
          <DialogContent>
            <Typography>
              Are you sure you want to delete this banner image
              {bannerToDelete ? ` (type: ${bannerToDelete.type})` : ''}?
            </Typography>
          </DialogContent>
          <DialogActions>
            <Button onClick={() => setDeleteDialogOpen(false)} disabled={isDeleting}>Cancel</Button>
            <Button onClick={deleteBanner} color="error" variant="contained" disabled={isDeleting}>
              {isDeleting ? 'Deleting…' : 'Delete'}
            </Button>
          </DialogActions>
        </Dialog>
      </Box>
    </div>
  );
};

export default Banner;
