import React, { useEffect, useState } from 'react';
import axios from 'axios';

type Bonus = {
  bonus_type: string;
  deposit_multiplier: number;
  wager_multiplier: number;
  min_deposit: number;
};

const BonusConfig: React.FC = () => {
  const [bonuses, setBonuses] = useState<Bonus[]>([]);
  const [editValues, setEditValues] = useState<Bonus[]>([]);
  const [snackbar, setSnackbar] = useState<string | null>(null);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);

  useEffect(() => {
    fetchBonusData();
  }, []);

  const fetchBonusData = async () => {
    try {
      setLoading(true);
      const response = await axios.get('https://api.air1bet.com/api/bonus/config-public', {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });
      const data = response.data;

      // Validate that the data is an array of Bonus objects
      if (Array.isArray(data) && data.every(item => 
        item.bonus_type && 
        typeof item.deposit_multiplier === 'number' && 
        typeof item.wager_multiplier === 'number' && 
        typeof item.min_deposit === 'number'
      )) {
        setBonuses(data);
        setEditValues([...data]); // Sync editValues with fetched data
       
      } else {
        throw new Error('Invalid data format received from API');
      }
    } catch (error) {
      console.error('Failed to fetch bonus data:', error);
      setError('Failed to load bonus data. Please try again later.');
    } finally {
      setLoading(false);
    }
  };

  const handleInputChange = (
    index: number,
    field: keyof Bonus,
    value: string,
  ) => {
    const numValue = parseFloat(value);
    if (numValue < 0) return; // Prevent negative values
    const updatedValues = [...editValues];
    updatedValues[index] = {
      ...updatedValues[index],
      [field]: isNaN(numValue) ? 0 : numValue,
    };
    setEditValues(updatedValues);
  };

  const handleSave = async (index: number) => {
    try {
      const updatedBonus = { ...editValues[index] };
      await axios.put(
  'https://api.air1bet.com/api/bonus/updateConfig',
  updatedBonus,
  {
    headers: {
      'Authorization': `Bearer ${localStorage.getItem('token')}`,
      'Content-Type': 'application/json'
    }
  }
);
      const updatedBonuses = [...bonuses];
      updatedBonuses[index] = updatedBonus;
      setBonuses(updatedBonuses);
      setSnackbar(`Bonus for ${updatedBonus.bonus_type} updated successfully`);
      setTimeout(() => setSnackbar(null), 3000);
    } catch (error) {
      console.error('Failed to save bonus data:', error);
      setSnackbar('Failed to save changes. Please try again.');
      setTimeout(() => setSnackbar(null), 3000);
    }
  };

  if (loading) {
    return (
      <div className="p-8 bg-gray-900 min-h-screen lg:ml-[250px]">
        <p className="text-white text-center">Loading...</p>
      </div>
    );
  }

  if (error) {
    return (
      <div className="p-8 bg-gray-900 min-h-screen lg:ml-[250px]">
        <p className="text-red-500 text-center">{error}</p>
      </div>
    );
  }

  return (
    <div className="p-8 bg-gray-900 min-h-screen lg:ml-[250px]">
      {snackbar && (
        <div className="mb-4 text-center text-white bg-green-500 py-2 px-4 rounded-lg shadow-md">
          {snackbar}
        </div>
      )}

      <h1 className="text-2xl text-white font-semibold mb-6">{`Total Bonuses: ${bonuses.length}`}</h1>

      <div className="space-y-6">
        {bonuses.map((bonus, index) => (
          <div
            key={index}
            className="bg-[#37384A] p-6 rounded-lg shadow-md border border-gray-700"
          >
            <h2 className="text-xl text-white font-semibold mb-4 capitalize">
              {bonus.bonus_type} Bonus
            </h2>
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div>
                <label className="block text-sm text-gray-300 mb-1">
                  Deposit Bonus (%)
                </label>
                <input
                  type="number"
                  value={editValues[index]?.deposit_multiplier ?? 0}
                  onChange={(e) =>
                    handleInputChange(index, 'deposit_multiplier', e.target.value)
                  }
                  className="w-full bg-[#2a2b38] text-white border border-gray-600 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                />
              </div>
              <div>
                <label className="block text-sm text-gray-300 mb-1">
                  Wager Multiplier
                </label>
                <input
                  type="number"
                  value={editValues[index]?.wager_multiplier ?? 0}
                  onChange={(e) =>
                    handleInputChange(index, 'wager_multiplier', e.target.value)
                  }
                  className="w-full bg-[#2a2b38] text-white border border-gray-600 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                />
              </div>
              <div>
                <label className="block text-sm text-gray-300 mb-1">
                  Min Deposit ($)
                </label>
                <input
                  type="number"
                  value={editValues[index]?.min_deposit ?? 0}
                  onChange={(e) =>
                    handleInputChange(index, 'min_deposit', e.target.value)
                  }
                  className="w-full bg-[#2a2b38] text-white border border-gray-600 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
                />
              </div>
            </div>
            <div className="mt-4 flex justify-end">
              <button
                onClick={() => handleSave(index)}
                className="text-white bg-blue-500 hover:bg-blue-700 font-medium rounded-lg text-sm px-4 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500"
              >
                Save Changes
              </button>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
};

export default BonusConfig;