/* =========================================================================
   Deposit.tsx    – shows down-line deposits (INR)
   ========================================================================= */
import React, { useEffect, useMemo, useState } from "react";
import { ChevronLeft, ChevronRight, RefreshCw } from "lucide-react";
import { format } from "date-fns";
import { apiFetch } from "../utils/api";

/* ---------- API response type ------------------------------------------ */
interface TransferRow {
  id         : number;
  from_type  : "staff" | "user";
  from_id    : number;
  to_type    : "staff" | "user";
  to_id      : number;
  amount     : number;
  created_at : string;
  from_name  : string | null;
  to_name    : string | null;
  direction  : "deposit" | "withdraw";   // delivered but not used here
}

interface Pagination { page:number; size:number; }

const fmt = (iso:string)=>format(new Date(iso),"yyyy-MM-dd HH:mm:ss");

/* ----------------------------------------------------------------------- */
export default function Deposit() {
  const [rows,setRows]   = useState<TransferRow[]>([]);
  const [load,setLoad]   = useState(false);
  const [err ,setErr]    = useState("");
  const [pg  ,setPg]     = useState<Pagination>({page:1,size:25});

  /* simple client-side filters */
  const [q     ,setQ]     = useState("");
  const [from  ,setFrom ] = useState("");
  const [to    ,setTo  ]  = useState("");
  const [min   ,setMin ]  = useState("");
  const [max   ,setMax ]  = useState("");

  /* ---------------- fetch ---------------------------------------------- */
  const refetch = async()=>{
    try{
      setLoad(true); setErr("");
      const uid   = localStorage.getItem("currentUserId") || "";
      const base  = uid ? `/api/staff/transfers/${uid}` : "/api/staff/transfers";
      const qs    = new URLSearchParams({
        dir:"deposit", page:String(pg.page), limit:String(pg.size)
      }).toString();
      const { data } = await apiFetch<{status:string;data:TransferRow[]}>(`${base}?${qs}`);
      setRows(data);
    }catch(e:any){ setErr(e.message||"Server error"); }
    finally      { setLoad(false); }
  };
  useEffect(()=>{ refetch(); /* eslint-disable */},[pg.page]);

  /* ---------------- filter in memory ----------------------------------- */
  const view = useMemo(()=>rows.filter(r=>{
    if(q && !(r.from_id.toString()===q || r.to_id.toString()===q)) return false;
    if(from && r.created_at.slice(0,10)<from) return false;
    if(to   && r.created_at.slice(0,10)>to  ) return false;
    if(min && r.amount < +min) return false;
    if(max && r.amount > +max) return false;
    return true;
  }),[rows,q,from,to,min,max]);

  /* ---------------- render --------------------------------------------- */
  return (
    <div className="p-4 ml-0 lg:ml-[256px] space-y-6">
      <div className="flex items-center gap-4">
        <h1 className="text-lg font-semibold text-white">Deposit History</h1>
        <button onClick={()=>{setPg(p=>({...p,page:1})); refetch();}}
                className="flex items-center gap-1 text-sm text-gray-500">
          <RefreshCw className="w-4 h-4"/> reload
        </button>
      </div>

      {/* filters */}
      <div className="bg-white p-4 rounded-lg shadow space-y-4">
        <div className="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-5 gap-4">
          <input className="p-2 border rounded-lg" placeholder="Search ID"
                 value={q} onChange={e=>setQ(e.target.value)}/>
          <input type="date" className="p-2 border rounded-lg"
                 value={from} onChange={e=>setFrom(e.target.value)}/>
          <input type="date" className="p-2 border rounded-lg"
                 value={to}   onChange={e=>setTo(e.target.value)}/>
          <input className="p-2 border rounded-lg" placeholder="Min ₹"
                 value={min} onChange={e=>setMin(e.target.value)}/>
          <input className="p-2 border rounded-lg" placeholder="Max ₹"
                 value={max} onChange={e=>setMax(e.target.value)}/>
        </div>
        <button onClick={()=>{setQ("");setFrom("");setTo("");setMin("");setMax("");}}
                className="px-4 py-2 bg-gray-200 rounded-md text-sm">
          Clear
        </button>
      </div>

      {/* table */}
      <div className="bg-white rounded-lg shadow overflow-x-auto">
        <table className="min-w-full text-sm">
          <thead>
            <tr className="bg-gray-50">
              {["ID","From","To","BDT","When"].map(h=>(
                <th key={h} className="px-4 py-2 text-left font-medium text-gray-500 uppercase">{h}</th>
              ))}
            </tr>
          </thead>
          <tbody>
            {load?(
              <tr><td colSpan={5} className="py-6 text-center">Loading…</td></tr>
            ):err?(
              <tr><td colSpan={5} className="py-6 text-center text-red-600">{err}</td></tr>
            ):view.length===0?(
              <tr><td colSpan={5} className="py-6 text-center">No deposits</td></tr>
            ):view.map(r=>(
              <tr key={r.id} className="border-t">
                <td className="px-4 py-2">{r.id}</td>
                <td className="px-4 py-2">
                  {r.from_name ?? "(unknown)"}&nbsp;({r.from_id})
                </td>
                <td className="px-4 py-2">
                  {r.to_name   ?? "(unknown)"}&nbsp;({r.to_id})
                </td>
                <td className="px-4 py-2 font-mono">{r.amount?.toFixed(2)}</td>
                <td className="px-4 py-2">{fmt(r.created_at)}</td>
              </tr>
            ))}
          </tbody>
        </table>

        {/* pager */}
        <div className="flex items-center justify-between px-4 py-3 border-t">
          <button disabled={pg.page===1}
                  onClick={()=>setPg(p=>({...p,page:p.page-1}))}
                  className="p-2 disabled:opacity-40">
            <ChevronLeft className="w-5 h-5"/>
          </button>
          <span className="text-sm text-gray-600">Page {pg.page}</span>
          <button onClick={()=>setPg(p=>({...p,page:p.page+1}))} className="p-2">
            <ChevronRight className="w-5 h-5"/>
          </button>
        </div>
      </div>
    </div>
  );
}
