import React, { useEffect, useMemo, useState } from "react";
import axios from "axios";

// ====== Types ======
interface MarketWin {
  id: number;
  totalbets: number;
  matchid: number;
  eventid: number;
  team1ex: number;
  team2ex: number;
  drawex: number | null;
  winteam: string;
  matchname: string;
  payout: number;
  user_id: number | null;
  created_at: string; // ISO timestamp
}

interface PaginationState {
  totalCount: number;
  totalPages: number;
  currentPage: number;
  limit: number;
}

interface MarketWinResponse {
  data: MarketWin[]; // backend shape you control
  pagination: PaginationState;
}

// ====== Component ======
const MarketHistory: React.FC = () => {
  const staffId = localStorage.getItem("currentUserId");
  const token = localStorage.getItem("token");

  const [rows, setRows] = useState<MarketWin[]>([]);
  const [searchTerm, setSearchTerm] = useState<string>("");
  const [userId, setUserId] = useState<string>("");
  const [loading, setLoading] = useState<boolean>(false);
  const [pagination, setPagination] = useState<PaginationState>({
    totalCount: 0,
    totalPages: 0,
    currentPage: 1,
    limit: 10,
  });

  // ---- Configurable base URL (adjust to your API) ----
  const BASE_URL = useMemo(() => {
    // Prefer env if provided (Vite/CRA), fallback to production API you used for bets
    return (
      (import.meta as any)?.env?.VITE_API_BASE_URL ||
      process.env.REACT_APP_API_BASE_URL ||
      "https://api.air1bet.com"
    );
  }, []);

  // Debounced effects
  useEffect(() => {
    const t = setTimeout(() => {
      fetchData();
    }, 400);
    return () => clearTimeout(t);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [searchTerm]);

  useEffect(() => {
    fetchData();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [pagination.currentPage]);

  // ---- Fetch ----
  const fetchData = async () => {
    setLoading(true);
    try {
      // Endpoint contract: implement this on server side
      // GET /marketwins?page&limit&search&userId
      let url = `${BASE_URL}/sportsbetting/marketwins?page=${pagination.currentPage}&limit=${pagination.limit}`;
      if (searchTerm) url += `&search=${encodeURIComponent(searchTerm)}`;
      if (userId) url += `&userId=${encodeURIComponent(userId)}`;

      const headers: Record<string, string> = {
        ...(token ? { Authorization: `Bearer ${token}` } : {}),
        ...(staffId ? { "x-staff-id": staffId } : {}),
      };

      const { data } = await axios.get<MarketWinResponse>(url, { headers });

      // Support both {data, pagination} and {transactions, pagination}
      const list = (data as any).data || (data as any).transactions || [];
      setRows(list as MarketWin[]);
      setPagination((prev) => ({ ...prev, ...(data as any).pagination }));
    } catch (e) {
      console.error("Failed to fetch market history:", e);
    } finally {
      setLoading(false);
    }
  };

  // ---- Handlers ----
  const handleSearchChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setSearchTerm(e.target.value);
    setPagination((p) => ({ ...p, currentPage: 1 }));
  };

  const handleUserFilterChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setUserId(e.target.value);
    setPagination((p) => ({ ...p, currentPage: 1 }));
  };

  const handlePageChange = (newPage: number) => {
    if (newPage < 1 || (pagination.totalPages && newPage > pagination.totalPages)) return;
    setPagination((p) => ({ ...p, currentPage: newPage }));
  };

  const handleClearFilters = () => {
    setSearchTerm("");
    setUserId("");
    setPagination((p) => ({ ...p, currentPage: 1 }));
  };

  // ---- Formatters ----
  const formatNumber = (n: number | null | undefined) =>
    typeof n === "number" && !Number.isNaN(n) ? n.toFixed(2) : "-";

  const formatDate = (iso: string) => new Date(iso).toLocaleString();

  // ---- UI ----
  if (loading && rows.length === 0) {
    return (
      <div className="ml-0 lg:ml-[256px] p-6">
        <div className="flex items-center justify-center h-64">
          <div className="text-gray-500">Loading market history...</div>
        </div>
      </div>
    );
  }

  return (
    <div className="ml-0 lg:ml-[256px] p-6">
      <div className="bg-white rounded-lg shadow-md p-6 mb-6">
        <h1 className="text-2xl font-bold mb-6">Market history</h1>

        {/* Filters */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 mb-6">
          <div>
            <label htmlFor="search-input" className="block text-sm font-medium text-gray-700 mb-1">
              Search
            </label>
            <input
              id="search-input"
              type="text"
              placeholder="Search by match name, event id, or winner"
              value={searchTerm}
              onChange={handleSearchChange}
              className="w-full p-2 border rounded focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
          <div>
            <label htmlFor="user-id-input" className="block text-sm font-medium text-gray-700 mb-1">
              User ID
            </label>
            <input
              id="user-id-input"
              type="text"
              placeholder="Filter by User ID"
              value={userId}
              onChange={handleUserFilterChange}
              className="w-full p-2 border rounded focus:ring-blue-500 focus:border-blue-500"
            />
          </div>
        </div>

        <div className="mb-4">
          <span className="text-gray-600">
            Showing {rows.length} of {pagination.totalCount} records
          </span>
          {(searchTerm || userId) && (
            <button
              onClick={handleClearFilters}
              className="ml-4 text-blue-500 hover:text-blue-700"
              aria-label="Clear filters"
            >
              Clear filters
            </button>
          )}
        </div>

        {/* No results */}
        {rows.length === 0 && !loading && (
          <div className="text-center py-10">
            <p className="text-gray-500">No market results found</p>
          </div>
        )}

        {/* Table */}
        {rows.length > 0 && (
          <div className="overflow-x-auto">
            <table className="w-full bg-white shadow-md rounded-lg overflow-hidden">
              <thead className="bg-gray-100 border-b">
                <tr>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Match</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Winner</th>
                  <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Team1Ex</th>
                  <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Team2Ex</th>
                  <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">DrawEx</th>
                  <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Total Bets</th>
                  <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Payout</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Match ID</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Event ID</th>
                  <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                </tr>
              </thead>
              <tbody className="divide-y divide-gray-200">
                {rows.map((r) => (
                  <tr key={r.id} className="hover:bg-gray-50 transition-colors">
                    <td className="px-4 py-3 whitespace-nowrap">{r.matchname}</td>
                    <td className="px-4 py-3 whitespace-nowrap">{r.winteam}</td>
                    <td className="px-4 py-3 text-right whitespace-nowrap">{formatNumber(r.team1ex)}</td>
                    <td className="px-4 py-3 text-right whitespace-nowrap">{formatNumber(r.team2ex)}</td>
                    <td className="px-4 py-3 text-right whitespace-nowrap">{formatNumber(r.drawex)}</td>
                    <td className="px-4 py-3 text-right whitespace-nowrap">{formatNumber(r.totalbets)}</td>
                    <td className="px-4 py-3 text-right whitespace-nowrap">{formatNumber(r.payout)}</td>
                    <td className="px-4 py-3 whitespace-nowrap">{r.user_id ?? '-'}</td>
                    <td className="px-4 py-3 whitespace-nowrap">{r.matchid}</td>
                    <td className="px-4 py-3 whitespace-nowrap">{r.eventid}</td>
                    <td className="px-4 py-3 whitespace-nowrap">{formatDate(r.created_at)}</td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}

        {/* Pagination */}
        {pagination.totalPages > 1 && (
          <div className="flex items-center justify-center mt-6">
            <button
              onClick={() => handlePageChange(pagination.currentPage - 1)}
              disabled={pagination.currentPage === 1}
              className="px-4 py-2 mr-4 bg-gray-100 rounded border hover:bg-gray-200 disabled:opacity-50"
              aria-label="Previous page"
            >
              Previous
            </button>

            <div className="text-center px-4 py-2 bg-gray-50 rounded border min-w-[80px]">
              {pagination.currentPage} / {pagination.totalPages || 1}
            </div>

            <button
              onClick={() => handlePageChange(pagination.currentPage + 1)}
              disabled={pagination.currentPage === pagination.totalPages || pagination.totalPages === 0}
              className="px-4 py-2 ml-4 bg-gray-100 rounded border hover:bg-gray-200 disabled:opacity-50"
              aria-label="Next page"
            >
              Next
            </button>
          </div>
        )}
      </div>
    </div>
  );
};

export default MarketHistory;
