// CombinedSportsResults.tsx
import React, { useEffect, useMemo, useState } from "react";
import axios from "axios";
import {
  Box,
  Paper,
  Tabs,
  Tab,
  Typography,
  Grid,
  TextField,
  Stack,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Table,
  TableHead,
  TableRow,
  TableCell,
  TableBody,
  TableContainer,
  CircularProgress,
  IconButton,
  Chip,
  TablePagination,
  Button,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
} from "@mui/material";
import { Search as SearchIcon, Delete as DeleteIcon } from "@mui/icons-material";
import type { SelectChangeEvent } from "@mui/material";

// ------------------ Types ------------------
interface SportsBet {
  id: number;
  username: string;
  user_id: number;
  match_title: string;
  team_one: string;
  team_two: string;
  game_type: string;
  selection_name: string;
  game_name: string;
  bet_type: string;
  odds: number;
  stake_amount: number;
  usd_amount: number;
  status: string;
  match_start_time: string;
  match_end_time: string;
}

interface ApiResponse {
  success: boolean;
  bets: SportsBet[];
  pagination: {
    total: number;
    totalPages: number;
    page: number;
    limit: number;
  };
  error?: string;
}

interface Filters {
  status: string;
  gameType: string;
  betType: string;
  startDate: string;
  endDate: string;
  currency: string;
}

interface MarketWin {
  id: number;
  totalbets: number;
  matchid: number | string;
  eventid: number | string;
  team1ex: number;
  team2ex: number;
  drawex: number | null;
  winteam: string;
  matchname: string;
  payout: number;
  user_id: number | null;
  created_at: string; // ISO timestamp
}

interface FanWin {
  id: number;
  userid: number;
  fancyname: string;
  selection: string;
  runsodds: number;
  payout: number;
  eventid: number | string;
  matchid: number | string;
  created_at: string;
}

interface PaginationState {
  totalCount: number;
  totalPages: number;
  currentPage: number;
  limit: number;
}

// ------------------ Constants ------------------
const GAME_TYPES = [
  "Cricket",
  "Tennis",
  "Soccer",
  "American Football",
  "Baseball",
  "Basketball",
  "Boxing",
  "Darts",
  "Gaelic Games",
  "Golf",
  "Greyhound Racing",
  "Greyhound Todays Card",
  "Horse Race Todays Card",
  "Horse Racing",
  "Mixed Martial Arts",
  "Motor Sport",
  "Rugby League",
  "Rugby Union",
  "Snooker",
];

const STATUS_TYPES = [
  { value: "open", label: "Open" },
  { value: "closed", label: "Closed" },
];

const BET_TYPES = [
  { value: "back", label: "Back" },
  { value: "lay", label: "Lay" },
];

// ------------------ Helper TabPanel ------------------
function TabPanel(props: { children?: React.ReactNode; index: number; value: number }) {
  const { children, value, index } = props;
  return (
    <div role="tabpanel" hidden={value !== index}>
      {value === index && <Box sx={{ pt: 2 }}>{children}</Box>}
    </div>
  );
}

// ------------------ SportsBetting Component (your original adapted) ------------------
const SportsBetting: React.FC = () => {
  const [bets, setBets] = useState<SportsBet[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [totalCount, setTotalCount] = useState(0);
  const [searchTerm, setSearchTerm] = useState("");
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [selectedBetId, setSelectedBetId] = useState<number | null>(null);
  const staffId = localStorage.getItem("currentUserId");

  const [filters, setFilters] = useState<Filters>({
    status: "",
    gameType: "",
    betType: "",
    startDate: "",
    endDate: "",
    currency: "usd",
  });

  const fetchBets = async () => {
    try {
      setLoading(true);
      const searchParams = new URLSearchParams({
        page: (page + 1).toString(),
        limit: rowsPerPage.toString(),
        ...(filters.status && { status: filters.status }),
        ...(filters.gameType && { gameType: filters.gameType }),
        ...(filters.betType && { betType: filters.betType }),
        ...(filters.startDate && { startDate: filters.startDate }),
        ...(filters.endDate && { endDate: filters.endDate }),
        ...(searchTerm && { search: searchTerm }),
        currency: filters.currency,
      });

      const response = await fetch(
        `https://api.air1bet.com/sportsbetting/admin/bets?${searchParams.toString()}`,
        {
          headers: {
            Authorization: `Bearer ${localStorage.getItem("token")}`,
            "x-staff-id": staffId ?? "",
          },
        }
      );
      const data: ApiResponse = await response.json();

      if (data.success) {
        setBets(data.bets);
        setTotalCount(data.pagination?.total || 0);
      } else {
        setError(data.error || "Failed to fetch bets");
      }
    } catch (err) {
      console.error(err);
      setError("Failed to fetch bets");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchBets();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [page, rowsPerPage, filters, searchTerm]);

  const handleDelete = async () => {
    if (selectedBetId) {
      try {
        const response = await fetch(
          `https://api.air1bet.com/sportsbetting/admin/bets/${selectedBetId}`,
          {
            method: "DELETE",
            headers: {
              Authorization: `Bearer ${localStorage.getItem("token")}`,
            },
          }
        );
        const data = await response.json();
        if (data.success) {
          fetchBets();
          setDeleteDialogOpen(false);
        } else {
          setError(data.error || "Failed to delete bet");
        }
      } catch (err) {
        console.error(err);
        setError("Failed to delete bet");
      }
    }
  };

  const handleStatusUpdate = async (betId: number, newStatus: string) => {
    try {
      const response = await fetch(
        `https://api.air1bet.com/sportsbetting/admin/bets/${betId}/status`,
        {
          method: "PUT",
          headers: {
            "Content-Type": "application/json",
            Authorization: `Bearer ${localStorage.getItem("token")}`,
          },
          body: JSON.stringify({
            status: newStatus,
            resultDeclaredTime: new Date().toISOString(),
          }),
        }
      );
      const data = await response.json();
      if (data.success) {
        fetchBets();
      } else {
        setError(data.error || "Failed to update bet status");
      }
    } catch (err) {
      console.error(err);
      setError("Failed to update bet status");
    }
  };

  const handleStatusChange = (event: SelectChangeEvent<string>, betId: number) => {
    handleStatusUpdate(betId, event.target.value as string);
  };

  const getStatusColor = (status: string): "success" | "error" | "warning" | "default" => {
    switch (status.toLowerCase()) {
      case "open":
        return "warning";
      case "closed":
        return "success";
      default:
        return "default";
    }
  };

  return (
    <Box sx={{ p: 3 }}>
      <Paper sx={{ p: 3 }}>
        <Typography variant="h5" gutterBottom>
          Sports Betting Administration
        </Typography>

        <Grid container spacing={2} sx={{ mb: 3 }}>
          <Grid item xs={12} md={4}>
            <TextField
              fullWidth
              placeholder="Search bets..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              InputProps={{
                startAdornment: <SearchIcon sx={{ color: "action.active", mr: 1 }} />,
              }}
              size="small"
            />
          </Grid>
          <Grid item xs={12} md={8}>
            <Stack direction={{ xs: "column", sm: "row" }} spacing={2}>
              <FormControl size="small" sx={{ minWidth: 120 }}>
                <InputLabel>Status</InputLabel>
                <Select
                  value={filters.status}
                  label="Status"
                  onChange={(e) => setFilters({ ...filters, status: e.target.value })}
                >
                  <MenuItem value="">All Status</MenuItem>
                  {STATUS_TYPES.map((status) => (
                    <MenuItem key={status.value} value={status.value}>
                      {status.label}
                    </MenuItem>
                  ))}
                </Select>
              </FormControl>

              <FormControl size="small" sx={{ minWidth: 120 }}>
                <InputLabel>Game Type</InputLabel>
                <Select
                  value={filters.gameType}
                  label="Game Type"
                  onChange={(e) => setFilters({ ...filters, gameType: e.target.value })}
                >
                  <MenuItem value="">All Games</MenuItem>
                  {GAME_TYPES.map((game) => (
                    <MenuItem key={game} value={game}>
                      {game}
                    </MenuItem>
                  ))}
                </Select>
              </FormControl>

              <FormControl size="small" sx={{ minWidth: 120 }}>
                <InputLabel>Bet Type</InputLabel>
                <Select
                  value={filters.betType}
                  label="Bet Type"
                  onChange={(e) => setFilters({ ...filters, betType: e.target.value })}
                >
                  <MenuItem value="">All Bet Types</MenuItem>
                  {BET_TYPES.map((bt) => (
                    <MenuItem key={bt.value} value={bt.value}>
                      {bt.label}
                    </MenuItem>
                  ))}
                </Select>
              </FormControl>

              <TextField
                type="date"
                label="Start Date"
                value={filters.startDate}
                onChange={(e) => setFilters({ ...filters, startDate: e.target.value })}
                InputLabelProps={{ shrink: true }}
                size="small"
              />

              <TextField
                type="date"
                label="End Date"
                value={filters.endDate}
                onChange={(e) => setFilters({ ...filters, endDate: e.target.value })}
                InputLabelProps={{ shrink: true }}
                size="small"
              />
            </Stack>
          </Grid>
        </Grid>

        {error && (
          <Typography color="error" sx={{ mb: 2 }}>
            {error}
          </Typography>
        )}

        <TableContainer>
          <Table>
            <TableHead>
              <TableRow>
                <TableCell>User ID</TableCell>
                <TableCell>User Name</TableCell>
                <TableCell>Match</TableCell>
                <TableCell>Type</TableCell>
                <TableCell>Selection</TableCell>
                <TableCell>Odds</TableCell>
                <TableCell align="right">Stake</TableCell>
                <TableCell>Status</TableCell>
                <TableCell align="right">Actions</TableCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {loading ? (
                <TableRow>
                  <TableCell colSpan={9} align="center">
                    <CircularProgress />
                  </TableCell>
                </TableRow>
              ) : bets.length === 0 ? (
                <TableRow>
                  <TableCell colSpan={9} align="center">
                    No bets found
                  </TableCell>
                </TableRow>
              ) : (
                bets.map((bet) => (
                  <TableRow key={bet.id} hover>
                    <TableCell>
                      <Typography variant="body2">{bet.user_id}</Typography>
                    </TableCell>
                    <TableCell>
                      <Typography variant="body2">{bet.username}</Typography>
                    </TableCell>
                    <TableCell>
                      <Typography variant="subtitle2">{bet.match_title}</Typography>
                      <Typography variant="caption" color="textSecondary">
                        {bet.team_one} vs {bet.team_two}
                      </Typography>
                    </TableCell>
                    <TableCell>
                      <Typography variant="body2">{bet.game_type}</Typography>
                      <Typography variant="caption" color="textSecondary">
                        {bet.game_name}
                      </Typography>
                    </TableCell>
                    <TableCell>
                      <Typography variant="body2">{bet.selection_name}</Typography>
                      <Typography variant="caption" color="textSecondary">
                        {bet.bet_type}
                      </Typography>
                    </TableCell>
                    <TableCell>{bet.odds}</TableCell>
                    <TableCell align="right">${bet.usd_amount?.toFixed?.(2) ?? "0.00"}</TableCell>
                    <TableCell>
                      <FormControl size="small" fullWidth>
                        <Select
                          value={bet.status}
                          onChange={(e) => handleStatusChange(e as any, bet.id)}
                          renderValue={(value) => (
                            <Chip label={(value as string).toUpperCase()} size="small" color={getStatusColor(value as string)} />
                          )}
                        >
                          {STATUS_TYPES.map((status) => (
                            <MenuItem key={status.value} value={status.value}>
                              {status.label}
                            </MenuItem>
                          ))}
                        </Select>
                      </FormControl>
                    </TableCell>
                    <TableCell align="right">
                      <IconButton
                        size="small"
                        onClick={() => {
                          setSelectedBetId(bet.id);
                          setDeleteDialogOpen(true);
                        }}
                      >
                        <DeleteIcon />
                      </IconButton>
                    </TableCell>
                  </TableRow>
                ))
              )}
            </TableBody>
          </Table>
        </TableContainer>

        <TablePagination
          component="div"
          count={totalCount}
          page={page}
          onPageChange={(_, newPage) => setPage(newPage)}
          rowsPerPage={rowsPerPage}
          onRowsPerPageChange={(e) => {
            setRowsPerPage(parseInt(e.target.value, 10));
            setPage(0);
          }}
        />

        <Dialog open={deleteDialogOpen} onClose={() => setDeleteDialogOpen(false)}>
          <DialogTitle>Confirm Delete</DialogTitle>
          <DialogContent>Are you sure you want to delete this bet? This action cannot be undone.</DialogContent>
          <DialogActions>
            <Button onClick={() => setDeleteDialogOpen(false)}>Cancel</Button>
            <Button
              onClick={handleDelete}
              color="error"
              variant="contained"
            >
              Delete
            </Button>
          </DialogActions>
        </Dialog>
      </Paper>
    </Box>
  );
};

// ------------------ MarketHistory adapted to MUI layout ------------------
const MarketHistory: React.FC = () => {
  const staffId = localStorage.getItem("currentUserId");
  const token = localStorage.getItem("token");

  const [rows, setRows] = useState<MarketWin[]>([]);
  const [searchTerm, setSearchTerm] = useState<string>("");
  const [userId, setUserId] = useState<string>("");
  const [loading, setLoading] = useState<boolean>(false);
  const [pagination, setPagination] = useState<PaginationState>({
    totalCount: 0,
    totalPages: 0,
    currentPage: 1,
    limit: 10,
  });

  const BASE_URL = useMemo(() => {
    return (
      (import.meta as any)?.env?.VITE_API_BASE_URL ||
      process.env.REACT_APP_API_BASE_URL ||
      "https://api.air1bet.com"
    );
  }, []);

  useEffect(() => {
    const t = setTimeout(() => {
      fetchData();
    }, 400);
    return () => clearTimeout(t);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [searchTerm, userId, pagination.currentPage]);

  const fetchData = async () => {
    setLoading(true);
    try {
      let url = `${BASE_URL}/sportsbetting/marketwins?page=${pagination.currentPage}&limit=${pagination.limit}`;
      if (searchTerm) url += `&search=${encodeURIComponent(searchTerm)}`;
      if (userId) url += `&userId=${encodeURIComponent(userId)}`;

      const headers: Record<string, string> = {
        ...(token ? { Authorization: `Bearer ${token}` } : {}),
        ...(staffId ? { "x-staff-id": staffId } : {}),
      };

      const { data } = await axios.get(url, { headers });
      // Support flexible shape
      const list: MarketWin[] = (data as any).data || (data as any).transactions || [];
      setRows(list);
      setPagination((prev) => ({ ...prev, ...(data as any).pagination }));
    } catch (e) {
      console.error("Failed to fetch market history:", e);
    } finally {
      setLoading(false);
    }
  };

  const formatNumber = (n: number | null | undefined) =>
    typeof n === "number" && !Number.isNaN(n) ? n.toFixed(2) : "-";

  const formatDate = (iso: string) => (iso ? new Date(iso).toLocaleString() : "-");

  return (
    <Box sx={{ }}>
      <Paper sx={{ p: 3 }}>
        <Typography variant="h6" gutterBottom>
          Market results
        </Typography>

        <Grid container spacing={2} sx={{ mb: 2 }}>
          <Grid item xs={12} md={6}>
            <TextField
              label="Search by match / event / winner"
              value={searchTerm}
              onChange={(e) => {
                setSearchTerm(e.target.value);
                setPagination((p) => ({ ...p, currentPage: 1 }));
              }}
              fullWidth
              size="small"
            />
          </Grid>
          <Grid item xs={12} md={6}>
            <TextField
              label="User ID"
              value={userId}
              onChange={(e) => {
                setUserId(e.target.value);
                setPagination((p) => ({ ...p, currentPage: 1 }));
              }}
              fullWidth
              size="small"
            />
          </Grid>
        </Grid>

        <Typography variant="body2" sx={{ mb: 2 }}>
          Showing {rows.length} of {pagination.totalCount} records
        </Typography>

        {loading && rows.length === 0 ? (
          <Box sx={{ display: "flex", justifyContent: "center", py: 6 }}>
            <CircularProgress />
          </Box>
        ) : rows.length === 0 ? (
          <Typography>No market results found</Typography>
        ) : (
          <TableContainer>
            <Table size="small">
              <TableHead>
                <TableRow>
                  <TableCell>Match</TableCell>
                  <TableCell>Winner</TableCell>
                  <TableCell align="right">Team1Ex</TableCell>
                  <TableCell align="right">Team2Ex</TableCell>
                  <TableCell align="right">DrawEx</TableCell>
                  <TableCell align="right">Total Bets</TableCell>
                  <TableCell align="right">Payout</TableCell>
                  <TableCell>User</TableCell>
                  <TableCell>Match ID</TableCell>
                  <TableCell>Event ID</TableCell>
                  <TableCell>Date</TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {rows.map((r) => (
                  <TableRow key={r.id}>
                    <TableCell>{r.matchname}</TableCell>
                    <TableCell>{r.winteam}</TableCell>
                    <TableCell align="right">{formatNumber(r.team1ex)}</TableCell>
                    <TableCell align="right">{formatNumber(r.team2ex)}</TableCell>
                    <TableCell align="right">{formatNumber(r.drawex ?? null)}</TableCell>
                    <TableCell align="right">{formatNumber(r.totalbets)}</TableCell>
                    <TableCell align="right">{formatNumber(r.payout)}</TableCell>
                    <TableCell>{r.user_id ?? "-"}</TableCell>
                    <TableCell>{r.matchid}</TableCell>
                    <TableCell>{r.eventid}</TableCell>
                    <TableCell>{formatDate(r.created_at)}</TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </TableContainer>
        )}

        {pagination.totalPages > 1 && (
          <Box sx={{ display: "flex", justifyContent: "center", gap: 2, mt: 2 }}>
            <Button
              variant="outlined"
              onClick={() => setPagination((p) => ({ ...p, currentPage: Math.max(1, p.currentPage - 1) }))}
              disabled={pagination.currentPage === 1}
            >
              Previous
            </Button>
            <Typography sx={{ alignSelf: "center" }}>
              {pagination.currentPage} / {pagination.totalPages || 1}
            </Typography>
            <Button
              variant="outlined"
              onClick={() =>
                setPagination((p) => ({ ...p, currentPage: Math.min(p.totalPages || 1, p.currentPage + 1) }))
              }
              disabled={pagination.currentPage === pagination.totalPages || pagination.totalPages === 0}
            >
              Next
            </Button>
          </Box>
        )}
      </Paper>
    </Box>
  );
};

// ------------------ FanWins adapted to MUI ------------------
const FanWins: React.FC = () => {
  const staffId = localStorage.getItem("currentUserId");
  const token = localStorage.getItem("token");

  const [rows, setRows] = useState<FanWin[]>([]);
  const [searchTerm, setSearchTerm] = useState<string>("");
  const [userId, setUserId] = useState<string>("");
  const [loading, setLoading] = useState<boolean>(false);
  const [pagination, setPagination] = useState<PaginationState>({
    totalCount: 0,
    totalPages: 0,
    currentPage: 1,
    limit: 10,
  });

  const BASE_URL = useMemo(() => {
    return (
      (import.meta as any)?.env?.VITE_API_BASE_URL ||
      process.env.REACT_APP_API_BASE_URL ||
      "https://api.air1bet.com"
    );
  }, []);

  useEffect(() => {
    const t = setTimeout(() => fetchData(), 400);
    return () => clearTimeout(t);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [searchTerm, userId, pagination.currentPage]);

  const fetchData = async () => {
    setLoading(true);
    try {
      let url = `${BASE_URL}/sportsbetting/fanwins?page=${pagination.currentPage}&limit=${pagination.limit}`;
      if (searchTerm) url += `&search=${encodeURIComponent(searchTerm)}`;
      if (userId) url += `&userId=${encodeURIComponent(userId)}`;

      const headers: Record<string, string> = {
        ...(token ? { Authorization: `Bearer ${token}` } : {}),
        ...(staffId ? { "x-staff-id": staffId } : {}),
      };

      const { data } = await axios.get(url, { headers });
      // shape: { data: FanWin[], pagination }
      setRows((data as any).data || []);
      setPagination((p) => ({ ...p, ...(data as any).pagination }));
    } catch (e) {
      console.error("Failed to fetch fan wins:", e);
    } finally {
      setLoading(false);
    }
  };

  const formatCurrency = (n: number) => (typeof n === "number" && !Number.isNaN(n) ? n.toFixed(2) : "-");
  const formatDate = (iso: string) => (iso ? new Date(iso).toLocaleString() : "-");

  return (
    <Box sx={{ }}>
      <Paper sx={{ p: 3 }}>
        <Typography variant="h6" gutterBottom>
          Fancy results
        </Typography>

        <Grid container spacing={2} sx={{ mb: 2 }}>
          <Grid item xs={12} md={6}>
            <TextField
              label="Search (fancy / selection / event / match)"
              value={searchTerm}
              onChange={(e) => {
                setSearchTerm(e.target.value);
                setPagination((p) => ({ ...p, currentPage: 1 }));
              }}
              fullWidth
              size="small"
            />
          </Grid>
          <Grid item xs={12} md={6}>
            <TextField
              label="User ID"
              value={userId}
              onChange={(e) => {
                setUserId(e.target.value);
                setPagination((p) => ({ ...p, currentPage: 1 }));
              }}
              fullWidth
              size="small"
            />
          </Grid>
        </Grid>

        <Typography variant="body2" sx={{ mb: 2 }}>
          Showing {rows.length} of {pagination.totalCount} records
        </Typography>

        {loading && rows.length === 0 ? (
          <Box sx={{ display: "flex", justifyContent: "center", py: 6 }}>
            <CircularProgress />
          </Box>
        ) : rows.length === 0 ? (
          <Typography>No fan wins found</Typography>
        ) : (
          <TableContainer>
            <Table size="small">
              <TableHead>
                <TableRow>
                  <TableCell>User ID</TableCell>
                  <TableCell>Fancy</TableCell>
                  <TableCell>Selection</TableCell>
                  <TableCell align="right">Runs/Odds</TableCell>
                  <TableCell align="right">Payout</TableCell>
                  <TableCell>Event ID</TableCell>
                  <TableCell>Match ID</TableCell>
                  <TableCell>Date</TableCell>
                </TableRow>
              </TableHead>
              <TableBody>
                {rows.map((r) => (
                  <TableRow key={r.id}>
                    <TableCell>{r.userid}</TableCell>
                    <TableCell>{r.fancyname}</TableCell>
                    <TableCell>{r.selection}</TableCell>
                    <TableCell align="right">{r.runsodds}</TableCell>
                    <TableCell align="right">{formatCurrency(r.payout)}</TableCell>
                    <TableCell>{r.eventid}</TableCell>
                    <TableCell>{r.matchid}</TableCell>
                    <TableCell>{formatDate(r.created_at)}</TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          </TableContainer>
        )}

        {pagination.totalPages > 1 && (
          <Box sx={{ display: "flex", justifyContent: "center", gap: 2, mt: 2 }}>
            <Button
              variant="outlined"
              onClick={() => setPagination((p) => ({ ...p, currentPage: Math.max(1, p.currentPage - 1) }))}
              disabled={pagination.currentPage === 1}
            >
              Previous
            </Button>
            <Typography sx={{ alignSelf: "center" }}>
              {pagination.currentPage} / {pagination.totalPages || 1}
            </Typography>
            <Button
              variant="outlined"
              onClick={() =>
                setPagination((p) => ({ ...p, currentPage: Math.min(p.totalPages || 1, p.currentPage + 1) }))
              }
              disabled={pagination.currentPage === pagination.totalPages || pagination.totalPages === 0}
            >
              Next
            </Button>
          </Box>
        )}
      </Paper>
    </Box>
  );
};

// ------------------ Parent Combined Component with Tabs ------------------
const CombinedSportsResults: React.FC = () => {
  const [tab, setTab] = useState<number>(0); // 0: Sports Bets, 1: Results
  const [resultSubTab, setResultSubTab] = useState<number>(0); // 0: Market, 1: Fancy

  return (
    <Box sx={{ width: "100%", pl: 24 ,pt:2 }}>
      <Paper sx={{ p: 2 }}>
        <Tabs value={tab} onChange={(_, v) => setTab(v)}>
          <Tab label="Sports Bets" />
          <Tab label="Results" />
        </Tabs>
      </Paper>

      <TabPanel value={tab} index={0}>
        <SportsBetting />
      </TabPanel>

      <TabPanel value={tab} index={1}>
        <Paper sx={{  mb: 2 }}>
          <Tabs value={resultSubTab} onChange={(_, v) => setResultSubTab(v)}>
            <Tab label="Market result" />
            <Tab label="Fancy result" />
          </Tabs>
        </Paper>

        <TabPanel value={resultSubTab} index={0}>
          <MarketHistory />
        </TabPanel>
        <TabPanel value={resultSubTab} index={1}>
          <FanWins />
        </TabPanel>
      </TabPanel>
    </Box>
  );
};

export default CombinedSportsResults;
