import { CurrencyEuroIcon } from '@heroicons/react/20/solid';
import axios from 'axios';
import React, { useState, useEffect } from 'react';

interface House {
  uid: string;
  name: string;
  btc: number;
  eth: number;
  ltc: number;
  bch: number;
  usdt: number;
  trx: number;
  doge: number;
  ada: number;
  xrp: number;
  bnb: number;
  usdp: number;
  nexo: number;
  mkr: number;
  tusd: number;
  usdc: number;
  busd: number;
  nc: number;
  inr: number;
  shib: number;
  matic: number;
  bjb: number;
  sc: number;
  mvr: number;
  aed: number;
  npr: number;
  pkr: number;
  bdt: number;
}

interface TransactionHistory {
  id: number;
  uid: number;
  username: string;
  coin: string;
  operation: string;
  amount: number;
  previous_balance: number;
  new_balance: number;
  transaction_time: string;
  description: string | null;
}

interface WalletHistoryResponse {
  history: TransactionHistory[];
  count: number;
}

const Wallet: React.FC = () => {
  const [transferDetails, setTransferDetails] = useState<House[]>([]);
  const [debitValues, setDebitValues] = useState<Record<string, string>>({});
  const [creditValues, setCreditValues] = useState<Record<string, string>>({});
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCurrency, setSelectedCurrency] = useState<Record<string, string>>({});
  const [isLoading, setIsLoading] = useState(true);
  const [showToast, setShowToast] = useState(false);
  const [toastMessage, setToastMessage] = useState('');
  const [toastType, setToastType] = useState<'success' | 'error'>('success');
  
  // States for wallet history
  const [showHistoryModal, setShowHistoryModal] = useState(false);
  const [selectedUser, setSelectedUser] = useState<{uid: string, name: string} | null>(null);
  const [transactionHistory, setTransactionHistory] = useState<TransactionHistory[]>([]);
  const [historyLoading, setHistoryLoading] = useState(false);
  const [historyCoin, setHistoryCoin] = useState<string>('');

  const currencies = [
    'btc', 'eth', 'ltc', 'bch', 'usdt', 'trx', 'doge', 'ada',
    'xrp', 'bnb', 'usdp', 'nexo', 'mkr', 'tusd', 'usdc', 'busd',
    'nc', 'inr', 'shib', 'matic', 'bjb', 'sc', 'mvr', 'aed', 'npr', 'pkr','bdt'
  ];

  useEffect(() => {
    fetchTransferDetails();
  }, []);

  // Handle toast messages
  useEffect(() => {
    if (showToast) {
      const timer = setTimeout(() => {
        setShowToast(false);
      }, 3000);
      return () => clearTimeout(timer);
    }
  }, [showToast]);

  const handleCurrencyChange = (uid: string, currency: string) => {
    setSelectedCurrency(prev => ({ ...prev, [uid]: currency }));
  };

  const showToastNotification = (message: string, type: 'success' | 'error') => {
    setToastMessage(message);
    setToastType(type);
    setShowToast(true);
  };

  const fetchTransferDetails = async () => {
    setIsLoading(true);
    try {
      const response = await fetch('https://api.air1bet.com/getwallet', {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }
      const data = await response.json();
      setTransferDetails(data);
    } catch (error) {
      console.error('Error fetching transfer details:', error);
      setTransferDetails([]);
      showToastNotification('Failed to load wallet data', 'error');
    } finally {
      setIsLoading(false);
    }
  };

  const fetchWalletHistory = async (uid: string, coin?: string) => {
    setHistoryLoading(true);
    try {
      let url = `https://api.air1bet.com/wallethistory/${uid}`;
      if (coin) {
        url += `?coin=${coin}`;
      }
      
      const response = await axios.get<WalletHistoryResponse>(url, {
        headers: {
          'Authorization': `Bearer ${localStorage.getItem('token')}`
        }
      });
      setTransactionHistory(response.data.history);
    } catch (error) {
      console.error('Error fetching wallet history:', error);
      setTransactionHistory([]);
      showToastNotification('Failed to load transaction history', 'error');
    } finally {
      setHistoryLoading(false);
    }
  };

  const openHistoryModal = (user: House) => {
    setSelectedUser({uid: user.uid, name: user.name});
    setHistoryCoin('');
    setShowHistoryModal(true);
    fetchWalletHistory(user.uid);
  };

  const handleHistoryCoinChange = (coin: string) => {
    setHistoryCoin(coin);
    if (selectedUser) {
      fetchWalletHistory(selectedUser.uid, coin || undefined);
    }
  };

  const handleDebit = async (uid: string, coin: string, debitAmount: string ,currentbalance:string) => {
    if (!debitAmount || isNaN(Number(debitAmount)) || Number(debitAmount) <= 0) {
      showToastNotification('Please enter a valid amount', 'error');
      return;
    }
    if(currentbalance && parseFloat(currentbalance) < parseFloat(debitAmount)) {
      showToastNotification('Insufficient balance for debit operation', 'error');
      return;
    }

    try {
     const response = await axios.post(
  'https://api.air1bet.com/updatebalance',
  {
    uid,
    coin,
    amount: debitAmount,
    operation: 'debit'
  },
  {
    headers: {
      'Authorization': `Bearer ${localStorage.getItem('token')}`,
      'Content-Type': 'application/json'
    }
  }
);

      // Update local state to reflect the change
      await fetchTransferDetails();
      
      // Reset the debit input field for this user
      setDebitValues(prevValues => ({
        ...prevValues,
        [uid]: '',
      }));
       window.dispatchEvent(new Event("balance-changed"));
      showToastNotification(`Successfully debited ${debitAmount} ${coin.toUpperCase()}`, 'success');

    } catch (error) {
      console.error('Error updating wallet:', error);
      showToastNotification('Failed to process debit operation', 'error');
    }
  };

  const handleCredit = async (uid: string, coin: string, creditAmount: string) => {
    if (!creditAmount || isNaN(Number(creditAmount)) || Number(creditAmount) <= 0) {
      showToastNotification('Please enter a valid amount', 'error');
      return;
    }

    try {
      const response = await axios.post('https://api.air1bet.com/updatebalance', {
        uid,
        coin,
        amount: creditAmount,
        operation: 'credit'
      },
  {
    headers: {
      'Authorization': `Bearer ${localStorage.getItem('token')}`,
      'Content-Type': 'application/json'
    }
  });

      // Update local state to reflect the change
      await fetchTransferDetails();
      
      // Reset the credit input field for this user
      setCreditValues(prevValues => ({
        ...prevValues,
        [uid]: '',
      }));
      
      showToastNotification(`Successfully credited ${creditAmount} ${coin.toUpperCase()}`, 'success');
    } catch (error) {
      console.error('Error updating wallet:', error);
      showToastNotification('Failed to process credit operation', 'error');
    }
  };

  const handleInputChange = (uid: string, value: string, type: 'debit' | 'credit') => {
    // Only allow numbers and decimals
    const sanitizedValue = value.replace(/[^0-9.]/g, '');
    
    if (type === 'debit') {
      setDebitValues(prevValues => ({
        ...prevValues,
        [uid]: sanitizedValue,
      }));
    } else if (type === 'credit') {
      setCreditValues(prevValues => ({
        ...prevValues,
        [uid]: sanitizedValue,
      }));
    }
  };

  const handleSearchChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    setSearchTerm(event.target.value);
  };

  // Format balance to 4 decimal places
  const formatBalance = (value: number): string => {
    return value.toFixed(4);
  };

  // Format date for transaction history
  const formatDate = (dateString: string): string => {
    const date = new Date(dateString);
    return date.toLocaleString();
  };

  // Filter users based on the search term
  const filteredUsers = transferDetails.filter((user) =>
    user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    user.uid.toString().includes(searchTerm)
  );

  return (
    <div className="p-4 lg:ml-64 bg-gray-50 min-h-screen">
      {/* Toast Notification */}
      {showToast && (
        <div className={`fixed top-4 right-4 p-4 rounded-lg shadow-lg z-50 ${
          toastType === 'success' ? 'bg-green-500' : 'bg-red-500'
        } text-white`}>
          {toastMessage}
        </div>
      )}
      
      {/* Header and Search */}
      <div className="mb-6">
        <h1 className="text-2xl font-bold text-gray-800 mb-4">Wallet Management</h1>
        <div className="relative">
          <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
            <svg className="w-5 h-5 text-gray-500" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"></path>
            </svg>
          </div>
          <input
            type="text"
            placeholder="Search by name or UID"
            value={searchTerm}
            onChange={handleSearchChange}
            className="block w-full p-4 pl-10 text-sm text-gray-900 border border-gray-300 rounded-lg bg-white focus:ring-blue-500 focus:border-blue-500"
          />
        </div>
      </div>

      {/* Loading Spinner */}
      {isLoading ? (
        <div className="flex justify-center items-center h-64">
          <div className="animate-spin rounded-full h-12 w-12 border-t-2 border-b-2 border-blue-500"></div>
        </div>
      ) : (
        <div className="overflow-x-auto relative shadow-md sm:rounded-lg">
          <table className="w-full text-sm text-left text-gray-500">
            <thead className="text-xs text-white uppercase bg-blue-600">
              <tr>
                <th scope="col" className="py-3 px-4">S.No</th>
                <th scope="col" className="py-3 px-4">Name</th>
                <th scope="col" className="py-3 px-4">UID</th>
                <th scope="col" className="py-3 px-4">Coin</th>
                <th scope="col" className="py-3 px-4">Balance</th>
                <th scope="col" className="py-3 px-4">Actions</th>
                <th scope="col" className="py-3 px-4">History</th>
              </tr>
            </thead>
            <tbody>
              {filteredUsers.length > 0 ? (
                filteredUsers.map((user, index) => (
                  <React.Fragment key={user.uid}>
                    <tr className="bg-white border-b hover:bg-gray-50">
                      <td className="py-4 px-4">{index + 1}</td>
                      <td className="py-4 px-4 font-medium text-gray-900">{user.name}</td>
                      <td className="py-4 px-4">{user.uid}</td>
                      <td className="py-4 px-4">
                        <select 
                          value={selectedCurrency[user.uid] || 'btc'} 
                          onChange={(e) => handleCurrencyChange(user.uid, e.target.value)}
                          className="bg-gray-50 border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-blue-500 focus:border-blue-500 block w-full p-2.5"
                        >
                          {currencies.map(currency => (
                            <option key={currency} value={currency}>{currency.toUpperCase()}</option>
                          ))}
                        </select>
                      </td>
                      <td className="py-4 px-4 font-medium">
                        {formatBalance(user[selectedCurrency[user.uid] as keyof House] as number || user.btc)}
                      </td>
                      <td className="py-4 px-4 flex space-x-2">
                        <button
                          onClick={() => openHistoryModal(user)}
                          className="text-white bg-blue-600 hover:bg-blue-700 focus:ring-4 focus:outline-none focus:ring-blue-300 font-medium rounded-lg text-sm px-4 py-2"
                        >
                          View History
                        </button>
                      </td>
                      <td className="py-4 px-4">
                        <button
                          onClick={() => {
                            const row = document.getElementById(`actions-${user.uid}`);
                            if (row) {
                              row.classList.toggle('hidden');
                            }
                          }}
                          className="text-white bg-purple-600 hover:bg-purple-700 focus:ring-4 focus:outline-none focus:ring-purple-300 font-medium rounded-lg text-sm px-4 py-2 w-full"
                        >
                          Manage Funds
                        </button>
                      </td>
                    </tr>
                    {/* Expandable Action Row - Debit/Credit Controls */}
                    <tr id={`actions-${user.uid}`} className="hidden bg-gray-50">
                      <td colSpan={7} className="p-4">
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                          {/* Debit Controls */}
                          <div className="bg-white p-4 rounded-lg shadow">
                            <h4 className="font-medium text-red-600 mb-2">Debit Funds</h4>
                            <div className="flex items-center space-x-2">
                              <input
                                type="text"
                                placeholder="Amount to debit"
                                value={debitValues[user.uid] || ''}
                                onChange={(e) => handleInputChange(user.uid, e.target.value, 'debit')}
                                className="bg-gray-50 border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-red-500 focus:border-red-500 block w-full p-2.5"
                              />
                             <button
            onClick={() => handleDebit(
              user.uid, 
              selectedCurrency[user.uid] || 'btc', 
              debitValues[user.uid] || '0', 
              String(user[selectedCurrency[user.uid] as keyof House] ?? user.btc)
            )}
                                className="text-white bg-red-600 hover:bg-red-700 focus:ring-4 focus:outline-none focus:ring-red-300 font-medium rounded-lg text-sm px-4 py-2.5 whitespace-nowrap"
                              >
                                Debit Funds
                              </button>
                            </div>
                          </div>
                          
                          {/* Credit Controls */}
                          <div className="bg-white p-4 rounded-lg shadow">
                            <h4 className="font-medium text-green-600 mb-2">Credit Funds</h4>
                            <div className="flex items-center space-x-2">
                              <input
                                type="text"
                                placeholder="Amount to credit"
                                value={creditValues[user.uid] || ''}
                                onChange={(e) => handleInputChange(user.uid, e.target.value, 'credit')}
                                className="bg-gray-50 border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-green-500 focus:border-green-500 block w-full p-2.5"
                              />
                              <button
                                onClick={() => handleCredit(user.uid, selectedCurrency[user.uid] || 'btc', creditValues[user.uid] || '0')}
                                className="text-white bg-green-600 hover:bg-green-700 focus:ring-4 focus:outline-none focus:ring-green-300 font-medium rounded-lg text-sm px-4 py-2.5 whitespace-nowrap"
                              >
                                Credit Funds
                              </button>
                            </div>
                          </div>
                        </div>
                      </td>
                    </tr>
                  </React.Fragment>
                ))
              ) : (
                <tr>
                  <td colSpan={7} className="py-4 px-4 text-center">
                    {searchTerm ? 'No users found matching your search.' : 'No users found in the system.'}
                  </td>
                </tr>
              )}
            </tbody>
          </table>
        </div>
      )}

      {/* Transaction History Modal */}
      {showHistoryModal && selectedUser && (
        <div className="fixed inset-0 bg-gray-600 bg-opacity-75 overflow-y-auto h-full w-full z-50 flex items-center justify-center">
          <div className="relative bg-white rounded-lg shadow-xl max-w-4xl w-full mx-4 max-h-screen">
            <div className="flex justify-between items-center p-5 border-b rounded-t bg-blue-600 text-white">
              <h3 className="text-xl font-semibold">
                Transaction History - {selectedUser.name} (UID: {selectedUser.uid})
              </h3>
              <button
                type="button"
                onClick={() => setShowHistoryModal(false)}
                className="text-white bg-transparent hover:bg-blue-700 rounded-lg text-sm p-2 ml-auto inline-flex items-center"
              >
                <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg">
                  <path fillRule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clipRule="evenodd"></path>
                </svg>
              </button>
            </div>

            <div className="p-6">
              <div className="mb-4">
                <label className="block text-sm font-medium text-gray-700 mb-2">Filter by Currency</label>
                <select
                  value={historyCoin}
                  onChange={(e) => handleHistoryCoinChange(e.target.value)}
                  className="bg-gray-50 border border-gray-300 text-gray-900 text-sm rounded-lg focus:ring-blue-500 focus:border-blue-500 block w-full p-2.5"
                >
                  <option value="">All Currencies</option>
                  {currencies.map(currency => (
                    <option key={currency} value={currency}>{currency.toUpperCase()}</option>
                  ))}
                </select>
              </div>

              {historyLoading ? (
                <div className="flex justify-center items-center py-10">
                  <div className="animate-spin rounded-full h-10 w-10 border-t-2 border-b-2 border-blue-500"></div>
                </div>
              ) : transactionHistory.length > 0 ? (
                <div className="overflow-x-auto max-h-96">
                  <table className="w-full text-sm text-left text-gray-500">
                    <thead className="text-xs text-white uppercase bg-blue-600 sticky top-0">
                      <tr>
                        <th scope="col" className="py-3 px-4">Date & Time</th>
                        <th scope="col" className="py-3 px-4">Coin</th>
                        <th scope="col" className="py-3 px-4">Operation</th>
                        <th scope="col" className="py-3 px-4">Amount</th>
                        <th scope="col" className="py-3 px-4">Previous Balance</th>
                        <th scope="col" className="py-3 px-4">New Balance</th>
                      </tr>
                    </thead>
                    <tbody>
                      {transactionHistory.map((transaction) => (
                        <tr key={transaction.id} className="bg-white border-b hover:bg-gray-50">
                          <td className="py-3 px-4">{formatDate(transaction.transaction_time)}</td>
                          <td className="py-3 px-4 uppercase">{transaction.coin}</td>
                          <td className="py-3 px-4">
                            <span className={`px-2 py-1 rounded text-white ${
                              transaction.operation === 'credit' ? 'bg-green-500' : 'bg-red-500'
                            }`}>
                              {transaction.operation}
                            </span>
                          </td>
                          <td className="py-3 px-4">{transaction.amount}</td>
                          <td className="py-3 px-4">{transaction.previous_balance}</td>
                          <td className="py-3 px-4">{transaction.new_balance}</td>
                        </tr>
                      ))}
                    </tbody>
                  </table>
                </div>
              ) : (
                <div className="py-10 text-center text-gray-500">
                  No transaction history found {historyCoin && `for ${historyCoin.toUpperCase()}`}.
                </div>
              )}
            </div>

            <div className="flex items-center justify-end p-6 border-t border-gray-200 rounded-b">
              <button
                type="button"
                onClick={() => setShowHistoryModal(false)}
                className="text-white bg-blue-600 hover:bg-blue-700 focus:ring-4 focus:outline-none focus:ring-blue-300 font-medium rounded-lg text-sm px-5 py-2.5"
              >
                Close
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default Wallet;