/*  ──────────────────────────────────────────────────────────────
    MyAccountTab.tsx
    • Only “SuperAdmin” (case–insensitive) can edit profile
      or change password.
    • All other roles see the same UI but everything is read-only.
    • No layout / colour tweaks were removed.
    ────────────────────────────────────────────────────────────── */

import React, { useEffect, useState } from "react";
import {
  Box, Card, CardContent, Stack, Avatar, Chip, Typography, TextField,
  InputAdornment, IconButton, Button, Snackbar, Alert, CircularProgress
} from "@mui/material";
import {
  User, Mail, Phone, Globe, Eye, EyeOff, KeyRound, Lock,
  Edit, Save, X, Shield
} from "lucide-react";
import { apiFetch } from "../../utils/api";
import {useWhatsappRef,  WaRef } from "../../hooks/waHook";
/* ---------- helpers & constants -------------------------------------- */
interface UserData {
  name: string;
  email: string;
  role: string;
  phone: string;
  country: string;
}


/* =========================================================================
   COMPONENT
   ========================================================================= */
export default function MyAccountTab() {

    // ── Grab staffId from localStorage (as number) ─────────────────────────
 const rawId = localStorage.getItem("currentUserId");
  const staffId = rawId ? parseInt(rawId, 10) : null;

  // ── Fetch the user’s WhatsApp referral info ─────────────────────────────
  const waRef: WaRef | null = useWhatsappRef(staffId);
 
  /* ── base state ──────────────────────────────────────────────────── */
  const [me,  setMe]      = useState<UserData | null>(null);
  const [editMode, setEM] = useState(false);
  const [editData, setED] = useState<Partial<UserData>>({});

  /* ── password section ───────────────────────────────────────────── */
  const [oldPwd, setOld]  = useState("");
  const [newPwd, setNew]  = useState("");
  const [showOld, setSO]  = useState(false);
  const [showNew, setSN]  = useState(false);

  /* ── misc UI state ──────────────────────────────────────────────── */
  const [loadingPwd , setLoadPwd ] = useState(false);
  const [savingProf , setSaving  ] = useState(false);
  const [snack, setSnack] = useState<{open:boolean;msg:string;sev:"success"|"error"}>({
    open:false, msg:"", sev:"success"
  });
  const canEdit = React.useMemo(() => {
    /** role from API wins, otherwise fall back to localStorage */
    const raw = (me?.role || localStorage.getItem("userRole") || "")
                  .toLowerCase()
                  .replace(/\s+/g, "");      /* "Super Admin" → "superadmin" */
    return raw === "superadmin";
  }, [me]);
  /* ── fetch my profile once ──────────────────────────────────────── */
  useEffect(() => {
    const id = localStorage.getItem("currentUserId");
    if (!id) return;
    apiFetch<UserData>(`/api/staff/${id}`)
      .then(res => setMe({
        name:    res.name,
        email:   res.email,
        role:    res.role,
        phone:   res.phone   || "",
        country: res.country || ""
      }))
      .catch(() => {/* ignore – snack not needed on first load */});
  }, []);

  /* ─── helpers ───────────────────────────────────────────────────── */
  const startEdit   = () => { if (canEdit) { setED(me ?? {}); setEM(true);} };
  const cancelEdit  = () => { setEM(false); setED({}); };
  const field       = (k:keyof UserData) => editData[k] ?? "";

  /* ─── save profile ──────────────────────────────────────────────── */
  const saveProfile = async () => {
    if (!canEdit) return;
    const id = localStorage.getItem("currentUserId");
    if (!id)  return;

    setSaving(true);
    try {
      await apiFetch(`/api/staff/${id}`, {
        method: "PATCH",
        body: JSON.stringify({ name:field("name"), phone:field("phone"), country:field("country") })
      });
      setMe(prev => prev ? { ...prev, ...editData } as UserData : prev);
      setSnack({open:true,msg:"Profile updated!",sev:"success"});
      cancelEdit();
    } catch (e:any) {
      setSnack({open:true,msg:e.message||"Update failed",sev:"error"});
    } finally { setSaving(false); }
  };

  /* ─── change password ───────────────────────────────────────────── */
  const changePwd = async () => {
    // if (!canEdit) return;
    if (!oldPwd || !newPwd)
      return setSnack({open:true,msg:"Fill both password fields",sev:"error"});

    setLoadPwd(true);
    try {
      await apiFetch("/api/staff/password", {
        method:"PATCH",
        body:JSON.stringify({ oldPassword:oldPwd, newPassword:newPwd })
      });
      setSnack({open:true,msg:"Password updated!",sev:"success"});
      setOld(""); setNew("");
    } catch (e:any) {
      setSnack({open:true,msg:e.message||"Password change failed",sev:"error"});
    } finally { setLoadPwd(false); }
  };

  /* ─── loading skeleton ──────────────────────────────────────────── */
  if (!me)
    return (
      <Box sx={{display:"flex",justifyContent:"center",alignItems:"center",minHeight:"60vh"}}>
        <CircularProgress/>
      </Box>
    );

  /* ===================================================================
     RENDER
     =================================================================== */
  return (
    <Box sx={{p:3,background:"#f8fafc",minHeight:"100vh"}}>
      <Box sx={{maxWidth:800,mx:"auto"}}>
        <Stack spacing={3}>

          {/* ───────── Profile card ───────── */}
          <Card elevation={0} sx={{borderRadius:3,border:"1px solid #e2e8f0",
                                   background:"linear-gradient(135deg,#fff 0%,#f8fafc 100%)"}}>
            <CardContent sx={{p:4}}>
              {/* header */}
              <Box sx={{display:"flex",justifyContent:"space-between",mb:3,alignItems:"center"}}>
                <Box sx={{display:"flex",alignItems:"center",gap:3}}>
                  <Avatar sx={{width:80,height:80,bgcolor:"#6366f1",fontSize:"2rem",fontWeight:700}}>
                    {(editMode?field("name"):me.name).charAt(0).toUpperCase()}
                  </Avatar>
                  <Box>
                    <Typography variant="h5" sx={{fontWeight:700,color:"#0f172a",mb:1}}>
                      {editMode?field("name"):me.name}
                    </Typography>
                    <Chip label={me.role} size="small"
                          sx={{bgcolor:"#f0f0ff",color:"#6366f1",fontWeight:600}}/>
                  </Box>
                </Box>

                {/* edit buttons */}
                {canEdit && (
                  editMode ? (
                    <Box sx={{display:"flex",gap:1}}>
                      <IconButton onClick={saveProfile} disabled={savingProf}
                                  sx={{bgcolor:"#16a34a",color:"#fff",
                                       "&:hover":{bgcolor:"#15803d"}}}>
                        {savingProf?<CircularProgress size={20}/> : <Save size={20}/>}
                      </IconButton>
                      <IconButton onClick={cancelEdit}
                                  sx={{bgcolor:"#dc2626",color:"#fff",
                                       "&:hover":{bgcolor:"#b91c1c"}}}>
                        <X size={20}/>
                      </IconButton>
                    </Box>
                  ) : (
                    <IconButton onClick={startEdit}
                                sx={{bgcolor:"#6366f1",color:"#fff",
                                     "&:hover":{bgcolor:"#4f46e5"}}}>
                      <Edit size={20}/>
                    </IconButton>
                  )
                )}
              </Box>

              {/* grid of fields */}
              <Box sx={{display:"grid",gridTemplateColumns:{xs:"1fr",md:"1fr 1fr"},gap:3}}>
                {/* Name */}
                <ProfileRow icon={<User size={20}/>} iconBg="#f0fdf4" iconColor="#16a34a"
                            label="Full Name" editMode={editMode}
                            value={editMode?field("name"):me.name}
                            onChange={v=>setED(d=>({...d,name:v}))}/>
                {/* Email – readonly */}
                <ProfileRow icon={<Mail size={20}/>} iconBg="#fffbeb" iconColor="#f59e0b"
                            label="Email" value={me.email}/>
                {/* Phone */}
                <ProfileRow icon={<Phone size={20}/>} iconBg="#f0f9ff" iconColor="#0ea5e9"
                            label="Phone Number" editMode={editMode}
                            value={editMode?field("phone"):me.phone||"Not provided"}
                            onChange={v=>setED(d=>({...d,phone:v}))}/>
                {/* Country */}
                <ProfileRow icon={<Globe size={20}/>} iconBg="#fdf2f8" iconColor="#ec4899"
                            label="Country" editMode={editMode}
                            value={editMode?field("country"):me.country||"Not provided"}
                            onChange={v=>setED(d=>({...d,country:v}))}/>
              </Box>
            </CardContent>
          </Card>

          {/* ───────── Password card (SuperAdmin only) ───────── */}
          
            <Card elevation={0} sx={{borderRadius:3,border:"1px solid #e2e8f0",bgcolor:"#fff"}}>
              <CardContent sx={{p:4}}>
                <Box sx={{display:"flex",alignItems:"center",gap:2,mb:3}}>
                  <Box sx={{p:1.5,borderRadius:2,bgcolor:"#fee2e2",color:"#dc2626"}}>
                    <Lock size={20}/>
                  </Box>
                  <Box>
                    <Typography variant="h6" sx={{fontWeight:700,color:"#0f172a",fontSize:18}}>
                      Change Password
                    </Typography>
                    <Typography variant="body2" sx={{color:"#64748b"}}>
                      Update your password to keep your account secure
                    </Typography>
                  </Box>
                </Box>

                <Stack spacing={3}>
                  <PwdField label="Current Password"
                            value={oldPwd} onChange={setOld}
                            show={showOld}  setShow={setSO}/>
                  <PwdField label="New Password"
                            value={newPwd} onChange={setNew}
                            show={showNew} setShow={setSN}/>

                  <Button variant="contained" disabled={loadingPwd||!oldPwd||!newPwd}
                          onClick={changePwd}
                          startIcon={loadingPwd?<CircularProgress size={20}/>:<Shield size={20}/>}
                          sx={{py:1.5,px:4,borderRadius:2,bgcolor:"#6366f1",
                               fontWeight:600,alignSelf:"flex-start",
                               "&:hover":{bgcolor:"#4f46e5"},
                               "&:disabled":{bgcolor:"#94a3b8"}}}>
                    {loadingPwd?"Updating…":"Update Password"}
                  </Button>
                </Stack>
              </CardContent>
            </Card>
         
          {/* ───────── WhatsApp Referral Card ───────── */}
          {/* {waRef && (
            <Card elevation={0} sx={{borderRadius:3,border:"1px solid #e2e8f0"}}>
              <CardContent>
                <Typography variant="h6" sx={{mb:2,fontWeight:700}}>
                  Your WhatsApp Referral
                </Typography>
                <Typography>
                  <strong>Slug:</strong> {waRef.slug}
                </Typography>
                <Typography>
                  <strong>Phone:</strong> {waRef.phone}
                </Typography>
                <Box sx={{mt:2}}>
                  <Button
                    variant="contained"
                    href={waRef.link}
                    target="_blank"
                    sx={{
                      textTransform:"none",
                      bgcolor:"#25D366", color:"#fff",
                      "&:hover": { bgcolor:"#128C7E" }
                    }}
                  >
                    Open in WhatsApp
                  </Button>
                </Box>
              </CardContent>
            </Card>
          )} */}
        </Stack>
      </Box>

      {/* Snackbar */}
      <Snackbar open={snack.open} autoHideDuration={4000}
                onClose={()=>setSnack(s=>({...s,open:false}))}
                anchorOrigin={{vertical:"top",horizontal:"right"}}>
        <Alert severity={snack.sev} sx={{borderRadius:2,fontWeight:500}}>
          {snack.msg}
        </Alert>
      </Snackbar>
    </Box>
  );
}

/* =======================================================================
   REUSABLE SMALL SUB-COMPONENTS
   ======================================================================= */
type RowProps = {
  icon: JSX.Element;
  iconBg: string;
  iconColor: string;
  label: string;
  value: string;
  editMode?: boolean;
  onChange?: (v:string)=>void;
};
function ProfileRow({icon,iconBg,iconColor,label,value,editMode,onChange}:RowProps){
  return (
    <Box sx={{display:"flex",alignItems:"center",gap:2}}>
      <Box sx={{p:1.5,borderRadius:2,backgroundColor:iconBg,color:iconColor}}>
        {icon}
      </Box>
      <Box sx={{flex:1}}>
        <Typography variant="body2" sx={{color:"#64748b",fontSize:12}}>{label}</Typography>
        {editMode && onChange
          ? <TextField variant="standard" value={value} onChange={e=>onChange(e.target.value)}
                       sx={{"& .MuiInput-underline:before":{borderBottomColor:"#e2e8f0"},
                            "& .MuiInput-underline:hover:before":{borderBottomColor:"#6366f1"},
                            "& .MuiInput-underline:after":{borderBottomColor:"#6366f1"}}}/>
          : <Typography variant="body1" sx={{fontWeight:600,color:"#0f172a"}}>{value}</Typography>}
      </Box>
    </Box>
  );
}

function PwdField({label,value,onChange,show,setShow}:{label:string;value:string;
                   onChange:(s:string)=>void;show:boolean;setShow:(b:boolean)=>void}){
  return (
    <TextField fullWidth variant="outlined" label={label}
               type={show?"text":"password"} value={value}
               onChange={e=>onChange(e.target.value)}
               InputProps={{
                 startAdornment:<InputAdornment position="start">
                                  {label==="Current Password"
                                   ? <KeyRound size={20} style={{color:"#64748b"}}/>
                                   : <Lock size={20} style={{color:"#64748b"}}/>}
                                </InputAdornment>,
                 endAdornment:<InputAdornment position="end">
                                <Button onClick={()=>setShow(!show)}
                                        sx={{minWidth:"auto",p:1,color:"#64748b",
                                             "&:hover":{bgcolor:"transparent"}}}>
                                  {show?<EyeOff size={20}/>:<Eye size={20}/>}
                                </Button>
                              </InputAdornment>
               }}
               sx={{"& .MuiOutlinedInput-root":{bgcolor:"#f8fafc",
                     "&:hover .MuiOutlinedInput-notchedOutline":{borderColor:"#6366f1"},
                     "&.Mui-focused .MuiOutlinedInput-notchedOutline":{borderColor:"#6366f1"}}}}/>
  );
}
