const express = require('express');
const speakeasy = require('speakeasy');
const QRCode = require('qrcode');
const db = require('../General/Model');
const router = express.Router();

// Get 2FA status
router.get('/status/:uid', async (req, res) => {
    try {
        const { uid } = req.params;
        const result = await db.query(
            'SELECT is_enabled, secret_key FROM user_2fa WHERE uid = $1',
            [uid]
        );
        
        if (result.rows.length === 0) {
            return res.json({ 
                isEnabled: false,
                hasInitiated: false
            });
        }
        
        res.json({ 
            isEnabled: result.rows[0].is_enabled,
            hasInitiated: result.rows[0].secret_key !== null
        });
    } catch (error) {
        res.status(500).json({ error: 'Internal server error' });
    }
});

// Initialize 2FA setup
router.post('/enable', async (req, res) => {
  const client = await db.connect();
  try {
    const { uid } = req.body;

    await client.query('BEGIN');

    // Check if 2FA is already enabled
    const existing = await client.query(
      'SELECT is_enabled FROM user_2fa WHERE uid = $1',
      [uid]
    );

    if (existing.rows.length > 0 && existing.rows[0].is_enabled) {
      await client.query('ROLLBACK');
      return res.status(400).json({ error: '2FA is already enabled' });
    }

    // Generate secret key
    const secret = speakeasy.generateSecret({
      name: `boss707:${uid}`
    });

    // Generate QR code
    const qrCodeUrl = await QRCode.toDataURL(secret.otpauth_url);

    // Store secret in database (not enabled yet)
    await client.query(
      `INSERT INTO user_2fa (uid, is_enabled, secret_key) 
       VALUES ($1, false, $2)
       ON CONFLICT (uid) DO UPDATE SET secret_key = $2, is_enabled = false`,
      [uid, secret.base32]
    );

    await client.query('COMMIT');

    res.json({
      qrCode: qrCodeUrl,
      secret: secret.base32
    });
  } catch (error) {
    await client.query('ROLLBACK');
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    client.release();
  }
});


// Verify 2FA token and complete setup
router.post('/setup-verify', async (req, res) => {
  const client = await db.connect();
  try {
    const { uid, token } = req.body;

    await client.query('BEGIN');

    // Get user's secret (only if not already enabled)
    const result = await client.query(
      'SELECT secret_key FROM user_2fa WHERE uid = $1 AND is_enabled = false',
      [uid]
    );

    if (result.rows.length === 0) {
      await client.query('ROLLBACK');
      return res.status(400).json({ error: '2FA setup not initiated' });
    }

    const verified = speakeasy.totp.verify({
      secret: result.rows[0].secret_key,
      encoding: 'base32',
      token: token
    });

    if (!verified) {
      await client.query('ROLLBACK');
      return res.status(400).json({ error: 'Invalid token' });
    }

    // Enable 2FA after successful verification
    await client.query(
      'UPDATE user_2fa SET is_enabled = true WHERE uid = $1',
      [uid]
    );

    await client.query('COMMIT');
    res.json({ success: true });
  } catch (error) {
    await client.query('ROLLBACK');
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    client.release();
  }
});


// Verify 2FA token (for login)
router.post('/verify', async (req, res) => {
  const client = await db.connect();
  try {
    const { uid, token } = req.body;

    // Get user's secret
    const result = await client.query(
      'SELECT secret_key FROM user_2fa WHERE uid = $1 AND is_enabled = true',
      [uid]
    );

    if (result.rows.length === 0) {
      return res.status(400).json({ error: '2FA not enabled' });
    }

    const verified = speakeasy.totp.verify({
      secret: result.rows[0].secret_key,
      encoding: 'base32',
      token: token
    });

    if (verified) {
      res.json({ success: true });
    } else {
      res.status(400).json({ error: 'Invalid token' });
    }
  } catch (error) {
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    client.release();
  }
});

// Disable 2FA
router.post('/disable', async (req, res) => {
  const client = await db.connect();
  try {
    const { uid, token } = req.body;

    await client.query('BEGIN');

    // Fetch current secret key only if 2FA is enabled
    const result = await client.query(
      'SELECT secret_key FROM user_2fa WHERE uid = $1 AND is_enabled = true',
      [uid]
    );

    if (result.rows.length === 0) {
      await client.query('ROLLBACK');
      return res.status(400).json({ error: '2FA not enabled' });
    }

    const verified = speakeasy.totp.verify({
      secret: result.rows[0].secret_key,
      encoding: 'base32',
      token: token
    });

    if (!verified) {
      await client.query('ROLLBACK');
      return res.status(400).json({ error: 'Invalid token' });
    }

    // Disable 2FA and remove secret
    await client.query(
      'UPDATE user_2fa SET is_enabled = false, secret_key = null WHERE uid = $1',
      [uid]
    );

    await client.query('COMMIT');
    res.json({ success: true });
  } catch (error) {
    await client.query('ROLLBACK');
    res.status(500).json({ error: 'Internal server error' });
  } finally {
    client.release();
  }
});


module.exports = router;