const express = require('express');
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const BannerDatabase = require('./bannercontroller');

const router = express.Router();

// Configure Multer for file upload
const storage = multer.diskStorage({
    destination: (req, file, cb) => {
        const uploadDir = path.join(__dirname, 'banners');
        
        // Create banners directory if not exists
        if (!fs.existsSync(uploadDir)) {
            fs.mkdirSync(uploadDir);
        }
        
        cb(null, uploadDir);
    },
    filename: (req, file, cb) => {
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        cb(null, file.fieldname + '-' + uniqueSuffix + path.extname(file.originalname));
    }
});

const upload = multer({ 
    storage: storage,
    fileFilter: (req, file, cb) => {
        const ext = path.extname(file.originalname).toLowerCase();
        if(ext !== '.png' && ext !== '.jpg' && ext !== '.jpeg') {
            return cb(new Error('Only images are allowed'));
        }
        cb(null, true);
    }
});

router.post('/createBanner', upload.single('image'), async (req, res) => {
    console.log('File:', req.file);
    console.log('Body:', req.body);

    try {
        if (!req.file) {
            console.log('Image not uploaded');
            return res.status(400).json({ status: 'error', message: 'No image uploaded' });
        }

        if (!req.body.type) {
            console.log('Type parameter missing');
            return res.status(400).json({ status: 'error', message: 'Type parameter is required' });
        }

        console.log('Type:', req.body.type, 'Filename:', req.file.filename);

        const banner = await BannerDatabase.createBanner(req.body.type, req.file.filename);

        console.log('Banner created:', banner);

        res.status(201).json({
            status: 'success',
            data: { banner }
        });
    } catch (error) {
        console.error('Error creating banner:', error.message);
        res.status(500).json({
            status: 'error',
            message: error.message,
            stack: error.stack // Only for development debugging
        });
    }
});

// Get Banner by Type Route
router.get('/banner/:type', async (req, res) => {
    try {
        const banner = await BannerDatabase.getBannerByType(req.params.type);
        
        if (!banner) {
            return res.status(404).json({ status: 'error', message: 'Banner not found' });
        }

        res.status(200).json({
            status: 'success',
            banner
        });
    } catch (error) {
        res.status(500).json({
            status: 'error',
            message: error.message
        });
    }
});

// Get All Banners Route
router.get('/getBannerAll', async (req, res) => {
    try {
        const banners = await BannerDatabase.getAllBanners();
        
        res.status(200).json({
            status: 'success',
            banners
        });
    } catch (error) {
        res.status(500).json({
            status: 'error',
            message: error.message
        });
    }
});

// Update Banner Route
router.post('/updateBanner', upload.single('image'), async (req, res) => {
    try {
        if (!req.file) {
            return res.status(400).json({ status: 'error', message: 'No image uploaded' });
        }

        const banner = await BannerDatabase.updateBanner(
            req.body.type, 
            req.file.filename
        );

        if (!banner) {
            return res.status(404).json({ status: 'error', message: 'Banner not found' });
        }

        res.status(200).json({
            status: 'success',
            data: { banner }
        });
    } catch (error) {
        res.status(500).json({
            status: 'error',
            message: error.message
        });
    }
});

// Get Image Binary Route
router.get('/image/:imagePath', (req, res) => {
    const imagePath = path.join(__dirname, 'banners', req.params.imagePath);
    
    res.sendFile(imagePath, (err) => {
        if (err) {
            res.status(404).json({ 
                status: 'error', 
                message: 'Image not found' 
            });
        }
    });
});

router.get('/getAllImagesBinary', async (req, res) => {
    try {
        const imageBinaries = await BannerDatabase.getAllImagesBinary();
        
        res.status(200).json({
            status: 'success',
            count: imageBinaries.length,
            images: imageBinaries.map(img => ({
                id: img.id,
                type: img.type,
                filename: img.filename,
                // Convert binary to base64 for JSON transmission
                base64: img.binary.toString('base64')
            }))
        });
    } catch (error) {
        res.status(500).json({
            status: 'error',
            message: error.message
        });
    }
});

module.exports = router;