const _ = require('lodash');
const Rule = require('../Rule');
const Queue = require('../../General/Queue');
const { encode } = require('../../General/Buffer');
const C = require('../../General/Constant');
const H = require('../../General/Helper');
const Result = require('./Result');

class VideoPoker {
    constructor(io, client, data, id) {
        this.io = io
        this.client = client
        this.data = data
        this.id = _.toNumber(id)
    }

    /*
     * Send Data To Client
    */
    emit(data) {
        this.client.emit(C.PLAY_VIDEOPOKER, encode(data))
    }

    /*
     * Intialize Game
    */
    init() {

        let { command } = this.data;

        switch (command) {
            case 'play':
                return this.play();
            case 'busted':
                return this.busted();
        }
    }

    /*
     * Play Game
    */
    play() {
        let self = this;

        Rule.CanPlay(self.id, self.data, self.client, 'videopoker', (status, err) => {
            if (status !== true) {
                return this.emit({ command: 'error', uid: self.id, message: status, code: err })
            }

            Rule.preparePlay(self.client, self.id, 'videopoker', self.data, (isOk) => {
                if (!isOk) {
                    console.log('Error VideoPoker Playing: 54')
                    return false;
                }

                Rule.checkLimited(self.id, self.data.coin, (status) => {

                    //Generate Random Hash and Result
                    let randomResult = Result.make(status);
                    let result = randomResult.result;
                    let hash = randomResult.hash;

                    //Update Queue and Save Result and Hash
                    Queue.update(self.id, 'uid', 'hash', hash);
                    Queue.update(self.id, 'uid', 'result', result);

                    //Send Play Game
                    this.emit({ command: 'play', hash: hash, amount: self.data.amount });
                })
            })
        })
    }

    /*
     * Deal Cards / Busted
    */
    busted() {
        let self = this;

        let { cards, dealCards, winning, coin } = self.data;

        winning = _.toNumber(winning);

        //Get The Client Game Info
        let info = Queue.getOne('uid', self.id)

        if (!info) {
            console.log('Client Not Playing!', self.id)
            return false;
        }

        let hash = info.hash;
        let amount = info.data.amount;

        let profit = 0.00000000;
        let isWinner = false;

        if (winning !== 0) {
            isWinner = true;
            // profit = winning / 2;
            profit = amount / 4;
        }
        else{
            profit = -amount;
        }

        profit = H.CryptoSet(profit, coin);

        let initialHand = cards;
        let finalHand = dealCards;

        let result = { initialHand, finalHand }

        Rule.prepareBusted(self.client, self.io, self.id, 'videopoker', info.data, profit, result, hash, isWinner, (res) => {
            if (res) {
                self.emit({ command: 'busted',profit:profit })
            }
        })
    }
}

module.exports = VideoPoker;