const db = require('../General/Model');

// Get all unique teams with their members
exports.getAllTeams = async (req, res) => {
  try {
    // Fetch unique teams (most recent record per ownername)
    const teamsResult = await db.query(`
      SELECT t.ownername::text, t."referalCode"::text, t."createdAt", t."updatedAt",
             u.id as owner_id, u.name::text as owner_name, u.email::text as owner_email, 
             u.avatar::text as owner_avatar, u.level as owner_level
      FROM team t
      LEFT JOIN users u ON t.ownername = u.name
      WHERE t."createdAt" = (
        SELECT MAX("createdAt")
        FROM team t2
        WHERE t2.ownername = t.ownername
      )
      ORDER BY t."createdAt" DESC
    `);

    // Fetch all members for each team
    const teams = teamsResult.rows;
    const teamsWithMembers = await Promise.all(teams.map(async (team) => {
      const membersResult = await db.query(`
        SELECT u.id, u.name::text, u.email::text, u.avatar::text, 
               u.level, u.balance, u.games_played
        FROM users u
        WHERE u.name IN (
          SELECT membername
          FROM team t
          WHERE t.ownername = $1
        )
        ORDER BY u.level DESC, u.name ASC
      `, [team.ownername]);

      return {
        ...team,
        members: membersResult.rows
      };
    }));

    res.status(200).json({
      success: true,
      data: teamsWithMembers
    });
  } catch (error) {
    console.error('Error fetching teams:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch teams',
      error: error.message
    });
  }
};

// Get team members by team ownername
exports.getTeamMembers = async (req, res) => {
  try {
    const { teamId } = req.params; // teamId is ownername

    // Get team details
    const teamResult = await db.query(`
      SELECT "referalCode"::text, ownername::text
      FROM team
      WHERE ownername = $1
      LIMIT 1
    `, [teamId]);

    if (teamResult.rows.length === 0) {
      return res.status(404).json({
        success: false,
        message: 'Team not found'
      });
    }

    const { referalCode } = teamResult.rows[0];

    // Get all users who have this referalCode as their refree
    const membersResult = await db.query(`
      SELECT id, name::text, email::text, avatar::text, level, balance, games_played
      FROM users 
      WHERE refree = $1
      ORDER BY level DESC, name ASC
    `, [referalCode]);

    res.status(200).json({
      success: true,
      data: membersResult.rows
    });
  } catch (error) {
    console.error('Error fetching team members:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch team members',
      error: error.message
    });
  }
};

// Get all users with their teams in a tree structure
exports.getUsersWithTeams = async (req, res) => {
   try {
    const usersResult = await db.query(`
  SELECT DISTINCT u.id, u.name::text, u.email::text, u.avatar::text, 
         u.level, u.balance::text, u.games_played, u.refree::text, u.referalcode::text, 
         t.ownername::text as team_name
  FROM users u
  LEFT JOIN team t ON u.name::text = t.ownername::text
  ORDER BY u.level DESC, u.name::text ASC
`);

    const users = usersResult.rows;
    const userMap = new Map();
    const referralMap = {};

    // Build user map and referral map
    users.forEach(user => {
      userMap.set(user.id, {
        ...user,
        children: []
      });
      if (user.referalcode) {
        referralMap.set(user.referalcode, user.id);
      }
    });

    // Build hierarchy
    const rootUsers = [];
    users.forEach(user => {
      const userWithChildren = userMap.get(user.id);
      if (user.refree && referralMap.has(user.refree)) {
        const parentId = referralMap.get(user.refree);
        const parent = userMap.get(parentId);
        if (parent) {
          parent.children.push(userWithChildren);
        } else {
          rootUsers.push(userWithChildren);
        }
      } else {
        rootUsers.push(userWithChildren);
      }
    });

    res.status(200).json({
      status: 'success',
      data: rootUsers
    });
  } catch (error) {
    console.error('Error fetching users with teams:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch team data',
      error: error.message
    });
  }
};