const db = require('../General/Model');
class AffiliateUnlockController {
     // Calculate unlockable amount based on VIP level wager requirements
  calculateUnlockableAmount(wagerAmount) {
    const vipLevels = [
      { wager: 1000, reward: 0.50 },
      { wager: 5000, reward: 2.50 },
      { wager: 17000, reward: 5.00 },
      { wager: 49000, reward: 12.00 },
      { wager: 129000, reward: 25.00 },
      { wager: 321000, reward: 50.00 },
      { wager: 769000, reward: 80.00 },
      { wager: 1793000, reward: 120.00 },
      { wager: 4097000, reward: 205.00 },
      { wager: 9217000, reward: 500.00 }
    ];

    // Find the highest eligible reward tier
    for (let i = vipLevels.length - 1; i >= 0; i--) {
      if (wagerAmount >= vipLevels[i].wager) {
        return vipLevels[i].reward;
      }
    }
    return 0;
  }

  async processWagerAndUnlock(uid, membername, wagerAmount) {
    try {
        console.log('Starting wager processing:', { uid, membername, wagerAmount });

        // First get member's UID from users table
        const memberUidQuery = `
            SELECT id as uid 
            FROM users 
            WHERE name = $1
        `;
        const memberResult1 = await db.query(memberUidQuery, [membername]);
        if (!memberResult1.rows.length) {
            console.log('Member not found:', membername);
            return { success: false, message: 'Member not found.' };
        }
        const memberUid = memberResult1.rows[0].uid;
        console.log('Found member UID:', memberUid);

        // Get member's current wager
        const currentWagerQuery = `
            SELECT wager FROM userwager WHERE uid = $1
        `;
        const currentWagerResult = await db.query(currentWagerQuery, [memberUid]);
        
        if (!currentWagerResult.rows.length) {
            console.log('No existing wager entry found, processing new wager');
            return { success: false, message: 'No existing wager entry found.' };
        }

        const currentWager = currentWagerResult.rows[0].wager || 0;
        console.log('Current member wager:', currentWager);

        if (wagerAmount <= currentWager) {
            console.log('New wager not higher than current wager');
            return { success: false, message: 'New wager must be higher than the current wager.' };
        }

        const unlockableAmount = this.calculateUnlockableAmount(wagerAmount);
        console.log('Calculated unlockable amount:', unlockableAmount);

        if (unlockableAmount <= 0) {
            console.log('No unlockable amount available');
            return { success: false, message: 'No unlockable amount available.' };
        }

        // Start transaction
        await db.query('BEGIN');

        // Check for existing unlocks at this wager level
        const existingUnlockQuery = `
            SELECT id, wager_amount 
            FROM unlocked_rewards 
            WHERE membername = $1 AND wager_amount >= $2
            ORDER BY wager_amount DESC
            LIMIT 1
        `;
        const existingUnlock = await db.query(existingUnlockQuery, [membername, wagerAmount]);
        
        if (existingUnlock.rows.length > 0) {
            console.log('Already unlocked at this wager level:', existingUnlock.rows[0]);
            await db.query('ROLLBACK');
            return { success: false, message: 'Wager level already unlocked.' };
        }

        // Get referral info for the member
        const memberQuery = `
            SELECT 
                r.ownername, 
                r."referalCode", 
                r.amount as total_reward,
                r.type,
                r.locked
            FROM rewards r
            WHERE r.membername = $1 
                AND r.type = 'referral'
                AND r.locked = true
            LIMIT 1
        `;
        const memberResult = await db.query(memberQuery, [membername]);
        console.log('Member referral query result:', memberResult.rows);

        if (memberResult.rows.length === 0) {
            console.log('No locked referral record found for member:', membername);
            await db.query('ROLLBACK');
            return { success: false, message: 'No locked referral record found.' };
        }

        const { ownername, referalCode, total_reward } = memberResult.rows[0];
        console.log('Referral details:', { ownername, referalCode, total_reward });

        if (total_reward < unlockableAmount) {
            console.log('Insufficient reward balance:', { total_reward, unlockableAmount });
            await db.query('ROLLBACK');
            return { success: false, message: 'Insufficient reward balance.' };
        }

        // Update rewards table - deduct the unlocked amount
        const updateRewardsQuery = `
            UPDATE rewards 
            SET amount = amount - $1
            WHERE membername = $2 
                AND type = 'referral'
                AND locked = true
            RETURNING *
        `;
        const updateResult = await db.query(updateRewardsQuery, [unlockableAmount, membername]);
        console.log('Updated rewards:', updateResult.rows);

        // Create unlocked reward entry
        const createUnlockedQuery = `
            INSERT INTO unlocked_rewards (
                uid, 
                ownername, 
                membername, 
                "referalCode", 
                amount, 
                wager_amount, 
                "createdAt",
                claimed
            )
            VALUES ($1, $2, $3, $4, $5, $6, NOW(), false)
            RETURNING *
        `;
        const unlockedResult = await db.query(createUnlockedQuery, [
            uid,
            ownername,
            membername,
            referalCode,
            unlockableAmount,
            wagerAmount
        ]);
        console.log('Created unlock entry:', unlockedResult.rows[0]);

        // // Update user wager
        // const updateWagerQuery = `
        //     INSERT INTO userwager (uid, wager)
        //     VALUES ($1, $2)
        //     ON CONFLICT (uid) 
        //     DO UPDATE SET wager = EXCLUDED.wager
        //     RETURNING *
        // `;
        // await db.query(updateWagerQuery, [uid, wagerAmount]);

        await db.query('COMMIT');

        return {
            success: true,
            message: 'Wager processed and unlock created successfully.',
            unlockedReward: unlockedResult.rows[0],
            newWager: wagerAmount,
            previousWager: currentWager,
            unlockedAmount: unlockableAmount
        };

    } catch (error) {
        await db.query('ROLLBACK');
        console.error('Error processing wager unlock:', error);
        return { success: false, message: 'An error occurred during processing.', error: error.message };
    }
}


    // Get unclaimed rewards for a user
    async getUnclaimedRewards(uid) {
      try {
        const query = `
          SELECT 
            id,
            ownername,
            membername,
            amount,
            cointype,
            wager_amount,
            "createdAt"
          FROM unlocked_rewards
          WHERE uid = $1 AND claimed = false
          ORDER BY "createdAt" DESC
        `;
        const result = await db.query(query, [uid]);
        
        return {
          unclaimedRewards: result.rows,
          totalUnclaimed: result.rows.reduce((sum, reward) => sum + Number(reward.amount), 0)
        };
      } catch (error) {
        console.error('Error fetching unclaimed rewards:', error);
        throw error;
      }
    }
  
    async claimReward(uid, rewardId) {
      try {
        await db.query('BEGIN');
  
        // Get reward details
      const rewardQuery = `
    SELECT 
        id,
        uid,
        ownername,
        membername,
        referalCode,
        amount,
        cointype,
        wager_amount,
        createdAt,
        claimed
    FROM unlocked_rewards
    WHERE id = $1 AND uid = $2 AND claimed = false
`;
        const rewardResult = await db.query(rewardQuery, [rewardId, uid]);
  
        if (rewardResult.rows.length === 0) {
          await db.query('ROLLBACK');
          throw new Error('Reward not found or already claimed');
        }
  
        const reward = rewardResult.rows[0];
  
        // Update reward status to claimed
        const updateRewardQuery = `
          UPDATE unlocked_rewards
          SET claimed = true
          WHERE id = $1
        `;
        await db.query(updateRewardQuery, [rewardId]);
  
        // Add amount to user's bjb balance in credits table
        const updateCreditsQuery = `
          INSERT INTO credits (uid, bjb)
          VALUES ($1, $2)
          ON CONFLICT (uid) 
          DO UPDATE SET bjb = credits.bjb + $2
          WHERE credits.uid = $1
        `;
        await db.query(updateCreditsQuery, [uid, reward.amount]);
  
        await db.query('COMMIT');
        return {
          success: true,
          message: 'Reward claimed successfully',
          claimedAmount: reward.amount,
          rewardId: reward.id
        };
  
      } catch (error) {
        await db.query('ROLLBACK');
        console.error('Error claiming reward:', error);
        throw error;
      }
    }
    async claimAllRewards(uid) {
        try {
            console.log('Starting reward claiming for user:', uid);
    
            await db.query('BEGIN');
    
            // Fetch all claimable rewards for the user
          const fetchRewardsQuery = `
    SELECT 
        id,
        uid,
        ownername,
        membername,
        referalCode,
        amount,
        cointype,
        wager_amount,
        createdAt,
        claimed
    FROM unlocked_rewards
    WHERE uid = $1 AND claimed = false
`;
            const rewardResults = await db.query(fetchRewardsQuery, [uid]);
    
            if (rewardResults.rows.length === 0) {
                console.log('No claimable rewards found for user:', uid);
                await db.query('ROLLBACK');
                return {
                    success: false,
                    message: 'No claimable rewards found for this user.'
                };
            }
    
            const claimableRewards = rewardResults.rows;
            let totalClaimedAmount = 0;
    
            for (const reward of claimableRewards) {
                const { id, amount } = reward;
    
                // Mark the reward as claimed
                const updateRewardQuery = `
                    UPDATE unlocked_rewards
                    SET claimed = true
                    WHERE id = $1
                `;
                await db.query(updateRewardQuery, [id]);
    
                // Update user's bjb balance in credits table
                const updateCreditsQuery = `
                    UPDATE credits
                    SET bjb = bjb + $1
                    WHERE uid = $2
                `;
                await db.query(updateCreditsQuery, [amount, uid]);
    
                totalClaimedAmount += amount;
                console.log(`Successfully claimed reward ${id}: Amount ${amount}`);
            }
    
            await db.query('COMMIT');
    
            return {
                success: true,
                message: 'All claimable rewards have been successfully claimed.',
                totalClaimedAmount,
                claimedRewards: claimableRewards.map(r => ({
                    id: r.id,
                    amount: r.amount,
                    ownername: r.ownername,
                    membername: r.membername
                }))
            };
    
        } catch (error) {
            await db.query('ROLLBACK');
            console.error('Error claiming all rewards:', error);
            return {
                success: false,
                message: 'Error claiming rewards.',
                error: error.message
            };
        }
    }
    
    
  
    // Get user's total unlocked and claimed rewards
    async getUserRewardsSummary(uid) {
      try {
        const query = `
          SELECT 
            SUM(CASE WHEN claimed = false THEN amount ELSE 0 END) as total_unclaimed,
            SUM(CASE WHEN claimed = true THEN amount ELSE 0 END) as total_claimed,
            COUNT(CASE WHEN claimed = false THEN 1 END) as unclaimed_count,
            COUNT(CASE WHEN claimed = true THEN 1 END) as claimed_count
          FROM unlocked_rewards
          WHERE uid = $1
        `;
        const result = await db.query(query, [uid]);
        return result.rows[0];
      } catch (error) {
        console.error('Error getting rewards summary:', error);
        throw error;
      }
    }
  }
  
  module.exports = new AffiliateUnlockController();