const { getVipLevelDetails } = require('./calculateVip');
const pg = require('../General/Model');

const calculateWagerBonus = async (userid, period, startDate, endDate) => {
  try {
    console.log(`Calculating ${period} bonus for user ${userid} from ${startDate.toISOString()} to ${endDate.toISOString()}`);
    
    // Fetch wager history
    const wagerHistoryQuery = `
      SELECT previous_wager, new_wager, updated_at
      FROM userwager_history 
      WHERE uid = $1 
      AND updated_at >= $2 
      AND updated_at < $3
      ORDER BY updated_at ASC
    `;
    const wagerHistoryResult = await pg.query(wagerHistoryQuery, [userid, startDate, endDate]);
    
    console.log(`Found ${wagerHistoryResult.rows.length} wager history records for user ${userid}`);
    
    if (wagerHistoryResult.rows.length === 0) {
      console.log(`No wager history found for user ${userid} in ${period} period (${startDate.toISOString()} to ${endDate.toISOString()})`);
      return { eligible: false, bonusAmount: 0, wagerChange: 0, reason: 'No wager activity in period' };
    }
    
    // Calculate total wager change
    let totalWagerChange = 0;
    let validRecords = 0;
    let lastWager = null;
    
    wagerHistoryResult.rows.forEach((row, index) => {
      try {
        console.log(`Parsing wager record ${index + 1}: previous_wager=${row.previous_wager}, new_wager=${row.new_wager}`);
        const previousWager = parseWagerValue(row.previous_wager);
        const newWager = parseWagerValue(row.new_wager);
        
        console.log(`Parsed values: previousWager=${previousWager}, newWager=${newWager}`);
        
        if (lastWager !== null && previousWager !== lastWager) {
          console.warn(`Wager history discontinuity for user ${userid} at record ${index + 1}: expected previous_wager=${lastWager}, got ${previousWager}`);
        }
        
        const wagerDiff = newWager - previousWager;
        if (wagerDiff > 0) {
          totalWagerChange += wagerDiff;
          validRecords++;
        }
        
        lastWager = newWager;
        
        console.log(`Record ${index + 1}: ${previousWager} -> ${newWager} (diff: ${wagerDiff})`);
      } catch (parseError) {
        console.error(`Error parsing wager values for user ${userid} record ${index}:`, parseError.message, parseError.stack);
      }
    });
    
    console.log(`Total positive wager change for user ${userid}: ${totalWagerChange} (from ${validRecords} valid records)`);
    
    if (totalWagerChange <= 0) {
      console.log(`User ${userid} has no positive wager change in ${period} period`);
      return { eligible: false, bonusAmount: 0, wagerChange: totalWagerChange, reason: 'No positive wager change' };
    }
    
    // Get current wager
    const wagerQuery = `SELECT wager FROM userwager WHERE uid = $1`;
    const wagerResult = await pg.query(wagerQuery, [userid]);
    
    if (!wagerResult.rows.length) {
      console.log(`No current wager found for user ${userid}`);
      return { eligible: false, bonusAmount: 0, wagerChange: totalWagerChange, reason: 'No current wager data' };
    }
    
    const currentWager = parseWagerValue(wagerResult.rows[0].wager);
    console.log(`Current wager for user ${userid}: ${currentWager}`);
    
    // Get VIP level
    let vipDetails;
    let vipLevelNum;
    try {
      vipDetails = getVipLevelDetails(currentWager);
      if (vipDetails.error) {
        console.error(`Error getting VIP level for user ${userid}: ${vipDetails.error}`);
        return { 
          eligible: false, 
          bonusAmount: 0, 
          wagerChange: totalWagerChange, 
          reason: `Invalid VIP level: ${vipDetails.error}` 
        };
      }
      vipLevelNum = vipDetails.level || extractVipLevelNumber(vipDetails);
      console.log(`User ${userid} VIP Level: ${vipLevelNum} (Details: ${JSON.stringify(vipDetails)})`);
    } catch (vipError) {
      console.error(`Error getting VIP level for user ${userid}: ${vipError.message}`);
      return { 
        eligible: false, 
        bonusAmount: 0, 
        wagerChange: totalWagerChange, 
        reason: `VIP level calculation failed: ${vipError.message}` 
      };
    }
    
    // VIP level requirements
    const minVipLevel = {
      daily: 20,
      weekly: 25,
      monthly: 30
    }[period];
    
    if (vipLevelNum < minVipLevel) {
      console.log(`User ${userid} VIP level ${vipLevelNum} is below required ${minVipLevel} for ${period} bonus`);
      return { 
        eligible: false, 
        bonusAmount: 0, 
        wagerChange: totalWagerChange, 
        reason: `Insufficient VIP level (${vipLevelNum}/${minVipLevel})` 
      };
    }
    
    // Calculate bonus
    const baseBonus = totalWagerChange * 0.01;
    const periodMultiplier = {
      daily: 0.50,
      weekly: 0.15,
      monthly: 0.20
    }[period] || 0;
    
    const vipMultiplier = vipDetails.bonusMultiplier || 1;
    const finalBonus = Math.floor(baseBonus * periodMultiplier * vipMultiplier * 100) / 100;
    
    console.log(`Bonus calculation for user ${userid}:`);
    console.log(`- Base bonus (1% of wager change): ${baseBonus}`);
    console.log(`- Period multiplier (${period}): ${periodMultiplier}`);
    console.log(`- VIP multiplier: ${vipMultiplier}`);
    console.log(`- Final bonus: ${finalBonus}`);
    
    const minBonusAmount = 0.1;
    if (finalBonus < minBonusAmount) {
      console.log(`Calculated bonus ${finalBonus} is below minimum threshold ${minBonusAmount}`);
      return { 
        eligible: false, 
        bonusAmount: 0, 
        wagerChange: totalWagerChange, 
        reason: `Bonus amount too low (${finalBonus})` 
      };
    }
    
    console.log(`✓ User ${userid} eligible for ${period} bonus: ${finalBonus}`);
    return { 
      eligible: true, 
      bonusAmount: finalBonus, 
      wagerChange: totalWagerChange,
      vipLevel: vipLevelNum,
      vipMultiplier: vipMultiplier
    };
    
  } catch (error) {
    console.error(`Error calculating ${period} bonus for user ${userid}:`, error.message, error.stack);
    throw error;
  }
};

function parseWagerValue(wagerValue) {
  if (typeof wagerValue === 'number' && !isNaN(wagerValue) && wagerValue >= 0) {
    console.log(`Parsed number wagerValue: ${wagerValue}`);
    return wagerValue;
  }
  
  if (typeof wagerValue === 'string') {
    const cleaned = wagerValue.replace(/,/g, '').trim();
    const parsed = parseFloat(cleaned);
    if (isNaN(parsed) || parsed < 0) {
      console.warn(`Invalid wager value (string): "${wagerValue}" parsed as 0`);
      return 0;
    }
    console.log(`Parsed string wagerValue: ${wagerValue} -> ${parsed}`);
    return parsed;
  }
  
  console.warn(`Invalid wager value (type ${typeof wagerValue}): ${wagerValue} parsed as 0`);
  return 0;
}

function extractVipLevelNumber(vipDetails) {
  try {
    if (!vipDetails || !vipDetails.previousVipLevel) {
      console.warn('No VIP details provided:', vipDetails);
      return 0;
    }
    
    const vipLevel = vipDetails.previousVipLevel.toString();
    const match = vipLevel.match(/(\d+)/);
    const level = match ? parseInt(match[1]) : 0;
    
    if (level === 0) {
      console.warn(`Failed to extract VIP level from: ${vipLevel}`);
    }
    
    return level;
  } catch (error) {
    console.error('Error extracting VIP level number:', error.message);
    return 0;
  }
}

const calculateWagerBonusForUsers = async (userIds, period, startDate, endDate) => {
  const results = [];
  const batchSize = 10;
  let errorCount = 0;
  
  for (let i = 0; i < userIds.length; i += batchSize) {
    const batch = userIds.slice(i, i + batchSize);
    const batchPromises = batch.map(userid => 
      calculateWagerBonus(userid, period, startDate, endDate)
        .catch(error => {
          errorCount++;
          console.error(`Error processing bonus for user ${userid} in batch ${i/batchSize + 1}:`, error.message);
          return {
            userid,
            eligible: false,
            bonusAmount: 0,
            wagerChange: 0,
            error: error.message
          };
        })
    );
    
    const batchResults = await Promise.all(batchPromises);
    results.push(...batchResults.map((result, index) => ({
      userid: batch[index],
      ...result
    })));
    
    if (i + batchSize < userIds.length) {
      await new Promise(resolve => setTimeout(resolve, 100));
    }
  }
  
  if (errorCount > 0) {
    console.warn(`Completed batch bonus calculation with ${errorCount} errors for ${userIds.length} users`);
  } else {
    console.log(`Successfully calculated bonuses for ${userIds.length} users`);
  }
  
  return results;
};

module.exports = { 
  calculateWagerBonus,
  calculateWagerBonusForUsers
};