const express = require('express');
const pg = require('../General/Model');
const { getVipLevelDetails } = require('./calculateVip');
const {debugBonusCalculation} = require('./bonusCron');

const router = express.Router();

// Middleware to get user ID from query parameters
const getUserId = (req) => {
  const userid = req.query.userid;
  if (!userid) {
    throw new Error('User ID is required in query parameters');
  }
  return userid;
};

// Get available bonuses
// router.get('/api/bonuses', async (req, res) => {
//   try {
//     const userid = getUserId(req);

//     // Fetch current wager for VIP level
//     const wagerQuery = `SELECT wager FROM userwager WHERE uid = $1`;
//     const wagerResult = await pg.query(wagerQuery, [userid]);
//     const currentWager = wagerResult.rows[0]?.wager
//       ? parseFloat(wagerResult.rows[0].wager.replace(/,/g, ''))
//       : 0;
//     const vipDetails = getVipLevelDetails(currentWager);
//     const vipLevelNum = parseInt(vipDetails.previousVipLevel?.split(' ')[1]) || 0;

//     // Fetch available bonuses
//     const bonusesQuery = `
//       SELECT dailybonus, weeklybonus, monthlybonus
//       FROM userbonus
//       WHERE userid = $1
//     `;
//     const bonusesResult = await pg.query(bonusesQuery, [userid]);
//     const bonuses = bonusesResult.rows[0] || {
//       dailybonus: 0,
//       weeklybonus: 0,
//       monthlybonus: 0
//     };

//     // Fetch claimable bonus history
//     const historyQuery = `
//       SELECT bonus_type, bonus_amount, is_claimed, claim_deadline, is_unclaimable
//       FROM bonus_history
//       WHERE userid = $1 AND is_unclaimable = FALSE AND is_claimed = FALSE
//       ORDER BY created_at DESC
//     `;
//     const historyResult = await pg.query(historyQuery, [userid]);

//     res.json({
//       vipLevel: vipDetails.previousVipLevel,
//       bonuses: {
//         daily: {
//           amount: bonuses.dailybonus,
//           eligible: vipLevelNum >= 20,
//           claimable: historyResult.rows.find(h => h.bonus_type === 'daily' && new Date() <= new Date(h.claim_deadline))
//         },
//         weekly: {
//           amount: bonuses.weeklybonus,
//           eligible: vipLevelNum >= 25,
//           claimable: historyResult.rows.find(h => h.bonus_type === 'weekly' && new Date() <= new Date(h.claim_deadline))
//         },
//         monthly: {
//           amount: bonuses.monthlybonus,
//           eligible: vipLevelNum >= 30,
//           claimable: historyResult.rows.find(h => h.bonus_type === 'monthly' && new Date() <= new Date(h.claim_deadline))
//         }
//       },
//       history: historyResult.rows
//     });
//   } catch (error) {
//     console.error('Error fetching bonuses:', error);
//     res.status(error.message === 'User ID is required in query parameters' ? 400 : 500).json({
//       error: error.message || 'Internal server error'
//     });
//   }
// });
// FIXED: API Endpoint Logic
router.get('/api/bonuses', async (req, res) => {
  try {
    const userid = getUserId(req);

    /* ── current wager → VIP level ─────────────────────────────────── */
    const { rows: [wRow] } = await pg.query(
      'SELECT wager FROM userwager WHERE uid = $1', [userid]);
    const currentWager = wRow?.wager
      ? parseFloat(wRow.wager.replace(/,/g, '')) : 0;
    const vipDetails   = getVipLevelDetails(currentWager);
    const vipLevelNum  = parseInt(vipDetails.previousVipLevel?.split(' ')[1]) || 0;

    /* ── latest amounts from userbonus ─────────────────────────────── */
    const { rows: [bRow] } = await pg.query(
      'SELECT dailybonus, weeklybonus, monthlybonus FROM userbonus WHERE userid = $1',
      [userid]) || { dailybonus: 0, weeklybonus: 0, monthlybonus: 0 };

    /* ── active (still claim-able) history rows ────────────────────── */
    const { rows: history } = await pg.query(`
      SELECT bonus_type, bonus_amount, is_claimed,
             claim_deadline, is_unclaimable, created_at
        FROM bonus_history
       WHERE userid            = $1
         AND is_unclaimable     = FALSE
         AND is_claimed         = FALSE
         AND claim_deadline    >= NOW()      -- inclusive
       ORDER BY created_at DESC
    `, [userid]);

    const isClaimable = type => history.some(h => h.bonus_type === type);

    res.json({
      vipLevel : vipDetails.previousVipLevel,
      bonuses  : {
        daily  : {
          amount    : bRow.dailybonus   || 0,
          eligible  : vipLevelNum >= 20,
          claimable : isClaimable('daily')
        },
        weekly : {
          amount    : bRow.weeklybonus  || 0,
          eligible  : vipLevelNum >= 25,
          claimable : isClaimable('weekly')
        },
        monthly: {
          amount    : bRow.monthlybonus || 0,
          eligible  : vipLevelNum >= 30,
          claimable : isClaimable('monthly')
        }
      },
      history,
      debug: {
        currentTime : new Date().toISOString(),
        vipLevelNum,
        historyCount: history.length
      }
    });
  } catch (e) {
    console.error('Error fetching bonuses:', e);
    res.status(500).json({ error: e.message || 'Internal server error' });
  }
});
router.get('/api/bonus-history', async (req, res) => {
  try {
    const userid = getUserId(req);
    
    // Fetch bonus history
    const historyQuery = `
      SELECT id, bonus_type, wager_change, bonus_amount, is_claimed, 
             claim_deadline, created_at, claimed_at, is_unclaimable
      FROM bonus_history
      WHERE userid = $1
      ORDER BY created_at DESC
    `;
    
    const historyResult = await pg.query(historyQuery, [userid]);
    
    res.json({
      success: true,
      userId: userid,
      history: historyResult.rows
    });
  } catch (error) {
    console.error('Error fetching bonus history:', error);
    res.status(error.message === 'User ID is required in query parameters' ? 400 : 500).json({
      success: false,
      error: error.message || 'Internal server error'
    });
  }
});
// Claim a bonus
router.post('/api/bonuses/claim/:type', async (req, res) => {
  const client = await pg.connect();
  try {
    const userid = getUserId(req);
    const bonusType = req.params.type;

    if (!['daily', 'weekly', 'monthly'].includes(bonusType)) {
      return res.status(400).json({ error: 'Invalid bonus type' });
    }

    await client.query('BEGIN');

    // 1. Check if bonus is claimable
    const bonusQuery = `
      SELECT id, bonus_amount, claim_deadline, is_claimed, is_unclaimable
      FROM bonus_history
      WHERE userid = $1 AND bonus_type = $2 AND is_claimed = FALSE AND is_unclaimable = FALSE
      ORDER BY created_at DESC
      LIMIT 1
    `;
    const bonusResult = await client.query(bonusQuery, [userid, bonusType]);
    const bonus = bonusResult.rows[0];

    if (!bonus || new Date() > new Date(bonus.claim_deadline)) {
      await client.query('ROLLBACK');
      return res.status(400).json({ error: 'No claimable bonus found' });
    }

    // 2. Update bonus_history
    const updateHistoryQuery = `
      UPDATE bonus_history
      SET is_claimed = TRUE, claimed_at = NOW()
      WHERE id = $1
    `;
    await client.query(updateHistoryQuery, [bonus.id]);

    // 3. Reset bonus amount
    const resetBonusQuery = `
      UPDATE userbonus
      SET ${bonusType}bonus = 0
      WHERE userid = $1
    `;
    await client.query(resetBonusQuery, [userid]);

    // 4. Add to actual bonus
    const addBonusQuery = `
      UPDATE userbonus
      SET actual${bonusType}bonus = actual${bonusType}bonus + $1
      WHERE userid = $2
    `;
    await client.query(addBonusQuery, [bonus.bonus_amount, userid]);

    // 5. Add to credits balance
    const addBalanceQuery = `
      UPDATE credits
      SET bjb = COALESCE(bjb, 0) + $1
      WHERE uid = $2
    `;
    await client.query(addBalanceQuery, [bonus.bonus_amount, userid]);

    await client.query('COMMIT');
    res.json({ message: `Successfully claimed ${bonusType} bonus of ${bonus.bonus_amount}` });
  } catch (error) {
    await client.query('ROLLBACK');
    console.error(`Error claiming ${req.params.type} bonus:`, error);
    res.status(error.message === 'User ID is required in query parameters' ? 400 : 500).json({
      error: error.message || 'Internal server error'
    });
  } finally {
    client.release();
  }
});

router.get('/debug-bonus/:userId', async (req, res) => {
  const userId = req.params.userId;
  const result = await debugBonusCalculation(userId);
  res.json(result);
});

module.exports = router;