// routes.js
const express = require('express');
const router = express.Router();
const OTPService = require('./otpcontroller');
const EmailService = require('./controller');
const db = require('../General/Model');
// OTP Routes
router.post('/otp/send', async (req, res) => {
    try {
        const { email, purpose } = req.body;
        
        if (!['register', 'login', 'reset-password'].includes(purpose)) {
            throw new Error('Invalid OTP purpose');
        }

        const otp = await OTPService.generateAndSaveOTP(email, purpose);
        await EmailService.sendOTPEmail(email, otp, purpose);

        res.json({
            success: true,
            message: 'OTP sent successfully'
        });
    } catch (error) {
        res.status(200).json({
            success: false,
            error: error.message
        });
    }
});

router.post('/otp/resend', async (req, res) => {
    try {
        const { email, purpose } = req.body;

        // Check if we can resend OTP (2-minute cooldown)
        const canResend = await OTPService.canResendOTP(email, purpose);
        if (!canResend) {
            throw new Error('Please wait 2 minutes before requesting a new OTP');
        }

        const otp = await OTPService.generateAndSaveOTP(email, purpose);
        await EmailService.sendOTPEmail(email, otp, purpose);

        res.json({
            success: true,
            message: 'New OTP sent successfully'
        });
    } catch (error) {
        res.status(200).json({
            success: false,
            error: error.message
        });
    }
});

router.post('/otp/verify', async (req, res) => {
    try {
        const { email, otp, purpose } = req.body;
        await OTPService.verifyOTP(email, otp, purpose);
        res.json({
            success: true,
            message: 'OTP verified successfully'
        });
    } catch (error) {
        res.status(200).json({
            success: false,
            error: error.message
        });
    }
});

// General Email Routes
router.post('/email/send', async (req, res) => {
    try {
        const { to, subject, title, content, ctaLink, ctaText } = req.body;
        await EmailService.sendGeneralEmail(to, subject, {
            title,
            content,
            ctaLink,
            ctaText
        });
        res.json({
            success: true,
            message: 'Email sent successfully'
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            error: error.message
        });
    }
});

router.post('/email/bulk', async (req, res) => {
    try {
        const { emails, subject, title, content, ctaLink, ctaText } = req.body;
        const results = await EmailService.sendBulkEmails(
            emails,
            subject,
            {
                title,
                content,
                ctaLink,
                ctaText
            }
        );
        res.json({
            success: true,
            data: results
        });
    } catch (error) {
        res.status(500).json({
            success: false,
            error: error.message
        });
    }
});


// 2FA Reset Routes
router.post('/2fa/reset', async (req, res) => {
    try {
        const { identifier } = req.body;

        if (!identifier) {
            throw new Error('Email or username is required');
        }

        // Find user by email or name
        const userResult = await db.query(
            'SELECT id, email FROM users WHERE email = $1 OR name = $1',
            [identifier]
        );

        if (!userResult.rows.length) {
            throw new Error('User not found');
        }

        const { id: uid, email } = userResult.rows[0];

        // Check if 2FA is enabled
        const twoFAResult = await db.query(
            'SELECT is_enabled FROM user_2fa WHERE uid = $1',
            [uid]
        );

        if (!twoFAResult.rows.length || !twoFAResult.rows[0].is_enabled) {
            throw new Error('2FA is not enabled');
        }

        // Generate and send OTP
        const otp = await OTPService.generateAndSaveOTP(email, 'reset-2fa');
        await EmailService.sendOTPEmail(email, otp, 'reset-2fa');

        res.json({
            success: true,
            message: 'OTP sent successfully for 2FA reset'
        });
    } catch (error) {
        res.status(200).json({
            success: false,
            error: error.message
        });
    }
});

router.post('/2fa/reset-verify', async (req, res) => {
    const client = await db.connect();
    try {
        const { identifier, otp } = req.body;

        if (!identifier || !otp) {
            throw new Error('Email or username and OTP are required');
        }

        await client.query('BEGIN');

        // Fetch user
        const userResult = await client.query(
            'SELECT id, email FROM users WHERE email = $1 OR name = $1',
            [identifier]
        );
        if (!userResult.rows.length) {
            throw new Error('User not found');
        }

        const { id: uid, email } = userResult.rows[0];

        // Verify OTP
        await OTPService.verifyOTP(email, otp, 'reset-2fa');

        // Disable 2FA
        await client.query(
            'UPDATE user_2fa SET is_enabled = false, secret_key = null WHERE uid = $1',
            [uid]
        );

        await client.query('COMMIT');
        res.json({ success: true, message: '2FA has been reset successfully' });
    } catch (error) {
        await client.query('ROLLBACK');
        res.status(200).json({ success: false, error: error.message });
    } finally {
        client.release();
    }
});

module.exports = router;