const pg = require('../General/Model');// Adjust path as needed to your database connection

// Get all fiat withdrawal data
exports.getFiatWithdrawData = async (req, res) => {
  try {
    const query = `
      SELECT fw.*, u.name 
      FROM fiat_withdrawals fw
      LEFT JOIN users u ON fw.uid = u.id
      ORDER BY fw.date DESC
    `;
    
    const result = await pg.query(query);
    res.json(result.rows);
  } catch (error) {
    console.error('Error fetching fiat withdrawal data:', error);
    res.status(500).json({ error: 'Failed to fetch fiat withdrawal data' });
  }
};

// Create a new fiat withdrawal request
// exports.createFiatWithdrawal = async (req, res) => {
//   const { 
//     uid, 
//     amount, 
//     currency, 
//     bank_name, 
//     account_number, 
//     account_holder_name, 
//     ifsc_code, 
//     upi_id 
//   } = req.body;

//   // Validation
//   if (!uid || !amount || !currency || !account_holder_name) {
//     return res.status(400).json({ error: 'Missing required fields' });
//   }

//   if (currency === 'INR' && !ifsc_code && !upi_id) {
//     return res.status(400).json({ error: 'For INR withdrawals, either IFSC code or UPI ID is required' });
//   }

//   if (currency !== 'INR' && (!bank_name || !account_number || !ifsc_code)) {
//     return res.status(400).json({ error: 'Bank details are required for non-INR withdrawals' });
//   }

//   try {
//     const query = `
//       INSERT INTO fiat_withdrawals 
//         (uid, amount, currency, bank_name, account_number, account_holder_name, ifsc_code, upi_id, status)
//       VALUES 
//         ($1, $2, $3, $4, $5, $6, $7, $8, 'In Queue')
//       RETURNING *
//     `;
    
//     const values = [
//       uid, 
//       amount, 
//       currency, 
//       bank_name || null, 
//       account_number || null, 
//       account_holder_name, 
//       ifsc_code || null, 
//       upi_id || null
//     ];
    
//     const result = await pg.query(query, values);
//     res.status(201).json(result.rows[0]);
//   } catch (error) {
//     console.error('Error creating fiat withdrawal:', error);
//     res.status(500).json({ error: 'Failed to create fiat withdrawal request' });
//   }
// };
exports.createFiatWithdrawal = async (req, res) => {
  const { 
    uid, 
    amount, 
    currency, 
    bank_name, 
    account_number, 
    account_holder_name, 
    ifsc_code, 
    upi_id 
  } = req.body;

  // Validation
  if (!uid || !amount || !currency || !account_holder_name) {
    return res.status(400).json({ error: 'Missing required fields' });
  }

  if (currency === 'INR' && !ifsc_code && !upi_id) {
    return res.status(400).json({ error: 'For INR withdrawals, either IFSC code or UPI ID is required' });
  }

  if (currency !== 'INR' && (!bank_name || !account_number || !ifsc_code)) {
    return res.status(400).json({ error: 'Bank details are required for non-INR withdrawals' });
  }

  try {
    // Start a transaction
    await pg.query('BEGIN');
    
    // Check if user has sufficient balance
    const currencyLower = currency.toLowerCase();
    const balanceQuery = `SELECT ${currencyLower} FROM credits WHERE uid = $1`;
    const balanceResult = await pg.query(balanceQuery, [uid]);
    
    if (balanceResult.rows.length === 0) {
      await pg.query('ROLLBACK');
      return res.status(404).json({ error: 'User not found' });
    }
    
    const currentBalance = parseFloat(balanceResult.rows[0][currencyLower] || 0);
    
    if (currentBalance < amount) {
      await pg.query('ROLLBACK');
      return res.status(400).json({ error: 'Insufficient balance' });
    }
    
    // Deduct amount from user's credits
    const updateQuery = `
      UPDATE credits 
      SET ${currencyLower} = ${currencyLower} - $1 
      WHERE uid = $2
      RETURNING ${currencyLower} as new_balance
    `;
    
    const updateResult = await pg.query(updateQuery, [amount, uid]);
    
    // Create withdrawal record
    const withdrawalQuery = `
      INSERT INTO fiat_withdrawals 
        (uid, amount, currency, bank_name, account_number, account_holder_name, ifsc_code, upi_id, status)
      VALUES 
        ($1, $2, $3, $4, $5, $6, $7, $8, 'In Queue')
      RETURNING *
    `;
    
    const values = [
      uid, 
      amount, 
      currency, 
      bank_name || null, 
      account_number || null, 
      account_holder_name, 
      ifsc_code || null, 
      upi_id || null
    ];
    
    const withdrawalResult = await pg.query(withdrawalQuery, values);
    
    // Commit the transaction
    await pg.query('COMMIT');
    
    // Return both the withdrawal request and the new balance
    res.status(201).json({
      withdrawal: withdrawalResult.rows[0],
      new_balance: updateResult.rows[0].new_balance
    });
    
  } catch (error) {
    // Rollback in case of any error
    await pg.query('ROLLBACK');
    console.error('Error processing fiat withdrawal:', error);
    res.status(500).json({ error: 'Failed to process fiat withdrawal request' });
  }
};
// Update fiat withdrawal status
exports.updateFiatWithdrawStatus = async (req, res) => {
  const { id, status } = req.body;
  if (!id || !status) {
    return res.status(400).json({ error: 'Missing required fields' });
  }
  try {
    const query = 'UPDATE fiat_withdrawals SET status = $1 WHERE id = $2 RETURNING *';
    const result = await pg.query(query, [status, id]);
    
    if (result.rows.length === 0) {
      return res.status(404).json({ error: 'Fiat withdrawal not found' });
    }
    
    res.json(result.rows[0]);
  } catch (error) {
    console.error('Error updating fiat withdrawal status:', error);
    res.status(500).json({ error: 'Failed to update fiat withdrawal status' });
  }
};