var fetch = require('node-fetch');
var config = require('./config');
var CryptoUtils = require('./cryptoUtils');
var logger = require('./logger');
var pg = require('../General/Model');
const _ = require("lodash");
var NodeCache = require("../General/Dummy");

if (!config.developer) {
  NodeCache = require("node-cache");
}

function GameController() {}

GameController.BASE_URL = config.gameApi.baseUrl;
GameController.API_KEY = config.gameApi.apiKey;
GameController.API_SECRET = config.gameApi.apiSecret;

// Helper to add auth headers
GameController.addAuthHeaders = function(req, body) {
  var timestamp = Date.now().toString();
  var params = Object.assign({}, body, { timestamp: timestamp });
  var signature = CryptoUtils.generateSignature(params, GameController.API_SECRET);
  return {
    'x-api-key': GameController.API_KEY,
    'x-timestamp': timestamp,
    'x-signature': signature,
    'Content-Type': 'application/json'
  };
};

// Launch Game

GameController.getGameLaunchURL = function(req, res) {
  var game_uid = req.body.game_uid;
  var user_id = req.body.user_id;
  var currency_code = req.body.currency_code;
  var language = req.body.language || 'en';

  if (!game_uid || !user_id || !currency_code) {
    return res.status(400).json({ success: false, error: 'Missing required fields' });
  }

  // First get user info to retrieve name
  pg.query('SELECT name FROM users WHERE id = $1', [user_id], function(err, userResult) {
    if (err) {
      logger.error('Database error fetching user', { error: err });
      return res.status(500).json({ success: false, error: 'Server error' });
    }

    if (!userResult.rows[0]) {
      return res.status(404).json({ success: false, error: 'User not found' });
    }

    var userName = userResult.rows[0].name;

    // Then get balance information
    pg.query(`
    SELECT 
        uid,
        btc,
        eth,
        ltc,
        bch,
        usdt,
        trx,
        doge,
        ada,
        xrp,
        bnb,
        usdp,
        nexo,
        mkr,
        tusd,
        usdc,
        busd,
        nc,
        inr,
        shib,
        matic,
        sc,
        mvr,
        bjb,
        aed,
        npr,
        pkr
    FROM credits 
    WHERE uid = $1
`, [user_id], function(err, balanceResult) {
      if (err) {
        logger.error('Database error fetching balance', { error: err });
        return res.status(500).json({ success: false, error: 'Server error' });
      }

      var userBalance = balanceResult.rows[0];
      if (!userBalance) {
        return res.status(404).json({ success: false, error: 'User not found in credits' });
      }

      var credit_amount = userBalance[currency_code.toLowerCase()] || 0;
      // if (credit_amount <= 0) {
      //   return res.status(400).json({ success: false, error: 'Insufficient balance' });
      // }

      var payload = {
        game_uid: game_uid,
        user_id: user_id.toString(),
        credit_amount: credit_amount.toString(),
        currency_code: currency_code,
        language: language,
        name: userName
      };

      fetch(GameController.BASE_URL + 'launch', {
        method: 'POST',
        headers: GameController.addAuthHeaders(req, payload),
        body: JSON.stringify(payload)
      })
      .then(function(response) { return response.json(); })
      .then(function(data) {
        if (!data.success || !data.data.game_launch_url) {
          return res.status(400).json({ success: false, error: 'Game under maintenance.' });
        }

        var game_launch_url = data.data.game_launch_url;
        var session_token = data.data.session_token;

        pg.query(
          'INSERT INTO game_sessions (user_id, game_uid, session_token, launch_url) VALUES ($1, $2, $3, $4)',
          [user_id, game_uid, session_token, game_launch_url],
          function(err) {
            if (err) {
              logger.error('Database error saving session', { error: err });
              return res.status(500).json({ success: false, error: 'Server error' });
            }

            logger.info('Game launched', { user_id: user_id, game_uid: game_uid });
            res.json({ success: true, data: { game_launch_url: game_launch_url, session_token: session_token } });
          }
        );
      })
      .catch(function(error) {
        logger.error('Game launch failed', { error: error });
        res.status(500).json({ success: false, error: 'Server error' });
      });
    });
  });
};
var CurrencyUtils = {
  convertFromUSD: function(usdAmount, currency, callback) {
    if (currency === 'usd') {
      return callback(null, Number(usdAmount.toFixed(8)));
    }

    pg.query(
      'SELECT usd_rate FROM exchangerate WHERE currency = $1',
      [currency.toUpperCase()], // Ensure currency matches table (e.g., USDT, INR)
      function(err, result) {
        if (err || !result.rows.length) {
          return callback(new Error('No exchange rate found for ' + currency));
        }

        var usdRate = parseFloat(result.rows[0].usd_rate);
        // Convert USD to target currency: amount / usd_rate
        var amountInCurrency = usdAmount / usdRate;
        callback(null, Number(amountInCurrency.toFixed(8)));
      }
    );
  }
};
// Init Cache
const myCache = new NodeCache({
  stdTTL: 100,
  checkperiod: 60,
  maxKeys: 10000,
  useClones: false,
});

GameController.processBetCallback = function(req, res) {
  const { user_id, transaction_type, amount, currency, balance, transaction_id } = req.body;

  if (!user_id || !transaction_type || !currency || balance === undefined) {
    logger.error('Invalid callback payload', { body: req.body });
    return res.status(400).json({ success: false, error: 'Invalid payload' });
  }

  const newBalance = Number(parseFloat(balance).toFixed(8));
  const currencyColumn = currency.toLowerCase() === 'inr' ? 'inr' : currency.toLowerCase();
  let rakebackAmount = 0;
  if (transaction_type === 'bet') {
    if (currency.toLowerCase() === 'inr') 
    {
      pg.query(
        'SELECT usd_rate FROM exchangerate WHERE currency = $1',
        ['INR'],
        function(err, rateResult) {
          if (err) {
            logger.error('Database error fetching exchange rate', { error: err });
            return res.status(500).json({ success: false, error: 'Server error' });
          }
          
          const inrToUsdRate = rateResult.rows[0].usd_rate;
          const amountInUsdt = parseFloat(amount) * inrToUsdRate;
          rakebackAmount = Number((amountInUsdt * 0.002).toFixed(8));
        }
      );
    }
    else
    {
      rakebackAmount = Number((parseFloat(amount) * 0.002).toFixed(8));
    }
  }
  pg.query(
    'INSERT INTO game_transactions (user_id, game_uid, transaction_type, amount, currency, external_transaction_id, additional_data) VALUES ($1, $2, $3, $4, $5, $6, $7) RETURNING id',
    [
      user_id,
      req.body.game_uid || 'unknown',
      transaction_type,
      amount,
      currency.toLowerCase(),
      transaction_id || 'unknown',
      JSON.stringify(req.body),
    ],
    function(err) {
      if (err) {
        logger.error('Database error saving transaction', { error: err });
        return res.status(500).json({ success: false, error: 'Server error' });
      }
      if (transaction_type === 'bet') {
        pg.query(
          'UPDATE users SET rakeamount = rakeamount + $1 WHERE id = $2 RETURNING rakeamount',
          [rakebackAmount, user_id],
          function(err, rakebackResult) {
            if (err) {
              logger.error('Database error updating rakeback', { error: err });
              return res.status(500).json({ success: false, error: 'Server error' });
            }

            logger.info('Rakeback updated', { user_id, rakebackAmount, newRakeback: rakebackResult.rows[0] });
          }
        );
      }
      pg.query('SELECT usdt, inr FROM credits WHERE uid = $1', [user_id], function(err, balanceResult) {
        if (err) {
          logger.error('Database error fetching balance', { error: err });
          return res.status(500).json({ success: false, error: 'Server error' });
        }

        if (balanceResult.rows.length) {
          pg.query(
            `UPDATE credits SET ${currencyColumn} = $1 WHERE uid = $2 RETURNING ${currencyColumn}`,
            [newBalance, user_id],
            function(err, updateResult) {
              if (err) {
                logger.error('Database error updating balance', { error: err });
                return res.status(500).json({ success: false, error: 'Server error' });
              }

              logger.info('Bet callback processed', { user_id, transaction_type, currency, newBalance });
              res.json({
                success: true,
                new_balance: updateResult.rows[0][currencyColumn].toString(),
              });
            }
          );
        } 
      });
    }
  );
};
GameController.getActiveGames = function(req, res) {
    var vendor = req.query.vendor;
    var page = req.query.page || '1';
    var per_page = req.query.per_page || '20';
  
    fetch(GameController.BASE_URL + 'games?vendor=' + (vendor || '') + '&page=' + page + '&per_page=' + per_page, {
      method: 'GET',
      headers: GameController.addAuthHeaders(req, {})
    })
    .then(function(response) { return response.json(); })
    .then(function(data) {
      // Pass through the admin API response as-is (including errors)
      if (!data.success) {
        return res.status(400).json(data); // Admin API will provide detailed errors
      }
      res.json({ success: true, data: data.data });
    })
    .catch(function(error) {
      logger.error('Failed to fetch games', { error: error });
      res.status(500).json({ success: false, error: 'Server error' });
    });
  };
  
  // Search Games (Simple proxy to admin API)
  GameController.userhistory = function(req, res) {
    var userid= req.query.userid || '';
   
  
    var url = 'https://games.ibitplay.com/api/user/'+ 'user/history?user_id=' + userid;
  
    fetch(url, {
      method: 'GET',
      headers: GameController.addAuthHeaders(req, {})
    })
    .then(function(response) { return response.json(); })
    .then(function(data) {
      // Pass through the admin API response as-is (including errors)
      if (!data.success) {
        return res.status(400).json(data); // Admin API will provide detailed errors
      }
      res.json({ success: true, data: data.data });
    })
    .catch(function(error) {
      logger.error('Failed to search games', { error: error });
      res.status(500).json({ success: false, error: 'Server error' });
    });
  };

  GameController.userhistoryAdmin = function(req, res) {

   
  
    var url = 'https://games.ibitplay.com/api/user/'+ 'historyToClient';
  
    fetch(url, {
      method: 'GET',
      headers: GameController.addAuthHeaders(req, {})
    })
    .then(function(response) { return response.json(); })
    .then(function(data) {
      // Pass through the admin API response as-is (including errors)
      if (!data.success) {
        return res.status(400).json(data); // Admin API will provide detailed errors
      }
      res.json({ success: true, data: data.data });
    })
    .catch(function(error) {
      logger.error('Failed to search games', { error: error });
      res.status(500).json({ success: false, error: 'Server error' });
    });
  };

  GameController.searchGames = function(req, res) {
    var keyword = req.query.keyword || '';
    var vendor = req.query.vendor;
    var page = req.query.page || '1';
    var per_page = req.query.per_page || '20';
  
    var url = GameController.BASE_URL + 'games/search?keyword=' + encodeURIComponent(keyword) +
              (vendor ? '&vendor=' + vendor : '') + '&page=' + page + '&per_page=' + per_page;
  
    fetch(url, {
      method: 'GET',
      headers: GameController.addAuthHeaders(req, {})
    })
    .then(function(response) { return response.json(); })
    .then(function(data) {
      // Pass through the admin API response as-is (including errors)
      if (!data.success) {
        return res.status(400).json(data); // Admin API will provide detailed errors
      }
      res.json({ success: true, data: data.data });
    })
    .catch(function(error) {
      logger.error('Failed to search games', { error: error });
      res.status(500).json({ success: false, error: 'Server error' });
    });
  };
module.exports = GameController;