const pg = require('../General/Model');
const path = require('path');
const fs = require('fs');
const multer = require('multer');

// Set up storage for document uploads
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    const uploadDir = path.join(__dirname, '../../uploads/kyc');
    
    // Create directory if it doesn't exist
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    
    cb(null, uploadDir);
  },
  filename: function (req, file, cb) {
    // Generate unique filename with userId, timestamp and original extension
    const userId = req.body.userId;
    const timestamp = Date.now();
    const fileExt = path.extname(file.originalname);
    const fileName = `${userId}_${file.fieldname}_${timestamp}${fileExt}`;
    cb(null, fileName);
  }
});

// File filter to only allow image files
const fileFilter = (req, file, cb) => {
  if (file.mimetype === 'image/jpeg' || file.mimetype === 'image/png') {
    cb(null, true);
  } else {
    cb(new Error('Only JPEG and PNG files are allowed'), false);
  }
};

// Set up multer with storage and size limits
const upload = multer({
  storage: storage,
  limits: {
    fileSize: 1024 * 1024 // 1MB
  },
  fileFilter: fileFilter
});

// Middleware to handle document uploads
const uploadDocuments = upload.fields([
  { name: 'documentFront', maxCount: 1 },
  { name: 'documentBack', maxCount: 1 }
]);

// Helper to handle async routes
const asyncHandler = fn => (req, res, next) =>
  Promise.resolve(fn(req, res, next)).catch(next);

// Get KYC status for a user
const getKycStatus = asyncHandler(async (req, res) => {
  const userId = req.params.userId;

  if (!userId) {
    return res.status(400).json({ error: 'User ID is required' });
  }

  try {
    const result = await pg.query(
      'SELECT status, rejection_reason FROM user_kyc WHERE user_id = $1',
      [userId]
    );

    if (result.rows.length === 0) {
      // If no record exists, return unverified status
      return res.status(200).json({ 
        status: 'Unverified',
        message: 'No KYC record found for this user'
      });
    }

    return res.status(200).json({
      status: result.rows[0].status,
      rejectionReason: result.rows[0].rejection_reason || null
    });
  } catch (error) {
    console.error('Database error when fetching KYC status:', error);
    return res.status(500).json({ error: 'Failed to retrieve KYC status' });
  }
});

// Submit KYC information
const submitKyc = asyncHandler(async (req, res) => {
  const {
    userId,
    firstName,
    lastName,
    gender,
    dob,
    address,
    city,
    country,
    documentType
  } = req.body;

  if (!userId || !firstName || !lastName || !gender || !dob || !address || !city || !country || !documentType) {
    return res.status(400).json({ error: 'All fields are required' });
  }

  if (!req.files || !req.files.documentFront) {
    return res.status(400).json({ error: 'Document front is required' });
  }

  const documentFrontFile = req.files.documentFront ? req.files.documentFront[0].filename : null;
  const documentBackFile = req.files.documentBack ? req.files.documentBack[0].filename : null;

  const documentFrontPath = documentFrontFile ? `/uploads/kyc/${documentFrontFile}` : null;
  const documentBackPath = documentBackFile ? `/uploads/kyc/${documentBackFile}` : null;
  const normalizedDocType = documentType.toLowerCase();

  let idFrontPath = null;
  let idBackPath = null;
  let passportPath = null;

  if (normalizedDocType === 'idcard') {
    idFrontPath = documentFrontPath;
    idBackPath = documentBackPath;
  } else if (normalizedDocType === 'passport') {
    passportPath = documentFrontPath;
  }

  try {
    const existingKyc = await pg.query(
      'SELECT id FROM user_kyc WHERE user_id = $1',
      [userId]
    );

    let result;
    if (existingKyc.rows.length > 0) {
      result = await pg.query(
        `UPDATE user_kyc 
         SET first_name = $1, last_name = $2, gender = $3, date_of_birth = $4, 
             address = $5, city = $6, country = $7, document_type = $8,
             id_front_path = $9, id_back_path = $10, passport_path = $11,
             status = 'Pending', rejection_reason = NULL, updated_at = CURRENT_TIMESTAMP
         WHERE user_id = $12
         RETURNING id`,
        [
          firstName, lastName, gender, dob, address, city, country, normalizedDocType,
          idFrontPath, idBackPath, passportPath, userId
        ]
      );
    } else {
      result = await pg.query(
        `INSERT INTO user_kyc 
         (user_id, first_name, last_name, gender, date_of_birth, address, city, country, 
          document_type, id_front_path, id_back_path, passport_path, status)
         VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, 'Pending')
         RETURNING id`,
        [
          userId, firstName, lastName, gender, dob, address, city, country, normalizedDocType,
          idFrontPath, idBackPath, passportPath
        ]
      );
    }

    return res.status(200).json({
      message: 'KYC information submitted successfully',
      kycId: result.rows[0].id,
      status: 'Pending'
    });

  } catch (error) {
    console.error('Error submitting KYC:', error);
    return res.status(500).json({ error: 'Failed to submit KYC information: ' + error.message });
  }
});




// For admin use: Update KYC status
const updateKycStatus = asyncHandler(async (req, res) => {
    const { userId, status, rejectionReason } = req.body;
  
    if (!userId || !status || !['Pending', 'Verified', 'Rejected'].includes(status)) {
      return res.status(400).json({ error: 'Valid user ID and status are required' });
    }
  
    // If status is rejected, rejection reason is required
    if (status === 'Rejected' && !rejectionReason) {
      return res.status(400).json({ error: 'Rejection reason is required when status is Rejected' });
    }
  
    try {
      const result = await pg.query(
        `UPDATE user_kyc 
         SET status = $1, rejection_reason = $2, updated_at = CURRENT_TIMESTAMP
         WHERE user_id = $3
         RETURNING id`,
        [status, rejectionReason || null, userId]
      );
  
      if (result.rows.length === 0) {
        return res.status(404).json({ error: 'No KYC record found for this user' });
      }
  
      return res.status(200).json({
        message: `KYC status updated to ${status} successfully`
      });
    } catch (error) {
      console.error('Database error when updating KYC status:', error);
      return res.status(500).json({ error: 'Failed to update KYC status' });
    }
  });
  const getAllKycApplications = asyncHandler(async (req, res) => {
    try {
      // You might want to add pagination in the future
      const result = await pg.query(
        `SELECT 
          uk.id, 
          uk.user_id as "userId", 
          uk.first_name as "firstName", 
          uk.last_name as "lastName", 
          uk.gender, 
          uk.date_of_birth as "dob", 
          uk.address, 
          uk.city, 
          uk.country, 
          uk.document_type as "documentType", 
          uk.id_front_path as "idFrontPath", 
          uk.id_back_path as "idBackPath", 
          uk.passport_path as "passportPath", 
          uk.status, 
          uk.rejection_reason as "rejectionReason", 
          uk.created_at as "createdAt", 
          uk.updated_at as "updatedAt"
        FROM user_kyc uk
        ORDER BY 
          CASE 
            WHEN uk.status = 'Pending' THEN 0 
            WHEN uk.status = 'Rejected' THEN 1
            ELSE 2 
          END, 
          uk.updated_at DESC`
      );
  
      return res.status(200).json(result.rows);
    } catch (error) {
      console.error('Database error when fetching KYC applications:', error);
      return res.status(500).json({ error: 'Failed to retrieve KYC applications' });
    }
  });

  
  
module.exports = {
  getKycStatus,
  submitKyc,
  updateKycStatus,
  uploadDocuments,
  getAllKycApplications
};