const db = require('../General/Model');

const getSportsConfig = async (req, res) => {
    try {
      // Fetch sports configuration from the database
      const sportsConfigQuery = await db.query(
        'SELECT id, game_id, game_name, enabled, created_at, updated_at FROM sports_config'
      );
  
      // Log the fetched data for debugging
      console.log('Sports Configuration:', sportsConfigQuery.rows);
  
      // Send the sports configuration data as a JSON response
      res.json(sportsConfigQuery.rows);
    } catch (error) {
      console.error('Sports config fetch error:', error);
      res.status(500).json({
        error: 'Error fetching sports configuration',
        details: error.message
      });
    }
  };
 
  
const addGame = async (req, res) => {
    try {
      const { game_id, game_name, enabled } = req.body;
  
      if (!game_id || !game_name || typeof enabled !== 'boolean') {
        return res.status(400).json({ error: 'Invalid or missing fields' });
      }
  
      const query = `
        INSERT INTO sports_config (game_id, game_name, enabled, created_at, updated_at)
        VALUES ($1, $2, $3, NOW(), NOW())
        RETURNING *;
      `;
      const values = [game_id, game_name, enabled];
  
      const result = await db.query(query, values);
      res.status(201).json({ message: 'Game added successfully', game: result.rows[0] });
    } catch (error) {
      console.error('Add game error:', error);
      res.status(500).json({ error: 'Error adding game', details: error.message });
    }
  };
  const updateGame = async (req, res) => {
    try {
      const { id } = req.params;
      const { game_name, enabled } = req.body;
  
      if (!game_name && typeof enabled !== 'boolean') {
        return res.status(400).json({ error: 'No fields to update' });
      }
  
      const fields = [];
      const values = [];
      let query = 'UPDATE sports_config SET ';
      let index = 1;
  
      if (game_name) {
        fields.push(`game_name = $${index++}`);
        values.push(game_name);
      }
  
      if (typeof enabled === 'boolean') {
        fields.push(`enabled = $${index++}`);
        values.push(enabled);
      }
  
      query += fields.join(', ');
      query += `, updated_at = NOW() WHERE id = $${index} RETURNING *`;
      values.push(id);
  
      const result = await db.query(query, values);
  
      if (result.rowCount === 0) {
        return res.status(404).json({ error: 'Game not found' });
      }
  
      res.json({ message: 'Game updated successfully', game: result.rows[0] });
    } catch (error) {
      console.error('Update game error:', error);
      res.status(500).json({ error: 'Error updating game', details: error.message });
    }
  };
  const deleteGame = async (req, res) => {
    try {
      const { id } = req.params;
  
      const query = 'DELETE FROM sports_config WHERE id = $1 RETURNING *';
      const result = await db.query(query, [id]);
  
      if (result.rowCount === 0) {
        return res.status(404).json({ error: 'Game not found' });
      }
  
      res.json({ message: 'Game deleted successfully', game: result.rows[0] });
    } catch (error) {
      console.error('Delete game error:', error);
      res.status(500).json({ error: 'Error deleting game', details: error.message });
    }
  };
  

  module.exports = {
    getSportsConfig,
    addGame,
    updateGame,
    deleteGame,
  };
  