const axios = require('axios');
const db = require('../General/Model');

const getEventsBySportsID = async (req, res) => {
  try {
    // Log the requested sports ID
    console.log('Requested Sports ID:', req.query.id);

    // Validate sports ID is provided
    if (!req.query.id) {
      return res.status(400).json({ 
        error: 'Sports ID is required' 
      });
    }

    // Check if the requested sports is enabled
    const enabledSportsQuery = await db.query(
      `SELECT
     id,
     game_id,
     game_name,
     enabled,
     created_at,
     updated_at
   FROM sports_config WHERE game_id = $1 AND enabled = true`, 
      [req.query.id]
    );

    // If the sports is not enabled, return error message
    if (enabledSportsQuery.rows.length === 0) {
      console.warn(`Sports ID ${req.query.id} is not enabled`);
      return res.status(403).json({ 
        error: 'Game not enabled',
        id: req.query.id
      });
    }

    // Make request to external API with sports ID
    const response = await axios.get(`http://89.116.20.218:8085/api/getEventsBySportsID?id=${req.query.id}`, {
      headers: {
        'x-rapidapi-key': process.env.RAPIDAPI_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31',
        'x-rapidapi-host': process.env.RAPIDAPI_HOST ?? 'betfair14.p.rapidapi.com',
        'X-ScoreSwift-Key': process.env.SCORESWIFT_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31'
      }
    });

    console.log('Fetched Events:', response.data);

    // Return the response directly
    res.json(response.data);

  } catch (error) {
    console.error('Events by Sports ID fetch error:', error);

    // Detailed error logging
    if (error.response) {
      console.error('Error Response Data:', error.response.data);
      console.error('Error Response Status:', error.response.status);
      
      // Pass through the external API's error response
      return res.status(error.response.status).json(error.response.data);
    }

    res.status(500).json({
      error: 'Error fetching events by sports ID',
      details: error.message
    });
  }
};

const validateSportsConfig = async (sportId) => {
    try {
      const enabledSportsQuery = await db.query(
       `SELECT
     id,
     game_id,
     game_name,
     enabled,
     created_at,
     updated_at
   FROM sports_config WHERE game_id = $1 AND enabled = true`,
        [sportId]
      );
  
      return enabledSportsQuery.rows.length > 0;
    } catch (error) {
      console.error('Sports config validation error:', error);
      throw error;
    }
  };


  const getEventsBySeriesAndSportsID = async (req, res) => {
    try {
      const { sid: seriesId, sportid: sportId } = req.query;
  
      // Validate series and sports IDs are provided
      if (!seriesId || !sportId) {
        return res.status(400).json({
          error: 'Series ID and Sports ID are required'
        });
      }
  
      // Check if the requested sports is enabled
      const isEnabled = await validateSportsConfig(sportId);
      if (!isEnabled) {
        console.warn(`Sports ID ${sportId} is not enabled`);
        return res.status(403).json({
          error: 'Game not enabled',
          id: sportId
        });
      }
  
      // Make request to external API with series and sports IDs
      const response = await axios.get(`http://89.116.20.218:8085/api/getEvents?sid=${seriesId}&sportid=${sportId}`, {
        headers: {
          'x-rapidapi-key': process.env.RAPIDAPI_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31',
          'x-rapidapi-host': process.env.RAPIDAPI_HOST ?? 'betfair14.p.rapidapi.com',
          'X-ScoreSwift-Key': process.env.SCORESWIFT_KEY ?? 'eNp57Jb7yHgI26S8Yw4u_v31'
        }
      });
  
      console.log('Fetched Events by Series and Sports ID:', response.data);
      res.json(response.data);
    } catch (error) {
      console.error('Events by Series and Sports ID fetch error:', error);
      
      if (error.response) {
        console.error('Error Response Data:', error.response.data);
        console.error('Error Response Status:', error.response.status);
        return res.status(error.response.status).json(error.response.data);
      }
  
      res.status(500).json({
        error: 'Error fetching events by series and sports ID',
        details: error.message
      });
    }
  };
  
  module.exports = { 
    getEventsBySportsID,
    getEventsBySeriesAndSportsID
  };