// controllers/depositController.js
const pg   = require('../../General/Model');
const fs        = require('fs');                // keep for sync checks
const fsp       = require('fs/promises');       // ← NEW: always present
const path      = require('path');
const { assertInHierarchy } = require('./middleware');
const { v4: uuid }= require('uuid');
const mime  = require('mime-types');
/* helper – user row from username */
async function fetchUser(username){
  const { rows:[u] } = await pg.query(
    'SELECT id, parent_staff_id FROM users WHERE name=$1',[username]);
  return u;
}
async function persistScreenshot(file) {
  if (file.path) return file.path;              // diskStorage branch

  const UPLOAD_DIR = path.resolve('uploads/deposit_screenshots');
  await fsp.mkdir(UPLOAD_DIR, { recursive: true });      // ← fsp

  const fname = `deposit_${Date.now()}_${file.originalname.replace(/\s+/g,'_')}`;
  const full  = path.join(UPLOAD_DIR, fname);

  await fsp.writeFile(full, file.buffer);                 // ← fsp
  return full;
}


/* ------------------------------------------------------------------ */
/*  Controller                                                        */
/* ------------------------------------------------------------------ */

exports.createDeposit = async (req, res) => {
  try {
    /* --------------------------------------------------------------
     *  1)  Validate inbound data
     * ------------------------------------------------------------ */
    const {
      username,          // the player
      handled_by,        // upline-staff id  (comes from localStorage)
      amount,
      transaction_id,
      bank_name,
      account_number,
      ifsc_code,
      account_holder_name,
      upi_id           = ''   // optional
    } = req.body;

    if (!req.file)
      return res.status(400).json({ error: 'Screenshot is required' });

    if (!username || !amount || !transaction_id)
      return res.status(400).json({ error: 'Missing mandatory fields' });


     const { rows: existing } = await pg.query(
      'SELECT deposit_id FROM fiat_deposits WHERE transaction_id = $1',
      [transaction_id.trim()]
    );
    if (existing.length > 0) {
      return res.status(200).json({ error: 'Transaction ID already exists' });
    }

    /* --------------------------------------------------------------
     *  2)  Resolve the user → user_id          (once – cheap)
     * ------------------------------------------------------------ */
    const { rows:[usr] } = await pg.query(
      'SELECT id FROM users WHERE name = $1',
      [username.trim()]
    );
    if (!usr)
      return res.status(404).json({ error: 'Unknown username' });

    const user_id = usr.id;

    /* --------------------------------------------------------------
     *  3)  Persist the screenshot if needed
     * ------------------------------------------------------------ */
    const screenshot_path = await persistScreenshot(req.file);

    /* --------------------------------------------------------------
     *  4)  Insert the deposit
     * ------------------------------------------------------------ */
    const sql = `
      INSERT INTO fiat_deposits (
          user_id, amount, currency,
          bank_name, account_number, ifsc_code, account_holder_name,
          upi_id, screenshot_path, transaction_id,
          status, handled_by
      ) VALUES (
          $1,$2,'INR',
          $3,$4,$5,$6,
          $7,$8,$9,
          'pending',$10
      )
      RETURNING deposit_id, status, created_at
    `;

    const vals = [
      user_id, amount,
      bank_name, account_number, ifsc_code, account_holder_name,
      upi_id || null, screenshot_path, transaction_id,
      handled_by
    ];

    const { rows:[row] } = await pg.query(sql, vals);

    return res.status(201).json({
      success : true,
      message : 'Deposit request created',
      data    : row
    });

  } catch (err) {
    console.error('createDeposit()', err);
    return res.status(500).json({ error: 'Server error' });
  }
};

/* ------------------------------------------------------------------ */
/* STAFF – pending list (only own)                                    */
/* ------------------------------------------------------------------ */
exports.getPendingDeposits = async (req,res)=>{
  const staffId = req.user.id;              // authenticated staff
  const { rows } = await pg.query(`
      SELECT fd.*, u.username
        FROM fiat_deposits fd
        JOIN users u ON u.id = fd.user_id
       WHERE fd.status='pending'
         AND fd.handled_by = $1
       ORDER BY fd.created_at ASC`,[staffId]);
  res.json({success:true,data:rows});
};
// backend/system/deposit/controller.js
exports.listDeposits = async (req, res) => {
  try {
    /* ------------------------------------------------------------
     * staff-id now comes FROM THE PATH  /admin/:staffId
     * ----------------------------------------------------------- */
    const staffId = Number(req.params.staffId);     // e.g. ".../admin/14"
    if (!staffId) return res.status(400).json({ error: 'staffId required' });

    const { status } = req.query;                  // ?status=pending …

    const conds  = ['fd.handled_by = $1'];
    const values = [staffId];

    if (status) {                                  // optional filter
      values.push(status);
      conds.push(`fd.status = $${values.length}`);
    }

    const { rows } = await pg.query(
      `
        SELECT fd.*, u.name AS username
          FROM fiat_deposits fd
          JOIN users u ON u.id = fd.user_id
         WHERE ${conds.join(' AND ')}
         ORDER BY fd.created_at DESC
      `,
      values
    );

    res.json({ success: true, data: rows });
  } catch (err) {
    console.error('listDeposits()', err);
    res.status(500).json({ error: 'Server error' });
  }
};

/* -------------------------------------------------------------------------- */
/*  helper: read caller’s staff-id from custom header                         */
/* -------------------------------------------------------------------------- */
function headerStaffId(req) {
  const id = Number(req.headers['x-staff-id']);
  if (!Number.isFinite(id)) throw new Error('x-staff-id header required');
  return id;
}

/* -------------------------------------------------------------------------- */
/*  internal util – shared by approve/reject                                  */
/* -------------------------------------------------------------------------- */
async function changeStatus(req, res, newStatus) {
  try {
    const depId     = req.params.id;                    // /approve/:id
    const comment   = req.body.admin_comment || '';
    const callerId  = headerStaffId(req);               // ← from header

    /* 1 ─ fetch the deposit row we’re about to touch */
    const { rows: [dep] } = await pg.query(
      `SELECT user_id, handled_by
         FROM fiat_deposits
        WHERE deposit_id = $1`,
      [depId]
    );
    if (!dep) return res.status(404).json({ error: 'Deposit not found' });

    /* 2 ─ permission check
       ────────────────
       Allowed IF caller == handled_by
       OR caller is ANY ancestor of handled_by in the staff_hierarchy table */
    const { rows: [allowed] } = await pg.query(
      `SELECT 1
         FROM staff_hierarchy
        WHERE ancestor_id   = $1      -- caller
          AND descendant_id = $2      -- original handler
        LIMIT 1`,
      [callerId, dep.handled_by]
    );
    if (!allowed && callerId !== dep.handled_by)
      return res.status(403).json({ error: 'Forbidden: outside your hierarchy' });

    /* 3 ─ do the update inside a transaction */
    await pg.query('BEGIN');

    await pg.query(
      `UPDATE fiat_deposits
          SET status        = $1,
              admin_comment = $2,
              handled_at    = NOW()
        WHERE deposit_id    = $3`,
      [newStatus, comment, depId]
    );

    /* 4 ─ credit the player if we approved */
    if (newStatus === 'approved') {
      await pg.query(
        `UPDATE credits
            SET inr = inr + (
                 SELECT amount FROM fiat_deposits WHERE deposit_id = $1
               )
          WHERE uid = $2`,
        [depId, dep.user_id]
      );
    }

    await pg.query('COMMIT');
    return res.json({ success: true });
  } catch (e) {
    /*  make sure a started transaction doesn’t stay open  */
    await pg.query('ROLLBACK').catch(() => {});
    return res.status(500).json({ error: e.message });
  }
}
async function hasHierarchyAccess(db, ancestorId, descendantId) {
  if (ancestorId === descendantId) return true;
  const { rowCount } = await db.query(
    `SELECT 1 FROM staff_hierarchy
      WHERE ancestor_id = $1 AND descendant_id = $2 LIMIT 1`,
    [ancestorId, descendantId]
  );
  return rowCount > 0;
}

/* -------------------------------------------------------------------------- */
/*  public endpoints used by the router                                       */
/* -------------------------------------------------------------------------- */
// exports.approveDeposit = (req, res) => changeStatus(req, res, 'approved');
exports.rejectDeposit  = (req, res) => changeStatus(req, res, 'rejected');
// ---------------------------------------------------------------------------
// PUT /api/fiat_deposits/admin/approve/:id
// ---------------------------------------------------------------------------
exports.approveDeposit = async (req, res) => {
  /* get a dedicated connection – do NOT reuse the pool handle */
  

  try {
    /* ------------------------------------------------------------------ */
    /* 0) basic input                                                     */
    /* ------------------------------------------------------------------ */
    const depositId = Number(req.params.id);
    const callerId  = headerStaffId(req);   // ← sent in x-staff-id header

    if (!depositId) {
      console.log("[approve] bad id →", req.params.id);
      return res.status(400).json({ error: "Bad id" });
    }
    console.log("[approve] caller =", callerId, "deposit =", depositId);

    /* ------------------------------------------------------------------ */
    /* 1) fetch + lock the deposit row                                    */
    /* ------------------------------------------------------------------ */
    const { rows: [dep] } = await pg.query(
      `SELECT user_id, amount, handled_by, status
         FROM fiat_deposits
        WHERE deposit_id = $1
          AND status     = 'pending'    
        FOR UPDATE`,                         
      [depositId]
    );

    if (!dep) {
      console.log("[approve] not-found / already done");
      return res.status(404).json({ error: "Deposit not found or already handled" });
    }
    console.log("[approve] deposit row =", dep);

    /* ------------------------------------------------------------------ */
    /* 2) hierarchy/permission                                            */
    /* ------------------------------------------------------------------ */
    const ok = await hasHierarchyAccess(pg, callerId, dep.handled_by);
    if (!ok) {
      console.log("[approve] forbidden – outside hierarchy");
      return res.status(403).json({ error: "Forbidden" });
    }

    /* ------------------------------------------------------------------ */
    /* 3) balance check on the approver                                   */
    /* ------------------------------------------------------------------ */
    const balRes = await pg.query(
      `SELECT inr FROM staff_balances WHERE staff_id = $1 FOR UPDATE`,
      [callerId]
    );
    const balance = balRes.rows[0]?.inr || 0;
    console.log("[approve] staff balance =", balance, "need", dep.amount);

    if (balance < dep.amount)
      return res.status(200).json({ error: "Insufficient INR" });

    /* ------------------------------------------------------------------ */
    /* 4) everything in ONE transaction                                   */
    /* ------------------------------------------------------------------ */
    await pg.query("BEGIN");

    /* 4a) mark approved ------------------------------------------------- */
    await pg.query(
      `UPDATE fiat_deposits
          SET status     = 'approved',
              handled_at = NOW()
        WHERE deposit_id = $1`,
      [depositId]
    );
    console.log("[approve] deposit marked approved");

    /* 4b) transfer record ---------------------------------------------- */
    await pg.query(
      `INSERT INTO staff_transfers
             (from_type, from_id, to_type, to_id, amount, direction)
       VALUES ('staff', $1, 'user', $2, $3, 'deposit')`,
      [callerId, dep.user_id, dep.amount]
    );
    console.log("[approve] staff_transfers row inserted");

    /* 4c) debit staff --------------------------------------------------- */
    await pg.query(
      `UPDATE staff_balances
          SET inr = inr - $2
        WHERE staff_id = $1`,
      [callerId, dep.amount]
    );

    /* 4d) credit player ------------------------------------------------- */
    await pg.query(
      `UPDATE credits
          SET inr = inr + $2
        WHERE uid = $1`,
      [dep.user_id, dep.amount]
    );

    await pg.query("COMMIT");
    console.log("[approve] COMMIT ok");
    return res.json({ success: true });

  } catch (err) {
    await pg.query("ROLLBACK").catch(() => {});
    console.error("approveDeposit error:", err);
    return res.status(500).json({ error: err.message });

  } finally {
    pg.release();      // ← always release!
  }
};


/* ------------------------------------------------------------------ */
exports.getUserDeposits = async (req,res)=>{
  const { username } = req.params;
  const user = await fetchUser(username);
  if(!user) return res.status(404).json({error:'not found'});

  await assertInHierarchy(req.user.id, user.id);
const { rows } = await pg.query(
  `SELECT
     deposit_id,
     user_id,
     amount,
     currency,
     bank_name,
     account_number,
     ifsc_code,
     account_holder_name,
     upi_id,
     screenshot_path,
     transaction_id,
     status,
     admin_comment,
     created_at,
     updated_at,
     handled_by,
     handled_at
   FROM fiat_deposits
   WHERE user_id = $1
   ORDER BY created_at DESC`,
  [user.id]
);


  res.json({success:true,data:rows});
};


// controller.js  – only the caller / authorisation part changed
exports.getDepositScreenshot = async (req, res) => {
  try {
    const depId   = Number(req.params.id);
    const rawSid  = req.params.staff_id;               // may be undefined
    const staffId = Number(rawSid);
    const staffMode = Number.isInteger(staffId) && staffId > 0;

    /* 1)  fetch deposit meta -------------------------------------- */
    const { rows:[dep] } = await pg.query(
      `SELECT screenshot_path, user_id, handled_by
         FROM fiat_deposits
        WHERE deposit_id = $1`,
      [depId]
    );
    if (!dep) return res.status(404).json({ error: 'not found' });

    /* 2)  authorise ------------------------------------------------ */
    if (staffMode) {
      // make sure that staffId exists *and* is in hierarchy
      await assertInHierarchy(staffId, dep.user_id);
    } else {
      // player – caller must be the owner => supplied via session / token
      if (!req.user || req.user.id !== dep.user_id) {
        return res.status(403).json({ error: 'forbidden' });
      }
    }

    /* 3)  stream image ------------------------------------------- */
    if (!fs.existsSync(dep.screenshot_path)) {
      return res.status(410).json({ error: 'file removed' });
    }
    const ctype = mime.lookup(path.extname(dep.screenshot_path)) || 'image/jpeg';
    res.type(ctype);
    fs.createReadStream(dep.screenshot_path).pipe(res);
  } catch (err) {
    console.error('getDepositScreenshot() error:', err);
    res.status(500).json({ error: 'server error' });
  }
};


// controllers/depositController.js  (add at bottom or replace old list fn)
exports.listByHierarchy = async (req, res) => {
  const staffId = Number(req.params.staffId);          // “1”, “7”, …
console.log(staffId);
  /* optional filter –  /admin/7?status=pending  */
  const status = (req.query.status ?? '').toLowerCase();  // pending | approved | rejected | ''
  const whereStatus = ['pending','approved','rejected'].includes(status)
    ? 'AND fd.status = $2'
    : '';
  const sqlParams  = whereStatus ? [staffId, status] : [staffId];
 console.log(sqlParams);
  /* ------------------------------------------------------------------
   * the staff_hierarchy table tells us which staff belong to *my* tree
   * ---------------------------------------------------------------- */
  const sql = `
    SELECT fd.*, u.name AS username, sh.ancestor_id AS staff_id
      FROM fiat_deposits  AS fd
      JOIN users          AS u  ON u.id = fd.user_id
      JOIN staff_hierarchy sh   ON sh.descendant_id = fd.handled_by
     WHERE sh.ancestor_id = $1
       ${whereStatus}
     ORDER BY fd.created_at DESC
  `;
 console.log(sql);
  const { rows } = await pg.query(sql, sqlParams);
  console.log('rowss',rows);
  res.json({ success: true, data: rows });
};

exports.getUserPL = async (req, res) => {
  try {
    /* --------------------------------------------------------------
     *  1) Validate inbound data
     * ------------------------------------------------------------ */
    const { userId } = req.body;

    if (!userId || isNaN(userId)) {
      return res.status(400).json({ error: 'Valid userId is required' });
    }

    /* --------------------------------------------------------------
     *  2) Verify user exists
     * ------------------------------------------------------------ */
    const { rows: [user] } = await pg.query(
      'SELECT id FROM users WHERE id = $1',
      [userId]
    );
    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    /* --------------------------------------------------------------
     *  3) Calculate total deposits from staff_transfers
     * ------------------------------------------------------------ */
    const { rows: [depositResult] } = await pg.query(
      `SELECT COALESCE(SUM(amount), 0) AS total_deposits
       FROM staff_transfers
       WHERE to_type = 'user' AND to_id = $1 AND direction = 'deposit'`,
      [userId]
    );

    const totalDeposits = parseFloat(depositResult.total_deposits) || 0;

    /* --------------------------------------------------------------
     *  4) Calculate total approved withdrawals from fiat_withdrawals
     * ------------------------------------------------------------ */
    const { rows: [withdrawalResult] } = await pg.query(
      `SELECT COALESCE(SUM(amount), 0) AS total_withdrawals
       FROM fiat_withdrawals
       WHERE uid = $1 AND status = 'approved'`,
      [userId]
    );

    const totalWithdrawals = parseFloat(withdrawalResult.total_withdrawals) || 0;

    /* --------------------------------------------------------------
     *  5) Calculate P/L
     * ------------------------------------------------------------ */
    const profitLoss = totalDeposits - totalWithdrawals;

    /* --------------------------------------------------------------
     *  6) Return result
     * ------------------------------------------------------------ */
    return res.status(200).json({
      success: true,
      message: 'P/L calculated successfully',
      data: {
        userId,
        totalDeposits,
        totalWithdrawals,
        profitLoss,
        currency: 'INR'
      }
    });

  } catch (err) {
    console.error('getUserPL()', err);
    return res.status(500).json({ error: 'Server error' });
  }
};

exports.getStaffPL = async (req, res) => {
  try {
    /* --------------------------------------------------------------
     *  1) Validate inbound data
     * ------------------------------------------------------------ */
    const { staffId } = req.body;

    if (!staffId || isNaN(staffId)) {
      return res.status(400).json({ error: 'Valid staffId is required' });
    }

    /* --------------------------------------------------------------
     *  2) Verify staff exists
     * ------------------------------------------------------------ */
    const { rows: [staff] } = await pg.query(
      'SELECT id FROM staff WHERE id = $1',
      [staffId]
    );
    if (!staff) {
      return res.status(404).json({ error: 'Staff not found' });
    }

    /* --------------------------------------------------------------
     *  3) Calculate total deposits from staff_transfers
     * ------------------------------------------------------------ */
    const { rows: [depositResult] } = await pg.query(
      `SELECT COALESCE(SUM(amount), 0) AS total_deposits
       FROM staff_transfers
       WHERE to_type = 'staff' AND to_id = $1 AND direction = 'deposit'`,
      [staffId]
    );

    const totalDeposits = parseFloat(depositResult.total_deposits) || 0;

    /* --------------------------------------------------------------
     *  4) Calculate total withdrawals from staff_transfers
     * ------------------------------------------------------------ */
    const { rows: [withdrawalResult] } = await pg.query(
      `SELECT COALESCE(SUM(amount), 0) AS total_withdrawals
       FROM staff_transfers
       WHERE from_type = 'staff' AND from_id = $1 AND direction = 'withdraw'`,
      [staffId]
    );

    const totalWithdrawals = parseFloat(withdrawalResult.total_withdrawals) || 0;

    /* --------------------------------------------------------------
     *  5) Calculate P/L
     * ------------------------------------------------------------ */
    const profitLoss = totalDeposits - totalWithdrawals;

    /* --------------------------------------------------------------
     *  6) Return result
     * ------------------------------------------------------------ */
    return res.status(200).json({
      success: true,
      message: 'P/L calculated successfully',
      data: {
        staffId,
        totalDeposits,
        totalWithdrawals,
        profitLoss,
        currency: 'INR'
      }
    });

  } catch (err) {
    console.error('getStaffPL()', err);
    return res.status(500).json({ error: 'Server error' });
  }
};