// backend/system/helpers/staffTree.js
// ----------------------------------
const pg = require('../../General/Model');

/** Return an array of ALL staff-ids that are below (or equal to) `staffId`. */
exports.visibleStaffIds = async function visibleStaffIds(staffId) {
  const { rows:[{ ids }] } = await pg.query(
    `WITH RECURSIVE t AS (
         SELECT id FROM staff WHERE id = $1
         UNION ALL
         SELECT s.id FROM staff s
         JOIN t ON s.parent_id = t.id
     )
     SELECT array_agg(id) AS ids FROM t`,
    [staffId]
  );
  return ids || [];
};

/** Every _player_ that sits anywhere **below** `staffId` (descendants). */
exports.visibleUserIds = async function visibleUserIds (staffId) {
  const { rows:[{ uids }] } = await pg.query(
    `WITH RECURSIVE branch(id) AS (
         SELECT id        FROM staff WHERE id = $1       -- myself
         UNION ALL
         SELECT s.id      FROM staff s
         JOIN   branch b  ON s.parent_id = b.id          -- all descendants
     )
     SELECT array_agg(u.id) AS uids
       FROM users u
       JOIN branch b ON b.id = u.parent_staff_id`,
    [staffId]
  );
  return uids || [];                // [] when no players yet
};

/** Every _staff_ that sits anywhere **above** the user (ancestors).  
 *  – Super-Admin (id 1) is always included.                        */
exports.uplineStaffIds = async function uplineStaffIds (userId) {
  const { rows } = await pg.query(
    `SELECT DISTINCT ancestor_id
       FROM staff_hierarchy h
       JOIN users u ON u.parent_staff_id = h.descendant_id
      WHERE u.id = $1`,
    [userId]
  );
  return rows.length ? rows.map(r => Number(r.ancestor_id)) : [1]; // fallback: root only
};