const crypto = require('crypto');
const axios = require('axios');
const pg = require('../General/Model');

const WPAY_HOST = 'api.wpay.one';
const WPAY_MCH_ID = '1905';
const WPAY_KEY = '04f727e3f6c240b2b00805041255f7a7';


const WPAY_CURRENCY_MCH_IDS = {
    'INR': '1905',
    'IDR': '1818',
    'PHP': '1819', 
    'BDT': '1820',
    'PKR': '1821'
};
const STATUS_CODES = {
    0: 'success',
    1: 'failed',
    2: 'mchId error',
    3: 'account does not exist',
    4: 'account abnormal',
    5: 'sign error',
    6: 'order already exists',
    7: 'order does not exist',
    8: 'no permission',
    9: 'Insufficient balance',
    10: 'Wrong amount',
    11: 'Channel maintenance',
    12: 'Currency does not exist',
    13: 'Channel does not exist',
    15: 'Channel failed',
    16: 'ip error',
    17: 'bank does not exist'
};

const ORDER_STATUS = {
    0: 'pending',
    1: 'success',
    2: 'failed'
};
const generateSignature = (params) => {
    console.log('Starting signature generation with params:', JSON.stringify(params, null, 2));
    
    // Step 1: Create clean object without empty values and 'sign'
    const cleanParams = {};
    for (const [key, value] of Object.entries(params)) {
        if (value !== '' && value !== null && value !== undefined && key !== 'sign') {
            // Ensure all values are strings
            cleanParams[key] = String(value);
        }
    }
    console.log('Clean params:', JSON.stringify(cleanParams, null, 2));

    // Step 2: Get keys and sort them by ASCII value
    const keys = Object.keys(cleanParams).sort();
    console.log('Sorted keys:', keys);

    // Step 3: Build signature string with strict ordering
    let signatureStr = '';
    for (let i = 0; i < keys.length; i++) {
        const key = keys[i];
        if (i === 0) {
            signatureStr = `${key}=${cleanParams[key]}`;
        } else {
            signatureStr += `&${key}=${cleanParams[key]}`;
        }
    }
    
    // Step 4: Add key at the end
    signatureStr += `&key=${WPAY_KEY}`;
    console.log('Final string before MD5:', signatureStr);

    // Step 5: Generate MD5
    const md5Hash = crypto.createHash('md5')
        .update(signatureStr)
        .digest('hex')
        .toLowerCase();
    
    console.log('Generated MD5:', md5Hash);
    return md5Hash;
};

class PaymentController {
    constructor() {
        this.storePayInTransaction = this.storePayInTransaction.bind(this);
        this.storePayOutTransaction = this.storePayOutTransaction.bind(this);
        this.updatePayInStatus = this.updatePayInStatus.bind(this);
        this.updatePayOutStatus = this.updatePayOutStatus.bind(this);
        this.initiatePayIn = this.initiatePayIn.bind(this);
        this.initiatePayOut = this.initiatePayOut.bind(this);
        this.handlePayInCallback = this.handlePayInCallback.bind(this);
        this.handlePayOutCallback = this.handlePayOutCallback.bind(this);
        this.getTransactionStatus = this.getTransactionStatus.bind(this);
        this.utrRepair = this.utrRepair.bind(this);
    }

    async storePayInTransaction(userId, transactionData) {
        
        try {
            
            await pg.query('BEGIN');

            const query = `
                INSERT INTO pay_in_transactions 
                (user_id, transaction_id, out_trade_no, currency, amount, status, pay_type)
                VALUES ($1, $2, $3, $4, $5, $6, $7)
                RETURNING id
            `;
            
            const values = [
                userId,
                transactionData.transaction_Id,
                transactionData.out_trade_no,
                transactionData.currency,
                transactionData.money,
                0,
                transactionData.pay_type
            ];

            const result = await pg.query(query, values);
            await pg.query('COMMIT');
            return result.rows[0];
        } catch (error) {
            if (pg) await pg.query('ROLLBACK');
            throw error;
        } finally {
            console.log("done");
        }
    }

    async storePayOutTransaction(userId, transactionData) {
        
        try {
            
            await pg.query('BEGIN');

            const query = `
                INSERT INTO pay_out_transactions 
                (user_id, transaction_id, out_trade_no, currency, amount, status, 
                pay_type, account, account_name, ifsc_code)
                VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
                RETURNING id
            `;
            
            const values = [
                userId,
                transactionData.transaction_Id,
                transactionData.out_trade_no,
                transactionData.currency,
                transactionData.money,
                0,
                transactionData.pay_type,
                transactionData.account,
                transactionData.userName,
                transactionData.reserve1
            ];

            const result = await pg.query(query, values);
            await pg.query('COMMIT');
            return result.rows[0];
        } catch (error) {
            if (pg) await pg.query('ROLLBACK');
            throw error;
        } finally {
            console.log("done");
        }
    }

 

    async updatePayOutStatus(callbackData) {
        
        try {
            
            await pg.query('BEGIN');

            const updateQuery = `
                UPDATE pay_out_transactions
                SET status = $1,
                    merchant_ratio = $2,
                    real_amount = $3,
                    updated_at = CURRENT_TIMESTAMP
                WHERE out_trade_no = $4
                RETURNING id
            `;

            const values = [
                callbackData.status,
                callbackData.merchant_ratio,
                callbackData.real_money,
                callbackData.out_trade_no
            ];

            await pg.query(updateQuery, values);
            await pg.query('COMMIT');
        } catch (error) {
            if (pg) await pg.query('ROLLBACK');
            throw error;
        } finally {
            console.log("done");
        }
    }


    async initiatePayOut(req, res) {
        try {
            const {
                userId,
                currency = 'INR',
                out_trade_no,
                pay_type = 'BANK',
                account,
                userName,
                money,
                attach = '',
                notify_url,
                reserve1
            } = req.body;

            if (!userId || !out_trade_no || !money || !notify_url || !account || !userName) {
                return res.status(400).json({
                    code: 1,
                    msg: 'Missing required fields'
                });
            }

            const payloadData = {
                mchId: WPAY_MCH_ID,
                currency,
                out_trade_no,
                pay_type,
                account,
                userName,
                money,
                attach,
                notify_url,
                reserve1
            };

            payloadData.sign = generateSignature(payloadData);

            const response = await axios.post(
                `https://${WPAY_HOST}/v1/Payout`,
                new URLSearchParams(payloadData).toString(),
                {
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded'
                    }
                }
            );

            if (response.data.code === 0) {
                await this.storePayOutTransaction(userId, {
                    ...payloadData,
                    transaction_Id: response.data.data.transaction_Id
                });
            }

            res.json(response.data);
        } catch (error) {
            console.error('PayOut Error:', error);
            res.status(500).json({
                code: 1,
                msg: STATUS_CODES[1],
                error: error.message
            });
        }
    }

    async updatePayInStatus(callbackData) {
        try {
            await pg.query('BEGIN');

            const updateTxQuery = `
                UPDATE pay_in_transactions
                SET status = $1,
                    pay_amount = $2,
                    merchant_ratio = $3,
                    real_amount = $4,
                    updated_at = CURRENT_TIMESTAMP
                WHERE out_trade_no = $5
                RETURNING user_id, real_amount, currency
            `;

            const txValues = [
                callbackData.status,
                callbackData.pay_money,
                callbackData.merchant_ratio,
                callbackData.real_money,
                callbackData.out_trade_no
            ];

            const txResult = await pg.query(updateTxQuery, txValues);

            if (callbackData.status === '1' && txResult.rows.length > 0) {
                const { user_id, real_amount, currency } = txResult.rows[0];
                const currencyLower = currency.toLowerCase();
                const amount=real_amount+real_amount*0.05;

                const updateCreditsQuery = `
                    UPDATE credits 
                    SET ${currencyLower} = ${currencyLower} + $1 
                    WHERE uid = $2
                `;

                await pg.query(updateCreditsQuery, [
                    amount,
                    user_id
                ]);
            }

            await pg.query('COMMIT');
            return txResult.rows[0];
        } catch (error) {
            if (pg) await pg.query('ROLLBACK');
            throw error;
        } finally {
            console.log("done");
        }
    }

    async initiatePayIn(req, res) {
        try {
            console.log('PayIn Request Body:', JSON.stringify(req.body, null, 2));

            const {
                userId,
                currency = 'INR',
                out_trade_no,
                pay_type,
                money,
                notify_url,
                returnUrl,
                phone
            } = req.body;

            // Validation
            if (!userId || !out_trade_no || !money || !notify_url) {
                return res.status(400).json({
                    code: 1,
                    msg: 'Missing required fields'
                });
            }

            // Get correct merchant ID for currency
            const mchId = WPAY_CURRENCY_MCH_IDS[currency];
            if (!mchId) {
                return res.status(400).json({
                    code: 12,
                    msg: 'Currency does not exist'
                });
            }

            // Create base payload with required fields
            const payload = {
                mchId: mchId,
                currency: currency,
                out_trade_no: out_trade_no,
                pay_type: pay_type,
                money: money,
                attach:"",
                notify_url: notify_url
            };

            // Add optional fields if they exist and are not empty
            if (returnUrl) payload.returnUrl = returnUrl;
            if (phone) payload.phone = phone;

            // Generate signature
            const sign = generateSignature(payload);
            payload.sign = sign;

            console.log('Final request payload:', JSON.stringify(payload, null, 2));

            // Make the API request
            const response = await axios.post(
                `https://${WPAY_HOST}/v1/Collect`,
                new URLSearchParams(payload).toString(),
                {
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded'
                    }
                }
            );

            console.log('API Response:', JSON.stringify(response.data, null, 2));

            if (response.data.code === 0) {
                await this.storePayInTransaction(userId, {
                    ...payload,
                    transaction_Id: response.data.data.transaction_Id
                });
            }

            res.json(response.data);
        } catch (error) {
            console.error('PayIn Error:', error);
            res.status(500).json({
                code: 1,
                msg: 'Payment initiation failed',
                error: error.message
            });
        }
    }


    async handlePayInCallback(req, res) {
        try {
            console.log('Received PayIn callback data:', JSON.stringify(req.body, null, 2));
            const callbackData = req.body;
            
            const receivedSign = callbackData.sign;
            console.log('Received signature:', receivedSign);
            
            const calculatedSign = generateSignature(callbackData);
            console.log('Calculated signature:', calculatedSign);

            if (receivedSign !== calculatedSign) {
                console.error('PayIn Callback: Invalid signature');
                console.error('Signature mismatch:', {
                    received: receivedSign,
                    calculated: calculatedSign
                });
                return res.status(400).send('Invalid signature');
            }

            console.log('Signature verified, updating transaction status...');
            await this.updatePayInStatus(callbackData);
            res.send('success');
        } catch (error) {
            console.error('PayIn Callback Error:', error);
            res.status(500).send('Internal server error');
        }
    }

    async handlePayOutCallback(req, res) {
        try {
            const callbackData = req.body;

            const receivedSign = callbackData.sign;
            const calculatedSign = generateSignature(callbackData);

            if (receivedSign !== calculatedSign) {
                console.error('PayOut Callback: Invalid signature');
                return res.status(400).send('Invalid signature');
            }

            await this.updatePayOutStatus(callbackData);
            res.send('success');
        } catch (error) {
            console.error('PayOut Callback Error:', error);
            res.status(500).send('Internal server error');
        }
    }

    async getTransactionStatus(req, res) {
        
        try {
            
            const { type, out_trade_no } = req.params;
           const query = type === 'payin'
  ? `SELECT
       id,
       user_id,
       transaction_id,
       out_trade_no,
       currency,
       amount,
       pay_amount,
       merchant_ratio,
       real_amount,
       status,
       pay_type,
       created_at,
       updated_at
     FROM pay_in_transactions
     WHERE out_trade_no = $1`
  : `SELECT
       id,
       user_id,
       transaction_id,
       out_trade_no,
       currency,
       amount,
       pay_amount,
       merchant_ratio,
       real_amount,
       status,
       pay_type,
       created_at,
       updated_at
     FROM pay_out_transactions
     WHERE out_trade_no = $1`;


            const result = await pg.query(query, [out_trade_no]);
            
            if (result.rows.length === 0) {
                return res.status(404).json({
                    code: 7,
                    msg: STATUS_CODES[7]
                });
            }

            res.json({
                code: 0,
                msg: 'success',
                data: result.rows[0]
            });
        } catch (error) {
            console.error('Get Transaction Status Error:', error);
            res.status(500).json({
                code: 1,
                msg: STATUS_CODES[1],
                error: error.message
            });
        } finally {
            console.log("done");
        }
    }

    async utrRepair(req, res) {
        try {
            const { utr, out_trade_no } = req.body;

            const payloadData = {
                mchId: WPAY_MCH_ID,
                utr,
                out_trade_no
            };

            payloadData.sign = generateSignature(payloadData);

            const response = await axios.post(
                `https://${WPAY_HOST}/v1/UtrRepair`,
                new URLSearchParams(payloadData).toString(),
                {
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded'
                    }
                }
            );

            res.json(response.data);
        } catch (error) {
            console.error('UTR Repair Error:', error);
            res.status(500).json({
                code: 1,
                msg: STATUS_CODES[1],
                error: error.message
            });
        }
    }
}

module.exports = new PaymentController();