import { useEffect, useState } from "react";
import styled from "styled-components";
import { getUID, sendNotfication } from "../../Helper";
import { FontAwesomeIcon } from '@fortawesome/react-fontawesome';
import { faArrowLeft, faSpinner } from '@fortawesome/free-solid-svg-icons';

const Container = styled.div`
    position: relative;
    display: flex;
    justify-content: space-between;
    gap:24px;
    align-items: center;
    margin: 16px auto;
    background: linear-gradient(135deg, #1a1a1a 0%, #000000 100%);
    color: #fff;
    overflow: hidden;
    padding: 16px;
    border: 1px solid #333;
    border-radius: 12px;
    box-shadow: 0 6px 24px rgba(248, 156, 3, 0.2);
    max-width: 1200px;
    width: 100%;

    @media (max-width: 768px) {
        flex-direction: column;
        align-items: flex-start;
        padding: 8px;
    }
`;

const BackgroundSVG = styled.svg`
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    z-index: 0;
    opacity: 0.2;
`;

const Content = styled.div`
    flex: 1;
    text-align: left;
    height: 270px;
    z-index: 1;
    background: rgba(255, 255, 255, 0.05);
    padding: 12px 16px;
    border-radius: 8px;

    @media (max-width: 768px) {
        padding: 6px 10px;
        width: 100%;
          height: 290px !important;
        background: linear-gradient(
            to bottom right,
            rgba(0, 0, 0, 0.4),
            rgba(248, 156, 3, 0.2)
        ), url(${props => props.$bgImage});
        background-size: cover;
        background-position: center;
        box-shadow: 
            inset 0 0 20px rgba(0, 0, 0, 0.5),
            0 4px 12px rgba(248, 156, 3, 0.2);
    }
`;

const Media = styled.div`
    flex: 1;
    text-align: right;
    z-index: 1;
    display: flex;
    justify-content: flex-end;
    align-items: center;

    @media (max-width: 768px) {
        justify-content: center;
        width: 100%;
        margin-top: 10px;
        display:none;
    }
`;

const VideoContainer = styled.div`
    position: relative;
    width:100%;
    height: 270px;
    border-radius: 12px;
    overflow: hidden;

    @media (max-width: 768px) {
        width: 200px;
        height: 120px;
        border-radius: 8px;
    }

    @media (min-width: 769px) and (max-width: 1024px) {
        width: 250px;
        height: 160px;
    }
`;

const Image = styled.img`
    width: 100%;
    height: 100%;
    object-fit: cover;
`;
const Title = styled.h1`
    font-size: 2.5rem;
    font-weight: bold;
    margin-bottom: 0.5rem;
    background: linear-gradient(90deg, #F89C03, #F89C03);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;

    @media (max-width: 768px) {
        display: none;
    }
`;

const Description = styled.p`
    font-size: 1rem;
    color: #ddd;
    padding-right: 9rem;
    text-align: left;
    margin-top: 1.5rem;
    margin-bottom: 0.5rem;
    line-height: 1.3;

    @media (max-width: 768px) {
        display: none;
    }
`;
const DescriptionMobile = styled.p`
    font-size: 1rem;
    color: #ddd;
    padding-right: 0rem;
    text-align: left;
    margin-top: 0.2rem;
    margin-bottom: 1.5rem;
    line-height: 1.3;
    display: none;
    @media (max-width: 768px) {
        display: block;
    }
`;
const PlayButton = styled.div`
    margin-top: 58px;
    padding: 10px 16px;
    background: #f89c03;
    border: none;
    text-align: center;
    border-radius: 6px;
    color: #1a1a1a;
    font-weight: bold;
    font-size: 0.9rem;
    cursor: pointer;
    width:160px;
    transition: transform 0.3s ease;
    
    &:hover {
        transform: scale(1.05);
        box-shadow: 0 4px 12px rgba(248, 156, 3, 0.3);
    }

    @media (max-width: 768px) {
     margin-top: 126px;
        padding: 6px 12px;
        font-size: 0.8rem;
          width:120px;
    }
`;

const FullScreenContainer = styled.div`
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    z-index: 999;
    background-color: #1b2336;
`;

const FullScreenIframe = styled.iframe`
    width: 100%;
    height: 100%;
    border: none;
    display: ${(props) => (props.loading ? 'none' : 'block')};
`;

const BackButton = styled.button`
    position: absolute;
    top: 10px;
    left: 10px;
    z-index: 1000;
    background: none;
    border: none;
    color: #000;
    font-size: 24px;
    cursor: pointer;

    @media (max-width: 768px) {
        font-size: 20px;
    }
`;

const LoadingSpinner = () => (
    <div style={{ display: 'inline-block', width: '16px', height: '16px', border: '2px solid #fff', borderTop: '2px solid transparent', borderRadius: '50%', animation: 'spin 1s linear infinite' }} />
);

const Esports = () => {
    const images = [
        '/assets/images/esports/first.png',
        '/assets/images/esports/first.png',
    ];

    const [currentVideoIndex, setCurrentVideoIndex] = useState(0);
    const [showTFGaming, setShowTFGaming] = useState(true);
    const [iframeUrl, setIframeUrl] = useState('');
    const [showIframe, setShowIframe] = useState(false);
    const [iframeLoading, setIframeLoading] = useState(false);
    const [loading, setLoading] = useState(false);
    const [message, setMessage] = useState(null);

    const imageStyle = {
        position: 'absolute',
        top: '50%',
        left: '50%',
        transform: 'translate(-50%, -50%)',
        width: '200px',
        animation: 'fadeinout 2s infinite',
    };
    const animationStyle = `
        @keyframes fadeinout {
            0%, 100% { opacity: 0; }
            50% { opacity: 1; }
        }
    `;

    useEffect(() => {
        const interval = setInterval(() => {
            setCurrentVideoIndex((prevIndex) => (prevIndex + 1) % images.length);
            setShowTFGaming((prev) => !prev);
        }, 5000);
        return () => clearInterval(interval);
    }, [images.length]);

   

    const fetchJSGamesByVendor = async (vendorName, page = 1, perPage = 20) => {

        const token = localStorage.getItem('token');
        try {
            const response = await fetch(`https://api.boss707.com/jsGames/games?vendor=${vendorName}&page=${page}&per_page=${perPage}`, {
                headers: {
                    'Authorization': `Bearer ${token}`,
                },
            });
            if (!response.ok) {
                throw new Error("Network response was not ok");
            }
            const result = await response.json();
            if (result.success && result.payload && result.payload.games) {
                return {
                    games: result.payload.games,
                    pagination: result.payload.pagination || {
                        current_page: page,
                        per_page: perPage,
                        total: result.payload.games.length,
                        total_pages: Math.ceil(result.payload.games.length / perPage)
                    }
                };
            } else {
                throw new Error("No games found or invalid response");
            }
        } catch (error) {
            console.error(`Error fetching ${vendorName} games:`, error.message);
            return { games: [], pagination: { current_page: 1, per_page: perPage, total: 0, total_pages: 0 } };
        }
    };

    const handleGameLaunchJSGames = async (game) => {
        const isLogged = localStorage.getItem('logged') === 'true';
        if (!isLogged) {
            sendNotfication('Please login to play', 'error');
            return;
        }
        setLoading(true);
        setMessage(null);
        try {
            const credits = localStorage.getItem('credit');
            const coins = localStorage.getItem('coin');
            const vendor = showTFGaming ? 'TFGaming' : 'TFGaming';
            if (vendor === 'TFGaming') {
                if (coins !== 'USDT' && coins !== 'INR') {
                    sendNotfication('Please swap to USDT or INR to play TFGaming', 'error');
                    setLoading(false);
                    return;
                }
            } 
            let id = getUID();
            const requestPayload = {
                game_uid: "4ee8e0051a035b463b47c3c473ce317d",
                user_id: id,
                credit_amount: credits,
                currency_code: coins,
                language: 'en',
            };
            const token = localStorage.getItem('token');
            const response = await fetch('https://api.boss707.com/jsGamesv2/launch', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Authorization': `Bearer ${token}`,
                },
                body: JSON.stringify(requestPayload),
            });
            if (!response.ok) {
                sendNotfication('Error launching game', 'error');
                setLoading(false);
                return;
            }
            const result = await response.json();
            if (result.success && result.data.game_launch_url) {
                setIframeUrl(result.data.game_launch_url);
                setShowIframe(true);
                setIframeLoading(true);
            } else {
                sendNotfication(result.message || 'Failed to launch game', 'error');
            }
        } catch (error) {
            sendNotfication(error.message || 'Error launching game', 'error');
        } finally {
            setLoading(false);
        }
    };

    const handlePlayNow = async () => {
        const isLogged = localStorage.getItem('logged') === 'true';
        if (!isLogged) {
            sendNotfication('Please login to play', 'error');
            return;
        }
        setLoading(true);
        const vendorName = 'esports';
        const targetVendor = showTFGaming ? 'TFGaming' : 'TFGaming';
        try {
            const { games } = await fetchJSGamesByVendor(vendorName);
            if (games.length > 0) {
                const selectedGame = games.find(game => game.game_name === targetVendor);
                if (selectedGame) {
                    handleGameLaunchJSGames(selectedGame);
                } else {
                    sendNotfication(`No ${targetVendor} game available`, 'error');
                    setLoading(false);
                }
            } else {
                sendNotfication('No games available for this vendor', 'error');
                setLoading(false);
            }
        } catch (error) {
            sendNotfication('Error fetching games', 'error');
            setLoading(false);
        }
    };

    const handleIframeLoad = () => {
        setIframeLoading(false);
    };

    const handleBackButton = () => {
        setShowIframe(false);
        setIframeUrl('');
        setIframeLoading(false);
    };

    if (showIframe) {
        return (
            <FullScreenContainer>
                <style>{animationStyle}</style>
                {iframeLoading && (
                    <img src="./assets/boss707.png" alt="Loading" style={imageStyle} />
                )}
                <FullScreenIframe
                    src={iframeUrl}
                    title="Game Iframe"
                    onLoad={handleIframeLoad}
                    loading={iframeLoading}
                />
                <BackButton onClick={handleBackButton}>
                    <FontAwesomeIcon icon={faArrowLeft} />
                </BackButton>
            </FullScreenContainer>
        );
    }

    return (
        <Container>
            <BackgroundSVG viewBox="0 0 1200 600">
                <defs>
                    <linearGradient id="patternGradient" x1="0%" y1="0%" x2="100%" y2="100%">
                        <stop offset="0%" stopColor="#F89C03" stopOpacity="0.8" />
                        <stop offset="100%" stopColor="#ffffff" stopOpacity="0.3" />
                    </linearGradient>
                    <filter id="glow" x="-20%" y="-20%" width="140%" height="140%">
                        <feGaussianBlur stdDeviation="3" result="blur" />
                        <feComposite in="SourceGraphic" in2="blur" operator="over" />
                    </filter>
                </defs>
                <pattern id="smallGrid" width="50" height="50" patternUnits="userSpaceOnUse">
                    <path d="M 50 0 L 0 0 0 50" fill="none" stroke="url(#patternGradient)" strokeWidth="0.5" opacity="0.3" />
                </pattern>
                <rect width="100%" height="100%" fill="url(#smallGrid)" />
                <path d="M 100 500 Q 300 400 500 500 T 900 500" stroke="url(#patternGradient)" strokeWidth="1.5" fill="none" filter="url(#glow)">
                    <animate attributeName="d" 
                        values="M 100 500 Q 300 400 500 500 T 900 500;
                                M 100 500 Q 400 300 600 500 T 900 500;
                                M 100 500 Q 300 400 500 500 T 900 500"
                        dur="6s" repeatCount="indefinite" />
                </path>
                <circle cx="150" cy="150" r="5" fill="url(#patternGradient)" filter="url(#glow)">
                    <animate attributeName="r" values="5;8;5" dur="4s" repeatCount="indefinite" />
                    <animate attributeName="opacity" values="0.8;0.4;0.8" dur="4s" repeatCount="indefinite" />
                </circle>
                <circle cx="600" cy="300" r="6" fill="url(#patternGradient)" filter="url(#glow)">
                    <animate attributeName="r" values="6;9;6" dur="5s" repeatCount="indefinite" />
                    <animate attributeName="opacity" values="0.7;0.3;0.7" dur="5s" repeatCount="indefinite" />
                </circle>
                <circle cx="1050" cy="450" r="4" fill="url(#patternGradient)" filter="url(#glow)">
                    <animate attributeName="r" values="4;7;4" dur="3s" repeatCount="indefinite" />
                    <animate attributeName="opacity" values="0.9;0.5;0.9" dur="3s" repeatCount="indefinite" />
                </circle>
                <line x1="0" y1="600" x2="1200" y2="0" stroke="url(#patternGradient)" strokeWidth="1" opacity="0.2">
                    <animate attributeName="stroke-dashoffset" from="0" to="50" dur="4s" repeatCount="indefinite" />
                    <animate attributeName="stroke-dasharray" from="0 50" to="50 0" dur="4s" repeatCount="indefinite" />
                </line>
            </BackgroundSVG>
            <Content $bgImage={images[currentVideoIndex]}>
                <Title style={{ 
                  
                }}>
                    {showTFGaming ? 'TF Gaming' : 'TF Gaming'}
                </Title>
                <Description style={{ 
                 
                }}>
                    Experience the ultimate E-sports games powered by {showTFGaming ? 'TFGaming' : 'TFGaming'}. 
                    Enjoy real-time odds, secure transactions, and thrilling gameplay.
                </Description>
                <DescriptionMobile style={{ 
                 
                }}>
                    Experience the ultimate E-sports games.
                    
                </DescriptionMobile>
                <PlayButton onClick={handlePlayNow} disabled={loading}>
                    {loading ? <><LoadingSpinner /> Loading...</> : 'Play Now'}
                </PlayButton>
            </Content>
            <Media>
                <VideoContainer>
                    <Image
                        src={images[currentVideoIndex]}
                        key={images[currentVideoIndex]}
                    />
                </VideoContainer>
            </Media>
        </Container>
    );
};

export default Esports;