
import React, { useEffect, useState } from 'react';

// If your images live in src/, keep these imports:
import exclusiveSprite from '../../../flipperAnimation/exclusive.png';
import sportsSprite from '../../../flipperAnimation/sports.png';
import casinoSprite from '../../../flipperAnimation/casino.png';
import tableSprite from '../../../flipperAnimation/table.png';
import fishingSprite from '../../../flipperAnimation/fishing.png';
import crashSprite from '../../../flipperAnimation/crash.png';
import arcadeSprite from '../../../flipperAnimation/arcade.png';
import lotterySprite from '../../../flipperAnimation/lottery.png';

// If your images live in /public/flipperAnimation, you can instead set sprite to "/flipperAnimation/xxx.png"
// and remove the imports above.

import './SportsNavFlipper.css';

function SportsNavFlipper() {
  const [selectedItem, setSelectedItem] = useState('Exclusive');
  const [animatingItem, setAnimatingItem] = useState(null);
  const [spritePositions, setSpritePositions] = useState({});

  // Configure your tabs here. `frames` must match your sprite sheet.
  const gridItems = [
    { id: 'Exclusive', label: 'Exclusive', sprite: exclusiveSprite, frames: 16, color: '#1a472a' },
    { id: 'Sports',   label: 'Sports',   sprite: sportsSprite,   frames: 16, color: '#1a3c47' },
    { id: 'Casino',   label: 'Casino',   sprite: casinoSprite,   frames: 16, color: '#471a3c' },
    { id: 'Table',    label: 'Table',    sprite: tableSprite,    frames: 16, color: '#1a4738' },
    { id: 'Fishing',  label: 'Fishing',  sprite: fishingSprite,  frames: 16, color: '#47411a' },
    { id: 'Crash',    label: 'Crash',    sprite: crashSprite,    frames: 16, color: '#471a1a' },
    { id: 'Arcade',   label: 'Arcade',   sprite: arcadeSprite,   frames: 16, color: '#351a47' },
    { id: 'Lottery',  label: 'Lottery',  sprite: lotterySprite,  frames: 16, color: '#1a471f' },
  ];

  // Reset non-selected items to frame 0 when selection changes and nothing is animating
  useEffect(() => {
    if (animatingItem) return;
    const next = {};
    for (const item of gridItems) {
      if (item.id !== selectedItem) next[item.id] = 0;
    }
    setSpritePositions(prev => ({ ...prev, ...next }));
   
  }, [selectedItem, animatingItem]);

  // Kick off initial animation for "Exclusive" once
  useEffect(() => {
    setAnimatingItem('Exclusive');
  }, []);

  // Core animation loop
  useEffect(() => {
    if (!animatingItem) return;

    const item = gridItems.find(i => i.id === animatingItem);
    if (!item) return;

    const totalFrames = Math.max(1, item.frames || 1);
    const animationDurationMs = 800; // total duration for one pass
    const frameInterval = Math.max(8, Math.floor(animationDurationMs / totalFrames)); // ms per frame

    let frame = 0;
    const intervalId = setInterval(() => {
      frame = (frame + 1) % totalFrames;

      setSpritePositions(prev => ({
        ...prev,
        [animatingItem]: frame
      }));

      if (frame === totalFrames - 1) {
        clearInterval(intervalId);
        setSelectedItem(animatingItem);
        setAnimatingItem(null);
      }
    }, frameInterval);

    return () => clearInterval(intervalId);
    
  }, [animatingItem]);

  const onClickItem = (id) => {
    if (id === selectedItem || animatingItem) return;
    // reset old selection frame
    setSpritePositions(prev => ({ ...prev, [selectedItem]: 0 }));
    setAnimatingItem(id);
  };

  return (
    <div className="snf-container">
      <div className="snf-scroller">
        <div className="snf-row">
          {gridItems.map((item) => {
            const isSelected = selectedItem === item.id;
            const frames = Math.max(1, item.frames || 1);
            const frameIndex = spritePositions[item.id] || 0;
            // background-size: frames * 100% makes the sheet span across frames
            // background-position: from 0% to 100% across the strip
            const bgSize = (frames * 100) + '% 100%';
            const bgPosX =
              frames > 1
                ? (frameIndex / (frames - 1)) * 100
                : 0;
            const bgPos = bgPosX + '% 0%';

            return (
              <div
                key={item.id}
                className={
                  'snf-item' +
                  (isSelected ? ' snf-item--selected snf-pulse' : '')
                }
                style={{ backgroundColor: isSelected ? item.color : '#2d2d2d' }}
                onClick={() => onClickItem(item.id)}
              >
                <div
                  className={'snf-sprite' + (isSelected ? ' snf-sprite--selected' : '')}
                  style={{
                    backgroundImage: `url(${item.sprite})`,
                    backgroundSize: bgSize,
                    backgroundPosition: bgPos
                  }}
                />
                <span className="snf-label">{item.label}</span>
              </div>
            );
          })}
        </div>
      </div>
    </div>
  );
}

export default SportsNavFlipper;
