import React, { Component } from 'react';
import io from 'socket.io-client';

class GameComponent extends Component {
  constructor(props) {
    super(props);
    this.state = {
      currentNamespace: '30s',
      timers: { '30s': 0, '1m': 0, '2m': 0 },
      sessions: { '30s': null, '1m': null, '2m': null },
      bettingOpen: { '30s': true, '1m': true, '2m': true },
      selectedValue: '',
      betAmount: '',
      betType: 'number', // 'number' or 'color'
      sessionData: { '30s': [], '1m': [], '2m': [] }
    };
    this.sockets = {};
    this.namespaces = ['30s', '1m', '2m'];
  }

  componentDidMount() {
    this.namespaces.forEach(namespace => {
      this.setupSocket(namespace);
    });
  }

  componentWillUnmount() {
    this.namespaces.forEach(namespace => {
      this.sockets[namespace].disconnect();
    });
  }

  setupSocket = (namespace) => {
    const socket = io(`https://api.boss707.com/${namespace}`);
    this.sockets[namespace] = socket;

    socket.on('sessionUpdate', (data) => this.handleSessionUpdate(namespace, data));
    socket.on('sessionTimer', (data) => this.handleSessionTimer(namespace, data));
    socket.on('bettingClosed', (data) => this.handleBettingClosed(namespace, data));
    socket.on('winningNumber', (data) => this.handleWinningNumber(namespace, data));
    socket.on('balanceUpdate', (data) => this.handleBalanceUpdate(namespace, data));
    socket.on('betOutcome', (data) => this.handleBetOutcome(namespace, data));
    socket.on('sessionData', (data) => this.handleSessionData(namespace, data));
  };

  handleSessionUpdate = (namespace, data) => {
    this.setState(prevState => ({
      sessions: { ...prevState.sessions, [namespace]: data.sessionId }
    }));
  };

  handleSessionTimer = (namespace, data) => {
    this.setState(prevState => ({
      timers: { ...prevState.timers, [namespace]: data.timeLeft }
    }));
  };

  handleBettingClosed = (namespace, data) => {
    this.setState(prevState => ({
      bettingOpen: { ...prevState.bettingOpen, [namespace]: false }
    }));
  };

  handleWinningNumber = (namespace, data) => {
    alert(`Winning number for session ${data.sessionId} in ${namespace}: ${data.number} (${data.color})`);
  };

  handleBalanceUpdate = (namespace, data) => {
    alert(`Balance update: ${data.message}, Win Amount: ${data.winAmount}, Coin Type: ${data.cointype}`);
  };

  handleBetOutcome = (namespace, data) => {
    alert(`Bet outcome for session ${data.sessionId} in ${namespace}: Bet Value: ${data.betValue}, Win Amount: ${data.winAmount}`);
  };

  handleSessionData = (namespace, data) => {
    this.setState(prevState => ({
      sessionData: { ...prevState.sessionData, [namespace]: data }
    }));
  };

  placeBet = () => {
    const { currentNamespace, selectedValue, betAmount, betType } = this.state;
    if (!selectedValue || !betAmount) {
      alert('Please select a value and enter bet amount.');
      return;
    }

    const betData = {
      uid: 'user1', // This should be the actual user ID in a real application
      value: selectedValue,
      amount: parseFloat(betAmount),
      cointype: 'coins',
      type: betType,
    };

    this.sockets[currentNamespace].emit('placeBet', betData);
  };

  handleNamespaceChange = (namespace) => {
    this.setState({ currentNamespace: namespace });
  };

  handleValueChange = (e) => {
    this.setState({ selectedValue: e.target.value });
  };

  handleAmountChange = (e) => {
    this.setState({ betAmount: e.target.value });
  };

  handleBetTypeChange = (e) => {
    this.setState({ betType: e.target.value });
  };

  render() {
    const { currentNamespace, timers, sessions, bettingOpen, selectedValue, betAmount, betType, sessionData } = this.state;

    return (
      <div>
        <div>
          {this.namespaces.map(namespace => (
            <button key={namespace} onClick={() => this.handleNamespaceChange(namespace)}>
              {namespace}
            </button>
          ))}
        </div>

        <h2>{currentNamespace} Timer</h2>
        <p>Session ID: {sessions[currentNamespace]}</p>
        <p>Time Left: {timers[currentNamespace]}s</p>
        <p>Betting Status: {bettingOpen[currentNamespace] ? 'Open' : 'Closed'}</p>

        <div>
          <h3>Place Your Bet</h3>
          <label>
            Bet Type:
            <select value={betType} onChange={this.handleBetTypeChange}>
              <option value="number">Number</option>
              <option value="color">Color</option>
            </select>
          </label>
          {betType === 'number' ? (
            <label>
              Select Number:
              <select value={selectedValue} onChange={this.handleValueChange}>
                {[...Array(10).keys()].map(num => (
                  <option key={num} value={num}>{num}</option>
                ))}
              </select>
            </label>
          ) : (
            <label>
              Select Color:
              <select value={selectedValue} onChange={this.handleValueChange}>
                <option value="red">Red</option>
                <option value="green">Green</option>
                <option value="violet">Violet</option>
              </select>
            </label>
          )}
          <label>
            Bet Amount:
            <input type="number" value={betAmount} onChange={this.handleAmountChange} />
          </label>
          <button onClick={this.placeBet}>Place Bet</button>
        </div>

        <div>
          <h3>Previous Sessions Data</h3>
          <table>
            <thead>
              <tr>
                <th>Session ID</th>
                <th>Status</th>
                <th>Winning Number</th>
                <th>Winning Color</th>
              </tr>
            </thead>
            <tbody>
              {sessionData[currentNamespace].map((session, index) => (
                <tr key={index}>
                  <td>{session.sessionId}</td>
                  <td>{session.status}</td>
                  <td>{session.betNumber}</td>
                  <td>{session.betColour}</td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>
    );
  }
}




