import styled, { keyframes } from "styled-components";
import { Search, X, Play, ChevronDown, Trash2 } from "lucide-react";
import { useGames } from "../../../../Context/GameContext";
import { useState, useEffect, useRef } from "react";

const slideIn = keyframes`
  from {
    opacity: 0;
    transform: translateY(-10px);
  }
  to {
    opacity: 1;
    transform: translateY(0);
  }
`;

const ModalOverlay = styled.div`
  position: fixed;
  top: 0;
  left: 0;
  right: 0;
  bottom: 0;
  background-color: rgba(30, 35, 40, 0.85);
  display: ${props => (props.$isOpen ? "flex" : "none")};
  justify-content: center;
  align-items: center;
  z-index: 1000;
`;

const Container = styled.div`
  width: 100%;
  max-width: 1200px;
  max-height: 90vh;
  background: #03030F;
  color: #E0E6E9;
  border-radius: 12px;
  overflow-y: auto;
  position: relative;
  font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Helvetica, Arial, sans-serif;

  @media (max-width: 768px) {
    max-height: 100vh;
    border-radius: 0;
  }
`;

const CloseButton = styled.button`
  background: rgba(30, 35, 40, 0.3);
  border: none;
  color: #E0E6E9;
  width: 32px;
  height: 32px;
  border-radius: 50%;
  display: flex;
  align-items: center;
  justify-content: center;
  cursor: pointer;
  margin: 4px 12px;
  z-index: 10;
  transition: all 0.2s ease;

  &:hover {
    background: rgba(30, 35, 40, 0.8);
    transform: rotate(90deg);
  }
`;

const GridContainer = styled.div`
  display: grid;
  grid-template-columns: repeat(3, 1fr);
  width: 100%;
  gap: 16px;
  padding: 0px;
`;

const Wrapper = styled.div`
  margin: 0 auto;
  padding: 1rem;
`;

const Header = styled.div`
  background: rgba(30, 35, 40, 0.4) !important;
  border-radius: 0.5rem;
  padding: 1px;
  margin-bottom: 1rem;
`;

const SearchBar = styled.div`
  display: flex;
  align-items: center;
  gap: 0.5rem;
`;

const SearchInput = styled.div`
  flex: 1;
  position: relative;

  input {
    width: 100%;
    background: rgba(30, 35, 40, 0.4);
    padding: 12px 12px 12px 40px;
    border: none;
    border-radius: 8px;
    color: #E0E6E9;
    font-size: 14px;

    &::placeholder {
      color: rgba(224, 230, 233, 0.6);
    }

    &:focus {
      outline: none;
      background: rgba(30, 35, 40, 0.6);
    }
  }
`;

const SearchIconWrapper = styled.div`
  position: absolute;
  left: 12px;
  top: 50%;
  transform: translateY(-50%);
  color: #E0E6E9;
  display: flex;
  align-items: center;
  justify-content: center;
`;

const ProviderDropdown = styled.div`
  position: relative;
  width: 200px;
`;

const ProviderButton = styled.button`
  width: 100%;
  background: rgb(30, 35, 40);
  padding: 12px;
  border: none;
  border-radius: 8px;
  color: #E0E6E9;
  font-size: 14px;
  font-weight: 500;
  display: flex;
  align-items: center;
  justify-content: space-between;
  cursor: pointer;
  transition: all 0.3s ease;
  box-shadow: 0 2px 6px rgba(30, 35, 40, 0.3);

  &:hover {
    background: linear-gradient(145deg, #3A4552, #2B343F);
    transform: scale(1.02);
    box-shadow: 0 4px 12px rgba(30, 35, 40, 0.4);
  }

  &:focus {
    outline: none;
    box-shadow: 0 0 0 3px rgba(59, 70, 82, 0.5);
  }
`;

const ProviderMenu = styled.div`
  position: absolute;
  top: 100%;
  left: 0;
  width: 320px;
  background: #03030F;
  border-radius: 8px;
  box-shadow: 0 6px 16px rgba(0, 0, 0, 0.5);
  z-index: 1001;
  max-height: 400px;
  overflow-y: auto;
  display: ${(props) => (props.$isOpen ? "block" : "none")};
  margin-top: 4px;
  padding: 12px;
  animation: ${slideIn} 0.2s ease-out;
`;

const SearchInputContainer = styled.div`
  position: relative;
  margin-bottom: 12px;
  
  input {
    width: 100%;
    background: rgba(30, 35, 40, 0.4);
    padding: 10px 10px 10px 36px;
    border: none;
    border-radius: 8px;
    color: #E0E6E9;
    font-size: 14px;

    &::placeholder {
      color: rgba(224, 230, 233, 0.6);
    }

    &:focus {
      outline: none;
      background: rgba(30, 35, 40, 0.6);
    }
  }
`;

const ProviderList = styled.div`
  display: flex;
  flex-direction: column;
  gap: 14px;
  margin-bottom: 16px;
`;

const ProviderItem = styled.div`
  display: flex;
  align-items: center;
  gap: 10px;
  cursor: pointer;
`;

const CheckboxContainer = styled.div`
  width: 22px;
  height: 22px;
  border: 2px solid ${props => props.checked ? '#F89C03' : '#3A4552'};
  border-radius: 50%;
  display: flex;
  align-items: center;
  justify-content: center;
  background: ${props => props.checked ? '#F89C03' : 'transparent'};
  transition: all 0.2s ease;

  &::after {
    content: ${props => props.checked ? '" "' : 'none'};
    width: 12px;
    height: 12px;
    background: #03030F;
    border-radius: 50%;
    transition: all 0.2s ease;
  }
`;

const CheckboxInput = styled.input`
  opacity: 0;
  position: absolute;
  cursor: pointer;
`;

const ProviderLogo = styled.div`
  display: flex;
  align-items: center;
  justify-content: flex-start;
  flex: 1;
`;

const Logo = styled.div`
  color: ${props => props.color || '#FFFFFF'};
  font-size: 14px;
  font-weight: 500;
`;

const ClearAllButton = styled.button`
  width: 100%;
  padding: 10px;
  background-color: transparent;
  color: #F89C03;
  border: none;
  border-radius: 8px;
  cursor: pointer;
  display: flex;
  align-items: center;
  justify-content: center;
  gap: 8px;
  font-size: 14px;

  &:hover {
    background-color: rgba(46, 231, 134, 0.1);
  }
`;

const GameOverlay = styled.div`
  position: absolute;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  background-color: rgba(30, 35, 40, 0.5);
  backdrop-filter: blur(5px);
  border-radius: 8px;
  display: flex;
  justify-content: center;
  align-items: center;
  opacity: 0;
  transition: opacity 0.2s ease;
`;

const GridWrapper = styled.div`
  display: grid;
  grid-template-columns: repeat(7, 1fr);
  gap: 12px;
  @media (max-width: 767px) {
    grid-template-columns: repeat(3, 1fr);
  }
`;

const GameCard = styled.div`
  min-width: 142px;
  height: 195px;
  border-radius: 8px;
  position: relative;
  cursor: pointer;
  display: flex;
  flex-direction: column;
  justify-content: center;
  align-items: center;
  transition: transform 0.2s ease;

  &:hover {
    ${GameOverlay} {
      opacity: 1;
    }
    transform: translateY(-5px);
  }

  @media (max-width: 768px) {
    min-width: 115px;
    height: 160px;
  }
`;

const GameImage = styled.img`
  width: 100%;
  height: 100%;
  object-fit: cover;
  border-radius: 8px;
  display: ${(props) => (props.$isLoading ? "none" : "block")};
`;

const PlayIconContainer = styled.div`
  width: 50px;
  height: 50px;
  border-radius: 50%;
  background-color: rgba(224, 230, 233, 0.2);
  display: flex;
  justify-content: center;
  align-items: center;
`;

const PlayIcon = styled(Play)`
  color: #E0E6E9;
  font-size: 24px;
`;

const SectionHeading = styled.div`
  font-size: 24px;
  font-weight: 700;
  margin: 32px 0 16px;
  color: #E0E6E9;
  position: relative;
  padding-left: 12px;

  &::before {
    content: "";
    position: absolute;
    left: 0;
    top: 0;
    height: 100%;
    width: 4px;
    background: linear-gradient(to bottom, #3A4552, #2B343F);
    border-radius: 2px;
  }
`;

const NoResultsWrapper = styled.div`
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;
  padding: 40px 20px;
  text-align: center;
  color: rgba(224, 230, 233, 0.8);
`;

const NoResultsText = styled.h3`
  font-size: 18px;
  font-weight: 500;
  margin-bottom: 8px;
`;

const NoResultsSubText = styled.p`
  font-size: 14px;
  color: rgba(224, 230, 233, 0.6);
`;

const spinAnimation = keyframes`
  0% { transform: rotate(0deg); }
  100% { transform: rotate(360deg); }
`;

const pulseAnimation = keyframes`
  0% { opacity: 0.6; }
  50% { opacity: 0.3; }
  100% { opacity: 0.6; }
`;

const LoadingWrapper = styled.div`
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;
  padding: 40px 20px;
  text-align: center;
`;

const Spinner = styled.div`
  width: 40px;
  height: 16px;
  border: 3px solid rgba(224, 230, 233, 0.1);
  border-radius: 50%;
  border-top-color: #E0E6E9;
  animation: ${spinAnimation} 0.8s linear infinite;
  margin-bottom: 16px;
`;

const LoadingText = styled.p`
  font-size: 16px;
  color: rgba(224, 230, 233, 0.8);
  margin: 0;
`;

const SkeletonLoader = styled.div`
  width: 100%;
  height: 100%;
  background: rgba(224, 230, 233, 0.1);
  border-radius: 8px;
  animation: ${pulseAnimation} 1.5s infinite ease-in-out;
`;

const LoadMoreButton = styled.button`
  width: 200px;
  padding: 10px;
  background-color: #3A4552;
  color: #E0E6E9;
  border: none;
  border-radius: 8px;
  cursor: pointer;
  margin: 20px auto;
  display: ${props => (props.$show ? 'block' : 'none')};

  &:hover {
    background-color: #2B343F;
  }
`;

const GlobalSearchModal = ({ isOpen, onClose }) => {
  if (!isOpen) return null;

  const {
    searchResults,
    handleGameLaunchXGaming,
    isLoading,
    fetchGamesForVendors,
    searchGames,
    loadMoreGames,
    currentPage,
  } = useGames();

  const [searchQuery, setSearchQuery] = useState('');
  const [providerSearchQuery, setProviderSearchQuery] = useState('');
  const [imageLoadingStates, setImageLoadingStates] = useState({});
  const [selectedVendors, setSelectedVendors] = useState([]);
  const [isDropdownOpen, setIsDropdownOpen] = useState(false);
  const dropdownRef = useRef(null);
  const debounceTimeout = useRef(null);

  const vendorCategories = [
    { key: 'evolution', displayName: 'Evolution Gaming', color: '#FFFFFF' },
    { key: 'pragmatic', displayName: 'Pragmatic Play', color: '#FFFFFF' },
    { key: 'pragmaticlive', displayName: 'Pragmatic Live', color: '#FFFFFF' },
    { key: 'spribe', displayName: 'Spribe', color: '#FFFFFF' },
    { key: 'ideal', displayName: 'Ideal', color: '#FFFFFF' },
    { key: 'microgaming', displayName: 'Microgaming', color: '#FFFFFF' },
    { key: 'pgsoft', displayName: 'PGSoft', color: '#FFFFFF' },
    { key: 'jili', displayName: 'Jili', color: '#FFFFFF' },
  ];

  useEffect(() => {
    const handleClickOutside = (event) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target)) {
        setIsDropdownOpen(false);
      }
    };
    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  useEffect(() => {
    const initialLoadingStates = {};
    searchResults.forEach((game) => {
      initialLoadingStates[game.id || game.title] = true;
    });
    setImageLoadingStates(initialLoadingStates);
  }, [searchResults]);

  const handleSearch = (e) => {
    const query = e.target.value.toLowerCase();
    setSearchQuery(query);

    if (debounceTimeout.current) {
      clearTimeout(debounceTimeout.current);
    }

    debounceTimeout.current = setTimeout(() => {
      if (query.trim() && selectedVendors.length > 0) {
        searchGames(query, selectedVendors);
      } else if (query.trim()) {
        searchGames(query, vendorCategories.map(v => v.key));
      }
    }, 500);
  };

  const handleVendorChange = (vendorKey) => {
    setSelectedVendors(prev => {
      const newVendors = prev.includes(vendorKey)
        ? prev.filter(v => v !== vendorKey)
        : [...prev, vendorKey];
      if (newVendors.length > 0) {
        fetchGamesForVendors(newVendors, 1);
      } else {
        fetchGamesForVendors([]); // Will revert to initial results
      }
      return newVendors;
    });
  };

  const clearAllVendors = () => {
    setSelectedVendors([]);
    fetchGamesForVendors([]); // Will revert to initial results
  };

  const handleImageLoad = (gameId) => {
    setImageLoadingStates(prev => ({
      ...prev,
      [gameId]: false,
    }));
  };

  const filteredProviders = vendorCategories.filter(
    provider => provider.displayName.toLowerCase().includes(providerSearchQuery.toLowerCase())
  );

  const renderContent = () => {
    if (isLoading && searchResults.length === 0) {
      return (
        <LoadingWrapper>
          <Spinner />
          <LoadingText>Loading games...</LoadingText>
        </LoadingWrapper>
      );
    }

    if (searchResults.length > 0) {
      return (
        <div>
          <GridWrapper>
            {searchResults.map((game, index) => {
              const gameId = game.id || game.title || index;
              const isImageLoading = imageLoadingStates[gameId] ?? true;

              return (
                <GameCard
                  key={gameId}
                  onClick={() => handleGameLaunchXGaming(game)}
                >
                  {isImageLoading && <SkeletonLoader />}
                  <GameImage
                    src={game.game_icon || game.details_thumbnails_300x300}
                    alt={`Game ${game.title || game.name || index}`}
                    $isLoading={isImageLoading}
                    onLoad={() => handleImageLoad(gameId)}
                    onError={() => handleImageLoad(gameId)}
                  />
                  <GameOverlay>
                    <PlayIconContainer>
                      <PlayIcon />
                    </PlayIconContainer>
                  </GameOverlay>
                </GameCard>
              );
            })}
          </GridWrapper>
          <LoadMoreButton $show={true} onClick={loadMoreGames}>Load More</LoadMoreButton>
        </div>
      );
    }

    return (
      <NoResultsWrapper>
        <NoResultsText>No games found</NoResultsText>
        <NoResultsSubText>
          Select a provider or search for a game to see results
        </NoResultsSubText>
      </NoResultsWrapper>
    );
  };

  useEffect(() => {
    return () => {
      if (debounceTimeout.current) {
        clearTimeout(debounceTimeout.current);
      }
    };
  }, []);

  return (
    <ModalOverlay $isOpen={isOpen}>
      <Container>
        <GridContainer>
          <div></div>
          <div style={{ textAlign: "center", marginTop: "6px", fontSize: "20px" }}>
            Most Popular Games
          </div>
          <div style={{ marginTop: "6px", marginRight: "10px", width: "100%", display: "flex", justifyContent: "flex-end" }}>
            <CloseButton onClick={onClose}>
              <X size={20} />
            </CloseButton>
          </div>
        </GridContainer>
        <Wrapper>
          <Header>
            <SearchBar>
              <ProviderDropdown ref={dropdownRef}>
                <ProviderButton onClick={() => setIsDropdownOpen(!isDropdownOpen)}>
                  <span>Providers</span>
                  {/* <span>Providers: {selectedVendors.length > 0 ? `+${selectedVendors.length}` : 'Select'}</span> */}
                  <ChevronDown size={18} />
                </ProviderButton>
                <ProviderMenu $isOpen={isDropdownOpen}>
                  <SearchInputContainer>
                    <SearchIconWrapper>
                      <Search size={18} color="#E0E6E9" opacity={0.7} />
                    </SearchIconWrapper>
                    <input
                      type="text"
                      placeholder="Search"
                      value={providerSearchQuery}
                      onChange={(e) => setProviderSearchQuery(e.target.value)}
                    />
                  </SearchInputContainer>
                  <ProviderList>
                    {filteredProviders.map((provider) => (
                      <ProviderItem key={provider.key} onClick={() => handleVendorChange(provider.key)}>
                        <CheckboxContainer checked={selectedVendors.includes(provider.key)}>
                          <CheckboxInput
                            type="checkbox"
                            checked={selectedVendors.includes(provider.key)}
                            onChange={() => {}}
                          />
                        </CheckboxContainer>
                        <ProviderLogo>
                          <Logo color={provider.color}>{provider.displayName}</Logo>
                        </ProviderLogo>
                      </ProviderItem>
                    ))}
                  </ProviderList>
                  <ClearAllButton onClick={clearAllVendors}>
                    <Trash2 size={16} color="#F89C03" />
                    Clear All
                  </ClearAllButton>
                </ProviderMenu>
              </ProviderDropdown>
              <SearchInput>
                <SearchIconWrapper>
                  <Search size={18} color="#E0E6E9" opacity={0.7} />
                </SearchIconWrapper>
                <input
                  type="text"
                  placeholder="Search games..."
                  value={searchQuery}
                  onChange={handleSearch}
                />
              </SearchInput>
            </SearchBar>
          </Header>
          <SectionHeading>Most Popular Games</SectionHeading>
          {renderContent()}
        </Wrapper>
      </Container>
    </ModalOverlay>
  );
};

export default GlobalSearchModal;