import React, { useEffect, useState, useCallback } from 'react';
import styled from 'styled-components';
import axios from 'axios';
import { getUID } from '../../../../Helper';
import { ChevronDown, Search } from 'lucide-react';

// Styled components remain unchanged
const DropdownContainer = styled.div`
  position: relative;
  // margin-bottom: 16px;
  flex: 1;
  max-width: 300px;
`;

const DropdownButton = styled.button`
  display: flex;
  align-items: center;
  justify-content: space-between;
  width: 100%;
  padding: 12px 16px;
  background: #03030F;
  border: 1px solid rgba(255, 255, 255, 0.1);
  border-radius: 16px;
  color: #ffffff;
  cursor: pointer;
  font-size: 16px;
  font-weight: 600;
  transition: all 0.2s ease;

  &:hover {
    transform: translateY(-1px);
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.2);
  }
`;

const DropdownPanel = styled.div`
  position: absolute;
  top: 100%;
  left: 0;
  right: 0;
  background: #03030F;
  border: 1px solid rgba(255, 255, 255, 0.1);
  border-radius: 16px;
  margin-top: 8px;
  max-height: 400px;
  overflow-y: auto;
  z-index: 1000;
  padding: 12px;
  backdrop-filter: blur(8px);
  box-shadow: 0 12px 28px rgba(0, 0, 0, 0.2);
`;

const SearchContainer = styled.div`
  position: relative;
  padding: 0px;
  border-radius: 8px;
  border: 1px solid rgb(58 65 66);
  margin-top: 12px;
`;

const SearchInput = styled.input`
  width: 100%;
  padding: 12px 12px 12px 40px;
  background: transparent;
  border: none;
  border-radius: 8px;
  color: #ffffff;
  font-size: 16px;
  &::placeholder {
    color: #fff;
  }
`;

const Input = styled.input`
  flex: 1;
  padding: 6px 10px;
  background: rgb(30, 35, 40);
  border: 1px solid rgba(255, 255, 255, 0.05);
  border-radius: 8px;
  color: #ffffff;
  font-size: 14px;
  &::placeholder {
    color: #fff;
  }
`;

const DepositButton = styled.div`
  margin-top: auto;
  flex: 1;
  padding: 7px 9px;
  display: flex;
  justify-content: center;
  align-items: center;
  background: #f09c06;
  color: white !important;
  border: none;
  border-radius: 0.5rem;
  font-weight: 600;
  cursor: pointer;
  
  &:hover {
    opacity: 0.9;
  }
`;

const SearchIcon = styled.div`
  position: absolute;
  left: 14px;
  top: 49%;
  transform: translateY(-50%);
  color: #fff;
`;

const CoinOption = styled.div`
  display: flex;
  align-items: center;
  padding: 14px;
  cursor: pointer;
  border-radius: 12px;
  transition: all 0.2s ease;

  &:hover {
    background: rgba(255, 255, 255, 0.12);
    transform: translateY(-1px);
  }

  ${props => props.isSelected && `
    background: #f09c06;
  `}
`;

const CoinName = styled.div`
  font-size: 16px;
  font-weight: bold;
`;

const Container = styled.div`
  color: #ffffff;
`;

const Container1 = styled.div`
  background: #03030F;
  color: white;
  padding: 24px;
  border-radius: 16px;
  width: 100%;
  box-shadow: 0 12px 28px rgba(0, 0, 0, 0.2);
  border: 1px solid rgba(255, 255, 255, 0.1);
  backdrop-filter: blur(8px);

  @media (max-width: 768px) {
    padding: 16px;
    border-radius: 12px;
  }
`;

const TableHeader = styled.div`
  display: grid;
  grid-template-columns: repeat(4, 1fr);
  gap: 16px;
  color: rgba(255, 255, 255, 0.7);
  font-size: 14px;
  margin-bottom: 16px;
  font-weight: 600;
  padding: 12px 0;
  border-bottom: 1px solid rgba(255, 255, 255, 0.1);

  @media (max-width: 768px) {
    display: none; // Hide headers on mobile
  }
`;

const TableRow = styled.div`
  display: grid;
  grid-template-columns: repeat(4, 1fr);
  gap: 16px;
  font-size: 14px;
  color: white;
  margin-bottom: 12px;
  padding: 16px;
  border-radius: 12px;
  transition: all 0.2s ease;
  background: rgba(255, 255, 255, 0.08);

  &:hover {
    background: rgba(255, 255, 255, 0.12);
    transform: translateY(-1px);
  }

  @media (max-width: 768px) {
    grid-template-columns: 1fr;
    gap: 8px;
    padding: 12px;
    margin-bottom: 8px;
    
    > div {
      display: grid;
      grid-template-columns: 90px 1fr;
      align-items: center;
      gap: 12px;
      
      &:before {
        content: attr(data-label);
        font-weight: 600;
        color: rgba(255, 255, 255, 0.7);
      }

      // Special handling for status div to maintain tag appearance
      &[data-label="Status"] {
        display: flex;
        justify-content: space-between;
        align-items: center;
        
        &:before {
          flex-shrink: 0;
        }
      }
    }
  }
`;

const EmptyState = styled.div`
  display: flex;
  flex-direction: column;
  align-items: center;
  justify-content: center;
  padding: 64px 0;
`;

const PlaceholderText = styled.div`
  color: #fff;
  font-size: 18px;
  margin-top: 16px;
`;

const StatusIndicator = styled.div`
  display: inline-flex;
  align-items: center;
  padding: 6px 12px;
  border-radius: 20px;
  font-size: 13px;
  font-weight: 600;
  text-transform: capitalize;
  
  ${props => {
    switch(props.status.toLowerCase()) {
      case 'processing':
        return `
          color: #FFB800;
          background: rgba(255, 184, 0, 0.15);
          border: 1px solid rgba(255, 184, 0, 0.2);
        `;
      case 'pending':
        return `
          color: #FFB800;
          background: rgba(255, 184, 0, 0.15);
          border: 1px solid rgba(255, 184, 0, 0.2);
        `;
      case 'approved':
        return `
          color: #00E096;
          background: rgba(0, 224, 150, 0.15);
          border: 1px solid rgba(0, 224, 150, 0.2);
        `;
      case 'rejected':
        return `
          color: #FF4747;
          background: rgba(255, 71, 71, 0.15);
          border: 1px solid rgba(255, 71, 71, 0.2);
        `;
      default:
        return `
          color: #f09c06;
          background: rgba(115, 198, 196, 0.15);
          border: 1px solid rgba(115, 198, 196, 0.2);
        `;
    }
  }}

  &::before {
    content: '';
    display: inline-block;
    width: 6px;
    height: 6px;
    border-radius: 50%;
    margin-right: 6px;
    background-color: currentColor;
  }

  @media (max-width: 768px) {
    width: fit-content;
  }
`;

const RefreshButton = styled.button`
  background: #f09c06;
  border: none;
  border-radius: 24px;
  color: #ffffff;
  cursor: pointer;
  padding: 12px 20px;
  margin-left: 16px;
  display: flex;
  align-items: center;
  gap: 8px;
  font-weight: 600;
  transition: all 0.2s ease;
  
  &:hover {
    background: #67b7b5;
    transform: translateY(-1px);
  }

  &:active {
    transform: translateY(0);
  }
`;

const TableContainer = styled.div`
  min-height: 300px;
  max-height: 500px;
  overflow-y: auto;
`;

const TopControls = styled.div`
  width: 100%;
  display: flex;
  justify-content: center;
  align-items: center;
  margin-bottom: 24px;

  @media (max-width: 768px) {
    flex-direction: column;
    gap: 12px;
    
    ${DropdownContainer} {
      max-width: 100%;
      width: 100%;
    }
    
    ${RefreshButton} {
      width: 100%;
      margin-left: 0;
      justify-content: center;
    }
  }
`;
const FilterContainer = styled.div`
  display: flex;
  flex-wrap: wrap;
  gap: 16px;
  margin: 16px 0;
  align-items: flex-end;
`;

const FilterGroup = styled.div`
  display: flex;
  flex-direction: column;
  min-width: 150px;
`;

const Label = styled.label`
  font-size: 12px;
  color: #fff;
  margin-bottom: 4px;
`;

const FilterSelect = styled.select`
  padding: 8px;
  border-radius: 4px;
  border: 1px solid #444;
  background: #222;
  color: #fff;
  font-size: 14px;
  outline: none;
  &:focus {
    border-color: #f09c06;
  }
`;

const FilterInput = styled.input`
  padding: 8px;
  border-radius: 4px;
  border: 1px solid #444;
  background: #222;
  color: #fff;
  font-size: 14px;
  outline: none;
  &:focus {
    border-color: #f09c06;
  }
`;

const FilterButton = styled.button`
  padding: 8px 16px;
  background: #f09c06;
  color: #fff;
  border: none;
  border-radius: 4px;
  cursor: pointer;
  font-size: 14px;
  &:hover {
    background: #e08b05;
  }
`;
const SkylaTransaction = () => {
  const [activeTab, setActiveTab] = useState("Deposit");
  const [openDropDown, setOpenDropDown] = useState(false);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [transactionData, setTransactionData] = useState([]);
  const [refreshTrigger, setRefreshTrigger] = useState(0);
  const [filters, setFilters] = useState({
    status: '',
    dateFrom: '',
    dateTo: '',
    type: ''
  });
  useEffect(() => {
    const handleClickOutside = (event) => {
      if (!event.target.closest(".dropdown-container")) {
        setOpenDropDown(false);
      }
    };

    document.addEventListener("mousedown", handleClickOutside);
    return () => document.removeEventListener("mousedown", handleClickOutside);
  }, []);

  const fetchData = useCallback(async () => {
    setLoading(true);
    setError(null);

    try {
      console.log("Fetching data for:", activeTab);
     const response = await axios.get(
  `https://api.boss707.com/api/${activeTab.toLowerCase()}New`,
  {
    params: { uid: getUID() },
    headers: {
      Authorization: `Bearer ${localStorage.getItem('token')}`
    }
  }
);

      // console.log("API Response:", response.data);

      let processedData = [];

      // Handle different response formats based on the activeTab
      if (activeTab === "Withdraw") {
        // For withdraw, combine both types of withdrawals if they exist
        const withdrawals = response.data.withdrawals || [];
        const fiatWithdrawals = response.data.fiatWithdrawals || [];
        
        // Map fiatWithdrawals to match your display format
        const formattedFiatWithdrawals = fiatWithdrawals.map(item => ({
          date: item.date,
          amount: `${item.amount} ${item.currency || ""}`,
          status: item.status,
          details: `Bank: ${item.bank_name || "N/A"}, Acc: ${item.account_number || "N/A"}`,
          type: "Fiat"
        }));
        
        // Format regular withdrawals if needed
        const formattedWithdrawals = withdrawals.map(item => ({
          date: item.date || item.created_at,
          amount: item.amount,
          status: item.status,
          details: item.details,
          type: "Crypto"
        }));
        
        processedData = [...formattedWithdrawals, ...formattedFiatWithdrawals];
      } else {
        // For deposits
        if (Array.isArray(response.data)) {
          // Direct array response
          processedData = response.data.map(item => ({
            date: item.date || item.created_at,
            amount: item.amount,
            status: item.status,
            // Use the actual transaction ID or orderid if available
            details: item.details,
            type: "Crypto"
          }));
        } else {
          // Object with deposits and fiatDeposits arrays
          const deposits = response.data.deposits || [];
          const fiatDeposits = response.data.fiatDeposits || [];
          
          // Format deposits
          const formattedDeposits = deposits.map(item => ({
            date: item.date || item.created_at,
            amount: item.amount,
            status: item.status,
            // Use the actual transaction ID or orderid
            details: item.details,
            type: "Crypto"
          }));
          
          // Format fiat deposits
          const formattedFiatDeposits = fiatDeposits.map(item => ({
            date: item.date || item.created_at,
            amount: `${item.amount} ${item.currency || ""}`,
            status: item.status,
            // Use the actual transaction ID
            details: item.details,
            type: "Fiat"
          }));
          
          processedData = [...formattedDeposits, ...formattedFiatDeposits];
        }
      }

      // Sort data by date in descending order (newest first)
      const sortedData = processedData.sort((a, b) => {
        return new Date(b.date) - new Date(a.date);
      });

      setTransactionData(sortedData);
    } catch (err) {
      console.error("API call failed", err);
      setError("Failed to fetch data.");
      setTransactionData([]);
    } finally {
      setLoading(false);
    }
  }, [activeTab, refreshTrigger]);

  useEffect(() => {
    fetchData();
  }, [fetchData]);

  const formatDateTime = (dateString) => {
    if (!dateString) return "N/A";
    return new Date(dateString).toLocaleDateString(undefined, {
      year: "numeric",
      month: "short",
      day: "numeric",
      hour: "2-digit",
      minute: "2-digit",
    });
  };

  const handleRefresh = () => {
    setRefreshTrigger(prev => prev + 1);
  };

  return (
    <Container>
      <Container1>
        <TopControls>
          <DropdownContainer className="dropdown-container">
            <DropdownButton
              onClick={(e) => {
                e.stopPropagation();
                setOpenDropDown(!openDropDown);
              }}
            >
              <CoinName>{activeTab}</CoinName>
              <ChevronDown size={20} />
            </DropdownButton>

            {openDropDown && (
              <DropdownPanel onClick={(e) => e.stopPropagation()}>
                {["Deposit", "Withdraw"].map((type) => (
                  <CoinOption
                    key={type}
                    onClick={() => {
                      if (activeTab !== type) {
                        setTransactionData([]); // Clear previous data
                        setActiveTab(type);
                        setFilters({ status: '', dateFrom: '', dateTo: '', type: '' }); // Reset filters
                      }
                      setOpenDropDown(false);
                    }}
                    isSelected={activeTab === type}
                  >
                    <CoinName>{type}</CoinName>
                  </CoinOption>
                ))}
              </DropdownPanel>
            )}
          </DropdownContainer>
          
          <RefreshButton onClick={handleRefresh}>
            <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
              <path d="M21 2v6h-6"></path>
              <path d="M3 12a9 9 0 0 1 15-6.7L21 8"></path>
              <path d="M3 22v-6h6"></path>
              <path d="M21 12a9 9 0 0 1-15 6.7L3 16"></path>
            </svg>
            Refresh
          </RefreshButton>
        </TopControls>
        <FilterContainer>
        <FilterGroup>
  <Label>Status</Label>
  <FilterSelect
    value={filters.status}
    onChange={(e) => setFilters(prev => ({ ...prev, status: e.target.value }))}
  >
    <option value="">All</option>
    {activeTab === "Deposit" ? (
      <>
        <option value="SUCCESS">Success</option>
        <option value="PENDING">Pending</option>
        <option value="FAILED">Failed</option>
        <option value="PROCESSING">Processing</option>
        <option value="APPROVED">Approved</option>
      </>
    ) : (
      <>
        <option value="In Queue">In Queue</option>
        <option value="Processing">Processing</option>
        <option value="Done">Done</option>
        <option value="Wager Not Completed">Wager Not Completed</option>
      </>
    )}
  </FilterSelect>
</FilterGroup>
  <FilterGroup>
    <Label>From Date</Label>
    <FilterInput
      type="date"
      value={filters.dateFrom}
      onChange={(e) => setFilters(prev => ({ ...prev, dateFrom: e.target.value }))}
    />
  </FilterGroup>
  <FilterGroup>
    <Label>To Date</Label>
    <FilterInput
      type="date"
      value={filters.dateTo}
      onChange={(e) => setFilters(prev => ({ ...prev, dateTo: e.target.value }))}
    />
  </FilterGroup>
  <FilterGroup>
    <Label>Type</Label>
    <FilterSelect
      value={filters.type}
      onChange={(e) => setFilters(prev => ({ ...prev, type: e.target.value }))}
    >
      <option value="">All</option>
      <option value="Crypto">Crypto</option>
      <option value="Fiat">Fiat</option>
    </FilterSelect>
  </FilterGroup>
  <FilterButton onClick={() => setFilters({ status: '', dateFrom: '', dateTo: '', type: '' })}>
    Reset Filters
  </FilterButton>
</FilterContainer>
        {loading ? (
          <EmptyState>
            <div className="spinner" style={{ width: "40px", height: "40px", borderRadius: "50%", border: "3px solid rgba(255,255,255,0.1)", borderTopColor: "#f09c06", animation: "spin 1s linear infinite" }}></div>
            <PlaceholderText>Loading data...</PlaceholderText>
            <style>{`
              @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
              }
            `}</style>
          </EmptyState>
        ) : error ? (
          <EmptyState>
            <div style={{ display: "flex", flexDirection: "column", justifyContent: "center", alignItems: "center", height: "300px", width: "100%" }}>
              <img src="/assets/images/bcgames/affiliate/empty.webp" alt="No transactions" style={{ height: "200px", width: "200px", objectFit: "contain" }} />
              <PlaceholderText>Oops! Failed to fetch data!</PlaceholderText>
              <RefreshButton onClick={handleRefresh} style={{ marginTop: "16px" }}>
                Try Again
              </RefreshButton>
            </div>
          </EmptyState>
        ) : transactionData.length === 0 ? (
          <EmptyState>
            <div style={{ display: "flex", flexDirection: "column", justifyContent: "center", alignItems: "center", height: "300px", width: "100%" }}>
              <img src="/assets/images/bcgames/affiliate/empty.webp" alt="No transactions" style={{ height: "200px", width: "200px", objectFit: "contain" }} />
              <PlaceholderText>No {activeTab.toLowerCase()} transactions found</PlaceholderText>
            </div>
          </EmptyState>
        ) : transactionData.filter(transaction => {
            let matches = true;
            if (filters.status) {
              matches = matches && transaction.status.toUpperCase() === filters.status.toUpperCase();
            }
            if (filters.dateFrom) {
              matches = matches && new Date(transaction.date) >= new Date(filters.dateFrom);
            }
            if (filters.dateTo) {
              matches = matches && new Date(transaction.date) <= new Date(filters.dateTo);
            }
            if (filters.type) {
              matches = matches && transaction.type === filters.type;
            }
            return matches;
          }).length === 0 ? (
          <EmptyState>
            <div style={{ display: "flex", flexDirection: "column", justifyContent: "center", alignItems: "center", height: "300px", width: "100%" }}>
              <img src="/assets/images/bcgames/affiliate/empty.webp" alt="No transactions" style={{ height: "200px", width: "200px", objectFit: "contain" }} />
              <PlaceholderText>No {activeTab.toLowerCase()} transactions match the applied filters</PlaceholderText>
              <FilterButton onClick={() => setFilters({ status: '', dateFrom: '', dateTo: '', type: '' })} style={{ marginTop: "16px" }}>
                Clear Filters
              </FilterButton>
            </div>
          </EmptyState>
        ) : (
          <TableContainer>
          <TableHeader>
            <div>Date</div>
            <div>Amount</div>
            <div>Status</div>
            <div>{activeTab === "Deposit" ? "Transaction" : "Details"}</div>
          </TableHeader>
      
          {transactionData
            .filter(transaction => {
              let matches = true;
              if (filters.status) {
                matches = matches && transaction.status.toUpperCase() === filters.status.toUpperCase();
              }
              if (filters.dateFrom) {
                matches = matches && new Date(transaction.date) >= new Date(filters.dateFrom);
              }
              if (filters.dateTo) {
                matches = matches && new Date(transaction.date) <= new Date(filters.dateTo);
              }
              if (filters.type) {
                matches = matches && transaction.type === filters.type;
              }
              return matches;
            })
            .map((transaction, index) => (
              <TableRow key={index}>
                <div data-label="Date">{formatDateTime(transaction.date)}</div>
                <div data-label="Amount">
                  {transaction.amount}
                  {transaction.type && <span style={{ marginLeft: "4px", fontSize: "12px", opacity: "0.7" }}>({transaction.type})</span>}
                </div>
                <div data-label="Status">
                  <StatusIndicator status={transaction.status}>
                    {transaction.status}
                  </StatusIndicator>
                </div>
                <div data-label={activeTab === "Deposit" ? "Transaction" : "Details"} style={{ wordBreak: "break-word" }}>
                  {activeTab === "Deposit" ? transaction.details : 'N/A'}
                </div>
              </TableRow>
            ))}
        </TableContainer>
        )}
      </Container1>
    </Container>
  );
};

export default SkylaTransaction;