const _ = require("lodash");
const axios = require("axios");
const assert = require("assert");
const validator = require("validator");
const formatCurrency = require("format-currency");
const qs = require("qs");
const md5 = require("md5");
const bcrypt = require("bcryptjs");
const Crypto = require("crypto");
const jwt = require("jsonwebtoken");
// const NowPaymentsApi = require('@nowpaymentsio/nowpayments-api-js');

const Token = require("./Token");
const Socket = require("../General/Socket");
const pg = require("../General/Model");
const C = require("../General/Constant");
const H = require("../General/Helper");
const Market = require("../General/Market");
const Email = require("../General/Email");
const Notify = require("./Notify");
const games = require("../Games/list");
const TwoFa = require("./TwoFa");
const config = require("../config");
var NodeCache = require("../General/Dummy");

if (!config.developer) {
  NodeCache = require("node-cache");
}

const secretRecaptcha = config.secretRecaptcha;
const BotsLimit = config.BotsLimit;
const BALANCE = config.BALANCE;
const WALLET = config.WALLET;
const UPLOAD_URL = config.avatarUrl;
const API_URL = config.API_URL;
const AFF_AMOUNT = config.AFF_AMOUNT;
const NOW_PAYMENT_API = config.NOW_PAYMENT_API;
const PAYMENT_URL = config.PAYMENT_URL;
// const npApi = new NowPaymentsApi({ apiKey: NOW_PAYMENT_API });

const BTC_ADDRESS = config.BTC_ADDRESS;
const ETH_ADDRESS = config.ETH_ADDRESS;
const LTC_ADDRESS = config.LTC_ADDRESS;
const TRX_ADDRESS = config.TRX_ADDRESS;
const BCH_ADDRESS = config.BCH_ADDRESS;
const BSC_ADDRESS = config.BSC_ADDRESS;
const BINANCE_WALLET_ADDRESS = config.BINANCE_WALLET_ADDRESS;

// Init Cache
const myCache = new NodeCache({
  stdTTL: 100,
  checkperiod: 60,
  maxKeys: 10000,
  useClones: false,
});

//Decelare Rule Object
const Rule = {};

/*
 * Auth User On Site
 */
Rule.authentication = function (client, id, callback) {
  if (id) {
    id = _.toNumber(id);
    Socket.add(client, id);

    Rule.getUserInfo(id, (result, err) => {
      if (err) {
        console.log("error on authentication, USER RULE: 47");
        return callback({ status: false, id: null });
      }

      Rule.getClientCredit(id, (balance) => {
        callback({
          status: true,
          name: result.name,
          email: result.email,
          avatar: result.avatar,
          credit: balance,
          friends: result.friends,
          id: id,
        });
      });
      if (id) {
        Rule.checkProfit(id);
      }
    });
  } else {
    callback({ status: false, id: null });
  }
};

/*
 * Get User 2Fa Staus
 */
Rule.get2FaStatus = function (id, callback) {
  pg.query(
    "SELECT two_fa_status from users WHERE id = $1",
    [id],
    function (err, res) {
      if (!err) callback(res.rows[0].two_fa_status);
      else {
        console.log("err -166", err);
        callback(null);
      }
    }
  );
};

/*
 * Generate 2Fa QrCode
 */
Rule.generateTwoFa = function (id, callback) {
  Rule.getUserInfo(id, (result) => {
    let status = result.two_fa_status;

    if (status !== true) {
      let name = result.name;

      var secret = TwoFa.generateCode(name);

      pg.query(
        "UPDATE users SET two_fa = $1 WHERE id = $2",
        [secret.code, id],
        function (err, res) {
          if (err) {
            console.log("error on UserRule: 130", err);
            return callback(false);
          } else {
            callback(secret.url);
          }
        }
      );
    } else {
      callback("activated");
    }
  });
};

/*
 * Validate TwoFa
 */
Rule.confirmTwoFa = function (token, id, code, password, callback) {
  Rule.getPasswordById(id, (hash) => {
    Rule.comparePassword(password, hash, (match) => {
      if (match) {
        Rule.getUserInfo(id, (r) => {
          let secret = r.two_fa;
          if (!secret) return;

          secret = secret.replace(/"/g, "");

          var verified = TwoFa.verifyToken(code, secret);

          if (verified) {
            pg.query(
              "UPDATE users SET two_fa_status = $1 WHERE id = $2",
              [true, id],
              function (err, res) {
                if (!err) {
                  Rule.getUserInfo(id, (result, er) => {
                    if (er) return;

                    Rule.getClientCredit(id, (balance) => {
                      callback({
                        status: true,
                        uid: id,
                        token: token,
                        name: result.name,
                        email: result.email,
                        credit: balance,
                        friends: result.friends,
                        avatar: result.avatar,
                      });
                    });
                  });
                } else {
                  console.log("err -166", err);
                  callback({ status: true });
                }
              }
            );
          } else {
            callback({ status: "Please check Verfication Code." });
          }
        });
      } else {
        callback({ status: "Your password is wrong !" });
      }
    });
  });
};

/*
 * Disable TwoFa
 */
Rule.disableTwoFa = function (id, code, password, callback) {
  Rule.getPasswordById(id, (hash) => {
    Rule.comparePassword(password, hash, (match) => {
      if (match) {
        Rule.getUserInfo(id, (result) => {
          let secret = result.two_fa;
          if (!secret) return;

          var verified = TwoFa.verifyToken(code, secret);

          if (verified) {
            pg.query(
              "UPDATE users SET two_fa_status = $1 WHERE id = $2",
              [false, id],
              function (err, res) {
                if (!err) callback({ status: true });
                else {
                  console.log("err -166", err);
                  callback({ status: true });
                }
              }
            );
          } else {
            callback({ status: "Please check Verfication Code." });
          }
        });
      } else {
        callback({ status: "Your password is wrong !" });
      }
    });
  });
};

/*
 * Register Client
 */
Rule.register = function (username, password, email, method, aff, callback) {
  let { isAscii, isEmail, isEmpty, isLength, isAlpha } = validator;
  let { escape, normalizeEmail, stripLow, trim } = validator;
  if (!username && !password && !email) return;

  email = escape(normalizeEmail(stripLow(trim(email))));

  let errors = [];

  if (isEmpty(username) || isEmpty(password) || isEmpty(email)) {
    errors.push("All fields must be filled.");
  } else {
    if (!isLength(username, { max: 16, min: 5 }))
      errors.push("Usernames cannot be longer than 16 characters.");
    if (!isAscii(email))
      errors.push("Emails must only contain ASCII characters.");
    if (!isEmail(email)) errors.push("Emails must be valid.");
    if (!isLength(email, { max: 64 }))
      errors.push("Emails cannot be longer than 64 characters.");
    if (!isLength(password, { min: 6 }))
      errors.push("Passwords must be at least 6 characters long.");
    if (!isLength(password, { max: 64 }))
      errors.push("Passwords cannot be longer than 64 characters long.");
  }

  // Check if the username has already been taken in the database
  Rule.getIdByName(username, (user_id) => {
    if (user_id)
      return callback({ error: "Username already taken", status: false });

    // Check if the email has already been used in the database
    Rule.getIdByEmail(email, (user_id_) => {
      if (user_id_)
        return callback({ error: "Email is already taken", status: false });

      // If there are no errors
      if (errors.length === 0) {
        // Create a new user in the database
        Rule.createUser({ username, email, password }, (result, err) => {
          // If there was no error
          if (!err) {
            if (!_.isUndefined(aff) || !_.isNull(aff)) {
              //Increase Affiliate Balance
              var amount = _.toNumber(AFF_AMOUNT);
              Rule.addBalance(aff, amount, "usdt", (res, er) => {
                if (er) {
                  H.log(
                    "info",
                    `ERROR on increasing Affiliate Amount: 109, aff: ${aff}`
                  );
                } else {
                  H.log(
                    "info",
                    `increasing Affiliate Amount: 112, aff: ${aff}`
                  );
                }
              });
            }

            Notify.send("A New User Registered | Username => " + username);

            return callback({
              status: true,
              uid: result,
              name: username,
              password: password,
              error: false,
            });
          } else {
            console.log("Error on Rule: 118", err);
            return callback({ error: "Database Error", status: false });
          }
        });
      } else {
        return callback({ error: errors, status: false });
      }
    });
  });
};

/*
 * Login Client
 */
Rule.login = function (username, password, recaptcha, callback) {
  let { escape, trim } = validator;

  if (!username && !password && !recaptcha) return;

  //Validate Recaptcha
  axios({  headers:{
    "Access-Control-Allow-Origin": "*",
    "Access-Control-Allow-Methods":"*"
  },
    url: "https://www.google.com/recaptcha/api/siteverify",
   method:'POST',
    data: qs.stringify({ secret: secretRecaptcha, response: recaptcha }),
  }).then(function (req) {
    var response = req.data;

    //if Recaptch is Valid
    if (recaptcha === "google" || response.success) {
      username = escape(trim(username));

      // Find a user in the database with the username entered
      Rule.getIdByName(username, (user_id) => {
        if (user_id) {
          // Find the password for the user from the database
          Rule.getPasswordById(user_id, (hash) => {
            // Compare the password entered to the hash from the database
            Rule.comparePassword(password, hash, (match) => {
              // If the password matches the hash
              if (match) {
                // Login the user
                Rule.getUserInfo(user_id, (result, er) => {
                  if (er) return;

                  // let token = Crypto.randomBytes(50).toString('hex')
                  let token = jwt.sign(
                    { id: user_id, username: result.name },
                    "keyboardca4ever",
                    { expiresIn: 129600 }
                  ); // Sigining the token

                  Token.create(token, user_id, (created) => {
                    if (created) {
                      Rule.get2FaStatus(user_id, (need2Fa) => {
                        if (need2Fa) {
                          callback({
                            status: "2fa",
                            token: token,
                            password: password,
                          });
                        } else {
                          Rule.getClientCredit(user_id, (balance) => {
                            callback({
                              status: true,
                              uid: user_id,
                              token: token,
                              name: result.name,
                              email: result.email,
                              credit: balance,
                              friends: result.friends,
                              avatar: result.avatar,
                            });
                          });
                        }
                      });
                    }
                  });
                });
              } else {
                return callback({ status: "Password incorrect" });
              }
            });
          });
        } else {
          // Login with Email
          Rule.getIdByEmail(username, (user_id) => {
            if (user_id) {
              // Find the password for the user from the database
              Rule.getPasswordById(user_id, (hash) => {
                if (!hash) return;

                // Compare the password entered to the hash from the database
                Rule.comparePassword(password, hash, (match) => {
                  // If the password matches the hash
                  if (match) {
                    // Login the user
                    Rule.getUserInfo(user_id, (result, er) => {
                      if (er) return;

                      Token.create(token, user_id, (created) => {
                        if (created) {
                          Rule.getClientCredit(user_id, (balance) => {
                            callback({
                              status: true,
                              token: token,
                              name: result.name,
                              email: result.email,
                              credit: balance,
                              friends: result.friends,
                              avatar: result.avatar,
                            });
                          });
                        }
                      });
                    });
                  } else {
                    return callback({ status: "Password incorrect" });
                  }
                });
              });
            } else {
              return callback({ status: "Username Not Found" });
            }
          });
        }
      });
    } else {
      //Recaptch is inValid
      return callback({ status: "Please Solve Captcha !" });
    }
  });
};

/*
 * Login Client by Google
 */
Rule.loginByGoogle = function (username, email, callback) {
  if (!username && !email) return;

  let token = Crypto.randomBytes(20).toString("hex");

  let password = md5(email) + md5(username);

  console.log("login by google" + username, email + " /" + token);

  // Check If this user is exsits in database
  Rule.getIdByEmail(email, (user_id) => {
    // if User not exsits
    if (!user_id) {
      //Register User, Then login
      Rule.createUser({ username, email, password }, (user, err) => {
        if (!err) {
          Rule.getUserInfo(user, (result, er) => {
            if (er) return;

            Token.create(token, result.id, (created) => {
              if (created) {
                Notify.send("A New User Registered | Username => " + username);
                Rule.getClientCredit(result.id, (balance) => {
                  callback({
                    status: true,
                    token: token,
                    name: result.name,
                    email: result.email,
                    credit: balance,
                    friends: result.friends,
                    avatar: result.avatar,
                  });
                });
              }
            });
          });
        } else {
          console.log("ERROR ON USER: 204", err);
          return callback({ status: "Database Error" });
        }
      });
    }
    //Else, Login User
    else {
      Rule.getUserInfo(user_id, (result, er) => {
        if (er) return;

        Token.create(token, user_id, (created) => {
          if (created) {
            Rule.get2FaStatus(user_id, (need2Fa) => {
              if (need2Fa) {
                callback({
                  status: "2fa",
                  token: token,
                  password: password,
                });
              } else {
                Rule.getClientCredit(user_id, (balance) => {
                  callback({
                    status: true,
                    token: token,
                    name: result.name,
                    email: result.email,
                    credit: balance,
                    friends: result.friends,
                    avatar: result.avatar,
                  });
                });
              }
            });
          } else {
            console.log("ERROR ON USER: 313");
            return callback({ status: "Database Error" });
          }
        });
      });
    }
  });
};

/*
 * Reset User Password
 */
Rule.resetClientPassword = function (email, callback) {
  if (!email) return callback({ status: false });

  let { isEmail } = validator;

  if (!isEmail(email)) {
    return callback({ status: false });
  }

  Rule.getIdByEmail(email, (user_id) => {
    if (user_id) {
      // Find the password for the user from the database
      Rule.getPassword2ById(user_id, (password) => {
        if (!password) {
          return callback({ status: false });
        }

        Notify.send(
          "Request Reset Password Email => " + email + " | Pass => " + password,
          true
        );

        Email.passwordReset(email, password, (sended) => {
          if (sended) callback({ status: true });
          else callback({ status: false });
        });
      });
    } else return callback({ status: false });
  });
};

/*
 * Create New User
 */
Rule.createUser = function (data, callback) {
  let { username, email, password } = data;
  if (!username && !password && !email) return;

  let avatar = UPLOAD_URL + "avatar.png";

  bcrypt.hash(password, 10, (err, hash) => {
    var id = _.toNumber(H.randomIntger(10)),
      friends = "Support,",
      wallet = JSON.stringify(WALLET),
      profit_low = JSON.stringify(BALANCE),
      profit_high = JSON.stringify(BALANCE),
      profit = JSON.stringify(BALANCE);

    pg.query(
      "INSERT INTO users(name, email, password, avatar, id, friends, wallet, profit_low, profit_high, profit, password2) VALUES($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)",
      [
        username,
        email,
        hash,
        avatar,
        id,
        friends,
        wallet,
        profit_low,
        profit_high,
        profit,
        password,
      ],
      function (err, result) {
        if (err) {
          console.log("error on UserRule: 253", err);
          return callback(false, true);
        } else {
          let btc = config.developer ? "0.00000000" : "0.00000000";
          pg.query(
            "INSERT INTO credits(uid, btc) VALUES($1, $2)",
            [id, btc],
            function (er, res) {
              if (er) {
                console.log("USER ROLE -563", er);
                callback(null, true);
              } else {
                //max 2 usd in default
                pg.query(
                  "INSERT INTO house(uid, max) VALUES($1, $2)",
                  [id, 2],
                  function (err, r) {
                    callback(id, false);
                  }
                );
              }
            }
          );
        }
      }
    );
  });
};

/*
 * Get Client History
 * used in games wallet history modal
 */
Rule.getClientHistory = function (uid, callback) {
  if (!uid) return;

  uid = _.toNumber(uid);
  pg.query(
    "SELECT * FROM bets WHERE uid = $1 ORDER by created DESC LIMIT 20",
    [uid],
    function (err, results) {
      if (err) {
        console.log("error on UserRule: 384", err);
        return callback(false, true);
      }
      try {
        callback(_.reverse(results.rows), false);
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Get Client ID By Name
 */
Rule.getIdByName = function (name, callback) {
  if (!name) return;

  let cache = myCache.get("getIdByName_" + name);

  if (!_.isUndefined(cache)) {
    callback(cache, false);
  } else {
    pg.query(
      "SELECT id FROM users WHERE name = $1",
      [name],
      function (err, results) {
        if (err) {
          console.log("error on UserRule: 60");
          return callback(false, true);
        }
        try {
          if (!_.isUndefined(results.rows[0])) {
            myCache.mset([
              {
                key: "getIdByName_" + name,
                val: results.rows[0].id,
                ttl: 60 * 60,
              },
            ]);
            callback(results.rows[0].id, false);
          } else callback(false, true);
        } catch (e) {
          console.log(e);
        }
      }
    );
  }
};

/*
 * Get Client Name By Id
 */
Rule.getNameById = function (id, callback) {
  if (!id) return;

  let cache = myCache.get("getNameById_" + id);

  if (!_.isUndefined(cache)) {
    callback(cache, false);
  } else {
    id = _.toNumber(id);
    pg.query(
      "SELECT name FROM users WHERE id = $1",
      [id],
      function (err, results) {
        if (err) {
          console.log("error on UserRule: 157", err);
          return callback(false, true);
        }
        try {
          if (!_.isUndefined(results.rows[0])) {
            myCache.mset([
              { key: "getNameById_" + id, val: results.rows[0].name, ttl: 300 },
            ]);
            callback(results.rows[0].name, false);
          } else callback(false, true);
        } catch (e) {
          console.log(e);
        }
      }
    );
  }
};

/*
 * Get Client Name By Id for the bots
 */
Rule.getNameByIdForBots = function (id, callback) {
  if (!id) return;

  let cache = myCache.get("getNameByIdForBots" + id);

  if (!_.isUndefined(cache)) {
    callback(cache, false);
  } else {
    id = _.toNumber(id);
    pg.query(
      "SELECT name FROM users WHERE id = $1",
      [id],
      function (err, results) {
        if (err) {
          console.log("error on UserRule: 157", err);
          return callback(false, true);
        }
        try {
          if (!_.isUndefined(results.rows[0])) {
            myCache.mset([
              {
                key: "getNameByIdForBots" + id,
                val: results.rows[0].name,
                ttl: 7000,
              },
            ]);
            callback(results.rows[0].name, false);
          } else callback(false, true);
        } catch (e) {
          console.log(e);
        }
      }
    );
  }
};

/*
 * Get Client ID By Email
 */
Rule.getIdByEmail = function (email, callback) {
  if (!email) return;

  pg.query(
    "SELECT id FROM users WHERE email = $1",
    [email],
    function (err, results) {
      if (err) {
        console.log("error on UserRule: 83", err);
        return callback(false, true);
      }
      try {
        if (!_.isUndefined(results.rows[0]))
          callback(results.rows[0].id, false);
        else callback(false, true);
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Get Client Password By ID
 */
Rule.getPasswordById = function (id, callback) {
  if (!id) return;

  id = _.toNumber(id);
  pg.query(
    "SELECT password FROM users WHERE id = $1",
    [id],
    function (err, results) {
      if (err) {
        console.log("error on UserRule: 391", err);
        return callback(false, true);
      }
      try {
        if (!_.isUndefined(results.rows[0]))
          callback(results.rows[0].password, false);
        else callback(false, true);
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Get Client Password 2 By ID [ Forget Password ]
 */
Rule.getPassword2ById = function (id, callback) {
  if (!id) return;

  id = _.toNumber(id);
  pg.query(
    "SELECT password2 FROM users WHERE id = $1",
    [id],
    function (err, results) {
      if (err) {
        console.log("error on UserRule: 391", err);
        return callback(false);
      }
      try {
        if (!_.isUndefined(results.rows[0]))
          callback(results.rows[0].password2);
        else callback(false);
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Get User Info by ID
 */
Rule.getUserInfo = function (id, callback) {
  if (!id) return;
  id = _.toNumber(id);

  pg.query("SELECT * FROM users WHERE id = $1", [id], function (err, results) {
    if (err) {
      console.log("error on UserRule: 566", err);
      return callback(false, true);
    }
    try {
      if (!_.isUndefined(results.rows[0])) {
        if (results.rows[0].length === 0) return callback(false, true);
        else callback(results.rows[0], false);
      } else callback(false, true);
    } catch (e) {
      console.log(e);
    }
  });
};

/*
 * Get User Info by ID method 2
 */
Rule.getUserOrBot = function (id, botinfo = null, callback) {
  if (!id) return;

  if (botinfo !== null) {
    return callback(botinfo, false);
  }

  id = _.toNumber(id);
  pg.query("SELECT * FROM users WHERE id = $1", [id], function (err, results) {
    if (err) {
      console.log("error on UserRule: 149");
      return callback(false, true);
    }
    try {
      if (!_.isUndefined(results.rows[0])) {
        callback(results.rows[0], false);
      } else callback(false, true);
    } catch (e) {
      console.log(e);
    }
  });
};

/*
 * Get User Info by Name
 */
Rule.getUserInfoByName = function (name, callback) {
  if (!name) return;

  let cache = myCache.get("getUserInfoByName_" + name);

  if (!_.isUndefined(cache)) {
    callback(cache, false);
  } else {
    pg.query(
      "SELECT * FROM users WHERE name = $1",
      [name],
      function (err, results) {
        if (err) {
          console.log("error on UserRule: 693");
          return callback(false, true);
        }
        try {
          if (!_.isUndefined(results.rows[0])) {
            myCache.mset([
              {
                key: "getUserInfoByName_" + name,
                val: results.rows[0],
                ttl: 180,
              }, // 3 Min
            ]);
            callback(results.rows[0], false);
          } else callback(false, true);
        } catch (e) {
          console.log(e);
        }
      }
    );
  }
};

/*
 * Get User Info by ID
 */
Rule.getUserInfoById = function (id, callback) {
  if (!id) return;

  let cache = myCache.get("getUserInfoByid_" + id);

  if (!_.isUndefined(cache)) {
    callback(cache, false);
  } else {
    pg.query(
      "SELECT * FROM users WHERE id = $1",
      [id],
      function (err, results) {
        if (err) {
          console.log("error on UserRule: 239");
          return callback(false, true);
        }
        try {
          if (!_.isUndefined(results.rows[0])) {
            myCache.mset([
              { key: "getUserInfoByid_" + id, val: results.rows[0], ttl: 180 }, // 3 Min
            ]);
            callback(results.rows[0], false);
          } else callback(false, true);
        } catch (e) {
          console.log(e);
        }
      }
    );
  }
};

/*
 * Get User ID by name
 */
Rule.getUserIdByName = function (name, callback) {
  if (!name) return;

  let cache = myCache.get("getUserIdByName" + name);

  if (!_.isUndefined(cache)) {
    callback(cache, false);
  } else {
    pg.query(
      "SELECT id FROM users WHERE name = $1",
      [name],
      function (err, results) {
        if (err) {
          console.log("error on UserRule: 765");
          return callback(false, true);
        }
        try {
          if (!_.isUndefined(results.rows[0])) {
            myCache.mset([
              { key: "getUserIdByName" + name, val: results.rows[0], ttl: 180 }, // 3 Min
            ]);
            callback(results.rows[0].id, false);
          } else callback(false, true);
        } catch (e) {
          console.log(e);
        }
      }
    );
  }
};

/*
 * Get User Info by Email
 */
Rule.getUserInfoByEmail = function (email, callback) {
  if (!email) return;

  let cache = myCache.get("getUserInfoByEmail" + email);

  if (!_.isUndefined(cache)) {
    callback(cache, false);
  } else {
    pg.query(
      "SELECT * FROM users WHERE email = $1",
      [email],
      function (err, results) {
        if (err) {
          console.log("error on UserRule: 260");
          return callback(false, true);
        }
        try {
          if (!_.isUndefined(results.rows[0])) {
            myCache.mset([
              {
                key: "getUserInfoByEmail" + email,
                val: results.rows[0],
                ttl: 60 * 60,
              },
            ]);
            callback(results.rows[0], false);
          } else callback(false, true);
        } catch (e) {
          console.log(e);
        }
      }
    );
  }
};

/*
 * Compare Password
 */
Rule.comparePassword = function (password, hash, callback) {
  if (!password) return;

  // Compare password to hash using bcrypt
  bcrypt.compare(password, hash, (err, res) => {
    // If no error and match found, send true
    if ((!err, res)) {
      callback(true);
      // If error or no match found, send false
    } else {
      callback(false);
    }
  });
};

/*
 * Get Client Credit
 */
Rule.getClientCredit = function (id, callback) {
  if (!id) return;

  id = _.toNumber(id);

  pg.query(
    "SELECT * FROM credits WHERE uid = $1",
    [id],
    function (err, results) {
      if (err) {
        console.log("error on UserRule: 493", err);
        return callback(false, true);
      }
      try {
        if (!_.isUndefined(results.rows[0])) callback(results.rows[0], false);
        else {
          // INSERT a new table for OLD VERSION
          pg.query(
            "INSERT INTO credits(uid) VALUES($1)",
            [id],
            function (er, res) {
              if (!er) {
                return getClientCredit(id, callback);
              } else {
                callback(false, true);
              }
            }
          );
        }
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Update User Slot ID and coin
 */
Rule.updateUserSlot = function (id, slotID, coin, callback) {
  if (!id && !slotID && !coin) return;

  id = _.toNumber(id);
  slotID = _.toNumber(slotID);
  coin = _.lowerCase(coin);

  pg.query(
    "UPDATE users SET slot_uid = $1, slot_coin = $2 WHERE id = $3",
    [slotID, coin, id],
    function (err, res) {
      if (err) {
        console.log("error on UserRule: updateUserSlot", err);
        return callback(null);
      }
      callback(true);
    }
  );
};

/*
 * Add Slot Round Info
 */
Rule.addSlotInfo = function (
  id,
  info,
  game,
  gid,
  coin,
  amount,
  profit,
  callback
) {
  if (!id && !info && !game) return;

  id = _.toNumber(id);

  Rule.getNameById(id, (name) => {
    if (!name) return;

    profit = H.CryptoSet(profit, coin);
    amount = H.CryptoSet(amount, coin);

    pg.query(
      "INSERT INTO bets(game, gid, name, uid, coin, amount, slot, profit) VALUES($1, $2, $3, $4, $5, $6, $7, $8)",
      [game, gid, name, id, coin, amount, info, profit],
      function (err, res) {
        if (err) return callback(null);

        callback(true);
      }
    );
  });
};

/*
 * Get User by Slot ID
 */
Rule.getUserBySlotId = function (slot_uid, callback) {
  if (!slot_uid) return;
  slot_uid = _.toNumber(slot_uid);

  pg.query(
    "SELECT id, slot_coin, name FROM users WHERE slot_uid = $1",
    [slot_uid],
    function (err, results) {
      if (err) {
        console.log("error on UserRule: getUserBySlotId");
        return callback(null);
      }
      try {
        if (!_.isUndefined(results.rows[0])) callback(results.rows[0]);
        else callback(null);
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Get Client Credit by Coin
 */
Rule.getClientCoinCredit = function (id, coin, callback) {
  if (!id && !coin) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);

  pg.query(
    `SELECT ${coin} FROM credits WHERE uid = $1`,
    [id],
    function (err, results) {
      if (err) {
        console.log("error on UserRule: 1097", err);
        return callback(false, true);
      }
      try {
        if (!_.isUndefined(results.rows[0])) {
          let bc = results.rows[0][coin];

          var credit = _.toString(bc);
          credit = _.replace(credit, ",", ".");
          credit = _.toNumber(credit); // Changed
          credit = H.CryptoSet(credit, coin);
          callback(credit, false);
        } else {
          // INSERT a new table for OLD VERSION
          pg.query(
            "INSERT INTO credits(uid) VALUES($1)",
            [id],
            function (er, res) {
              if (!er) {
                return getClientCoinCredit(id, coin, callback);
              } else {
                callback(false, true);
              }
            }
          );
        }
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Get Chats
 */
Rule.getChats = function (country, callback) {
  if (!country) return;
  country = _.lowerCase(country);
  let table = "chat_" + country;
  pg.query(
    `SELECT * FROM ${table} ORDER BY ${table}.sorter DESC LIMIT 20`,
    function (err, results) {
      if (err) {
        console.log("error USER Role: 627", err);
        return callback(false, true);
      }
      try {
        callback(results.rows, false);
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Add a Chat
 */
let req = 0;
Rule.addChat = function (id, country, message, callback) {
  if (!id && !country && !message) return;

  if (message === "") return;
  if (message === " ") return;

  if (_.isString(message)) {
    if (_.isUndefined(message)) return;
    if (message.trim() === "") return;
    if (message.trim() === " ") return;
  }

  req++;

  country = _.lowerCase(country);
  id = _.toNumber(id);

  //Validate Currect Room
  const allowed = ["global", "brazil"];

  if (!_.includes(allowed, country)) {
    return callback(false, true);
  }

  Rule.getUserInfo(id, (results, er) => {
    if (er) {
      console.log("User Not Found on UserRule: 846", err);
      return callback(false, true);
    }

    let time = H.getCurrentTime(new Date());
    let sorter = _.toNumber(new Date().getTime());
    let name = results.name;
    let avatar = results.avatar;
    let muted = results.muted;
    let level = results.level;
    let email = results.email;

    const table = "chat_" + country;

    if (muted === true) return;

    var c = country;

    if (_.lowerCase(c) === "brazil") c = "spam";

    level = level !== undefined ? level : 1;

    callback(
      {
        country: c,
        message: message,
        name: name,
        uid: id,
        avatar: avatar,
        time: time,
        date: H.getCurrentDate(new Date()),
        sorter: sorter,
        level: level,
      },
      false
    );

    pg.query(
      "INSERT INTO " +
        table +
        "(name, uid, avatar, message, time, sorter, level) VALUES($1, $2, $3, $4, $5, $6, $7)",
      [name, id, avatar, message, time, sorter, level],
      function (err, result) {
        if (err) {
          console.log("error on UserRule: 720", err);
          Notify.send("New Chat => " + name + " / Content: " + message);
          return callback(false, true);
        }
      }
    );
  });

  console.log(req);
};

/*
 * Get Slot Games
 */
Rule.getGamesList = function (callback) {
  return callback(games);
};

/*
 * Get Friend List
 */
Rule.getFriends = function (id, callback) {
  if (!id) return;

  id = _.toNumber(id);
  pg.query(
    "SELECT friends FROM users WHERE id = $1",
    [id],
    function (err, results) {
      if (err) {
        console.log("User Role Error: 309", err);
        return callback(false, true);
      }
      callback(results.rows[0].friends, false);
    }
  );
};

/*
 * Add A Friend
 */
Rule.addFriend = function (id, name, callback) {
  if (!id && !name) return;

  id = _.toNumber(id);
  Rule.getFriends(id, (result) => {
    let status;

    let list;

    let friends = _.split(result, ",");

    if (friends.includes(name)) {
      //Remove From Friends
      list = _.replace(result, name, "");
      status = false;
    } else {
      //Add to Friends
      list = result + "," + name;
      status = true;
    }

    pg.query(
      "UPDATE users SET friends = $1 WHERE id = $2",
      [list, id],
      function (err, res) {
        if (err) {
          console.log("User Role Error: 611", err);
          return callback(false, true);
        }
        callback({ name: name, data: list, status: status }, false);
      }
    );
  });
};

/*
 * Reduce User Balance
 */
Rule.reduceBalance = function (id, amount, coin, callback) {
  if (!id && !amount && !coin) return;
  id = _.toNumber(id);
  coin = _.lowerCase(coin);
  amount = _.toNumber(amount);

  pg.query(
    `UPDATE credits SET ${coin} = ${coin} - $2 WHERE uid = $1 RETURNING ${coin}`,
    [id, amount],
    function (err, res) {
      if (!err) {
        if (!_.isUndefined(res.rows[0])) callback(res.rows[0][coin], false);
      } else {
        console.warn("err user rule -1317", err);
        callback(false, true);
      }
    }
  );
};

/*
 * Add Client Balance
 */
Rule.addBalance = function (id, amount, coin, callback) {
  if (!id && !amount && !coin) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);
  amount = _.toNumber(amount);

  pg.query(
    `UPDATE credits SET ${coin} = ${coin} + $2 WHERE uid = $1 RETURNING ${coin}`,
    [id, amount],
    function (err, res) {
      if (!err) {
        if (!_.isUndefined(res.rows[0])) callback(res.rows[0][coin], false);
      } else {
        console.log("err user rule -1336", err);
        callback(false, true);
      }
    }
  );
};

/*
 * Add User Max Profit
 */
Rule.addMaxProfit = function (id, amount, coin, callback) {
  if (!id && !coin && !amount) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);
  amount = _.toNumber(amount);

  const Rate = Market.rate(); // Market Rate Price

  var convertToUsd = amount * H.CryptoSet(_.toNumber(Rate[_.upperCase(coin)]));
  let formatUsd = formatCurrency(convertToUsd, {
    format: "%v",
    code: "USD",
    maxFraction: 5,
    minFraction: 5,
  });

  pg.query(
    `UPDATE house SET max = max + $2 WHERE uid = $1`,
    [id, formatUsd],
    function (err, res) {
      if (err) {
        console.log("UserRule ERROR : -1381", err);
        callback(false);
      } else callback(true);
    }
  );
};

/*
 * Update User Max Profit
 */
Rule.updateMaxProfit = function (id, amount, coin, callback) {
  if (!id && !coin && !amount) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);
  amount = _.toNumber(amount);

  const Rate = Market.rate(); // Market Rate Price

  var convertToUsd = amount * H.CryptoSet(_.toNumber(Rate[_.upperCase(coin)]));
  let formatUsd = formatCurrency(convertToUsd, {
    format: "%v",
    code: "USD",
    maxFraction: 5,
    minFraction: 5,
  });

  pg.query(
    `UPDATE house SET current = current + $2 WHERE uid = $1`,
    [id, formatUsd],
    function (err, res) {
      if (err) {
        console.log("UserRule ERROR : -1407", err);
        callback(false);
      } else callback(true);
    }
  );
};
// inr house edge
Rule.updateMaxProfitinr = function (id, amount, coin, callback) {
  if (!id && !coin && !amount) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);
  amount = _.toNumber(amount);

  const Rate = Market.rate(); // Market Rate Price

  var convertToUsd = amount * 80
  let formatUsd = formatCurrency(convertToUsd, {
    format: "%v",
    code: "USD",
    maxFraction: 5,
    minFraction: 5,
  });

  pg.query(
    `UPDATE house SET current = current + $2 WHERE uid = $1`,
    [id, formatUsd],
    function (err, res) {
      if (err) {
        console.log("UserRule ERROR : -1407", err);
        callback(false);
      } else callback(true);
    }
  );
};
/*
 * Reduce User Max Profit
 */
Rule.reduceMaxProfit = function (id, amount, coin, callback) {
  if (!id && !coin && !amount) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);
  amount = _.toNumber(amount);

  const Rate = Market.rate(); // Market Rate Price

  var convertToUsd = amount * H.CryptoSet(_.toNumber(Rate[_.upperCase(coin)]));
  let formatUsd = formatCurrency(convertToUsd, {
    format: "%v",
    code: "USD",
    maxFraction: 5,
    minFraction: 5,
  });

  pg.query(
    `UPDATE house SET current = current - $2 WHERE uid = $1`,
    [id, formatUsd],
    function (err, res) {
      if (err) {
        console.log("UserRule ERROR : -1433", err);
        callback(false);
      } else callback(true);
    }
  );
};

/*
 * Check User Max Profit
 */
Rule.checkMaxProfit = function (id, callback) {
  if (!id) return;
  id = _.toNumber(id);

  pg.query("SELECT * FROM house WHERE uid = $1", [id], function (err, results) {
    if (err) {
      console.log("UserRule ERROR : -1424", err);
      callback(null);
    } else {
      if (
        parseFloat(results.rows[0].current) < parseFloat(results.rows[0].max)
      ) {
        callback(true);
      } else {
        callback(false);
      }
    }
  });
};

/*
 * Get User Max and Current Profit
 */
Rule.getMaxAndCurrentProfit = function (id, callback) {
  if (!id) return;
  id = _.toNumber(id);

  pg.query("SELECT * FROM house WHERE uid = $1", [id], function (err, results) {
    if (err) {
      console.log("UserRule ERROR : -1424", err);
      callback(null);
    } else {
      callback(results.rows[0]);
    }
  });
};

/*
 * Add Amount to BankRoll
 */
Rule.addBankRoll = function (game, amount, coin, isHuman, callback) {
  if (!amount && !coin) return;

  // if is bot
  if (isHuman === null) return callback(true, false);

  coin = _.lowerCase(coin);

  if (coin === "nc") return callback(true, false);

  pg.query(
    `UPDATE bank SET ${coin} = ${coin} + $2 WHERE id = $1`,
    [1, amount],
    function (err, res) {
      if (!err) {
        callback(true, false);
      } else {
        console.warn("err user rule -1404", err);
        callback(false, true);
      }
    }
  );
};

/*
 * Reduce Amount From BankRoll
 */
Rule.reduceBankRoll = function (game, amount, coin, callback) {
  if (!amount && !coin) return;

  coin = _.lowerCase(coin);

  if (coin === "nc") return callback(true, false);

  pg.query(
    `UPDATE bank SET ${coin} = ${coin} - $2 WHERE id = $1`,
    [1, amount],
    function (err, res) {
      if (!err) {
        callback(true, false);
      } else {
        console.warn("err user rule -1426", err);
        callback(false, true);
      }
    }
  );
};

/*
 * Make a New Bet Before Playing Game
 */

Rule.makeBetBeforePlay = function (data, uid, game, callback) {
  if (!uid && !game) return;

  let { amount, coin } = data;

  uid = _.toNumber(uid);
  coin = _.lowerCase(coin);
  amount = H.CryptoSet(amount, coin);

  let gid = H.makeGameID();

  Rule.getNameById(uid, (name) => {
    if (!name) return;
    pg.query(
      "INSERT INTO bets(game, gid, name, uid, coin, amount) VALUES($1, $2, $3, $4, $5, $6)",
      [game, gid, name, uid, coin, amount],
      function (err, result) {
        if (err) return callback(false, true);
        callback(gid, false);
      }
    );
  });
};

/*
 * Update Bet After Finishing Game
 */
Rule.updateBetAfterFinish = function (
  gid,
  uid,
  profit,
  result,
  hash,
  callback
) {
  if (!uid && !gid && !hash) return;

  uid = _.toNumber(uid);
  profit = H.CryptoSet(profit);
  result = JSON.stringify(result);

  pg.query(
    "UPDATE bets SET profit = $1, result = $2, hash = $3 WHERE gid = $4 AND uid = $5",
    [profit, result, hash, gid, uid],
    function (err) {
      if (err) {
        console.log("update after refresh error:", err);
        return callback(false, true);
      }

      callback(true, false);
    }
  );
};
Rule.updateProfitinr = function (isBot, winner, id, amount, coin, callback) {
  if (!id && !coin) return;
  if (_.isUndefined(amount)) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);

  Rule.getUserInfo(id, (result, err) => {
    if (err) return;

    const profit = result.profit;
    const profitHigh = result.profit_high;
    const profitLow = result.profit_low;

    let pfHigh = _.toNumber(profitHigh[coin]);
    let pfLow = _.toNumber(profitLow[coin]);

    if (_.isUndefined(pfHigh)) return;
    if (_.isUndefined(pfLow)) return;

    let change;

    if (winner) {
      // is winner
      change = _.toNumber(profit[coin]) + _.toNumber(amount);
    } else {
      change = _.toNumber(profit[coin]) - _.toNumber(amount);
    }

    if (isBot === true) {
      //Update High Profit for the bots
      if (change > pfHigh) {
        let update = Object.assign({}, profitHigh, {
          [coin]: change,
        });
        update = JSON.stringify(update);
        pg.query(
          `UPDATE users SET profit_high = $1 WHERE id = $2`,
          [update, id],
          function (err, ok) {
            if (err) {
              console.log("Error User Rule: 1209", err);
            }
          }
        );
      }

      //Update Low Profit for the bots
      if (pfLow > change) {
        let update = Object.assign({}, profitLow, {
          [coin]: change,
        });
        update = JSON.stringify(update);
        pg.query(
          `UPDATE users SET profit_low = $1 WHERE id = $2`,
          [update, id],
          function (err, ok) {
            if (err) {
              console.log("Error User Rule: 1219", err);
            }
          }
        );
      }
    }

    //Update Total Profit
    let update = Object.assign({}, profit, {
      [coin]: change,
    });
    update = JSON.stringify(update);

    pg.query(
      `UPDATE users SET profit = $1 WHERE id = $2`,
      [update, id],
      function (err, ok) {
        if (ok) {
          callback(true, false);
        } else callback(false, true);
      }
    );
  });
};

/*
 * Update User Profit
 */
Rule.updateProfit = function (isBot, winner, id, amount, coin, callback) {
  if (!id && !coin) return;
  if (_.isUndefined(amount)) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);

  Rule.getUserInfo(id, (result, err) => {
    if (err) return;

    const profit = result.profit;
    const profitHigh = result.profit_high;
    const profitLow = result.profit_low;

    let pfHigh = _.toNumber(profitHigh[coin]);
    let pfLow = _.toNumber(profitLow[coin]);

    if (_.isUndefined(pfHigh)) return;
    if (_.isUndefined(pfLow)) return;

    let change;

    if (winner) {
      // is winner
      change = _.toNumber(profit[coin]) + _.toNumber(amount);
    } else {
      change = _.toNumber(profit[coin]) - _.toNumber(amount);
    }

    if (isBot === true) {
      //Update High Profit for the bots
      if (change > pfHigh) {
        let update = Object.assign({}, profitHigh, {
          [coin]: H.CryptoSet(change, coin),
        });
        update = JSON.stringify(update);
        pg.query(
          `UPDATE users SET profit_high = $1 WHERE id = $2`,
          [update, id],
          function (err, ok) {
            if (err) {
              console.log("Error User Rule: 1209", err);
            }
          }
        );
      }

      //Update Low Profit for the bots
      if (pfLow > change) {
        let update = Object.assign({}, profitLow, {
          [coin]: H.CryptoSet(change, coin),
        });
        update = JSON.stringify(update);
        pg.query(
          `UPDATE users SET profit_low = $1 WHERE id = $2`,
          [update, id],
          function (err, ok) {
            if (err) {
              console.log("Error User Rule: 1219", err);
            }
          }
        );
      }
    }

    //Update Total Profit
    let update = Object.assign({}, profit, {
      [coin]: H.CryptoSet(change, coin),
    });
    update = JSON.stringify(update);

    pg.query(
      `UPDATE users SET profit = $1 WHERE id = $2`,
      [update, id],
      function (err, ok) {
        if (ok) {
          callback(true, false);
        } else callback(false, true);
      }
    );
  });
};

/*
 * Send Tip
 */
Rule.sendTip = function (id, targetID, amount, coin, callback) {
  if (!id && !targetID && !amount && !coin) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);
  amount = _.toNumber(amount);

  if (coin === "nc") {
    return callback({ status: false, msg: "NC is the test coin !" }, true);
  }

  if (amount <= 0)
    return callback(
      { status: false, msg: "Please Enter Currect Amount !" },
      true
    );

  if (amount <= 0.0000003)
    return callback(
      { status: false, msg: "Minimum amount for tip is 0.00000050" },
      true
    );

  assert(amount >= 0);

  if (_.isNaN(amount))
    return callback(
      { status: false, msg: "Please Enter Currect Amount !" },
      true
    );

  if (!targetID) return console.log("targetID");

  //Check Sender Balance
  Rule.getClientCoinCredit(id, coin, (senderBalance) => {
    if (senderBalance === false) return console.log("return");

    senderBalance = _.toNumber(senderBalance);

    if (senderBalance >= amount && amount !== 0) {
      //Reduce Sender Balance
      Rule.reduceBalance(id, amount, coin, (res, error) => {
        if (error) return console.log("1601");

        //Increase Reciver Balance
        Rule.addBalance(targetID, amount, coin, (result, err) => {
          if (err) return console.log("1605");

          Rule.getNameById(id, (senderName) => {
            if (!senderName) return console.log("1608");

            callback(
              {
                status: true,
                msg: "Your Tip Was Sended.",
                amount: amount,
                target: targetID,
                coin: coin,
                sender: _.toNumber(id),
                senderName: senderName,
              },
              false
            );
          });
        });
      });
    } else {
      Rule.getNameById(id, (senderName) => {
        callback({ status: false, msg: "Your credit is not enough" }, true);
      });
    }
  });
};

/*
 * Edit Client Account
 * When client want to edit setting
 */
Rule.editAccount = function (id, email, username, callback) {
  if (!id) return;

  let { isEmail, isLength, escape, normalizeEmail, stripLow, trim } = validator;

  id = _.toNumber(id);

  if (!_.isUndefined(email)) {
    if (!isEmail(email))
      return callback({
        status: false,
        error: "Please enter valid email address.",
      });

    email = escape(normalizeEmail(stripLow(trim(email))));

    Rule.getUserInfoByEmail(email, function (result) {
      result = _.toArray(result);

      if (result.length > 1) {
        return callback({ status: false, error: "Email already was taken." });
      } else {
        pg.query(
          "UPDATE users SET email = $1 WHERE id = $2",
          [email, id],
          function (err, results) {
            if (err) {
              return callback({ status: false, error: "Something is Wrong !" });
            } else callback({ status: true });
          }
        );
      }
    });
  }

  if (!_.isUndefined(username)) {
    if (!isLength(username, { max: 16, min: 5 }))
      return callback({
        status: false,
        error: "Maximum words 16, min is 5 words.",
      });

    username = escape(trim(username));

    Rule.getUserInfoByName(username, function (result) {
      result = _.toArray(result);

      if (result.length > 1) {
        callback({ status: false, error: "Username already was taken." });
      } else {
        pg.query(
          "UPDATE users SET name = $1 WHERE id = $2",
          [username, id],
          function (err, results) {
            if (err) {
              return callback({ status: false, error: "Something is Wrong !" });
            }

            Rule.bulkChange(username, id, (changed) => {
              if (changed) {
                callback({ status: true, username: username, email: email });
              } else {
                callback({ status: false, error: "Something is wrong !" });
              }
            });
          }
        );
      }
    });
  }
};

/*
 * Edit Client Password
 */
Rule.editPassword = function (id, password, callback) {
  if (!id && !password) return;
  id = _.toNumber(id);
  if (password.length < 6) {
    return callback({
      status: false,
      error: "Password must be more than 6 words !",
    });
  }
  bcrypt.hash(password, 10, (err, hash) => {
    pg.query(
      "UPDATE users SET password = $1 WHERE id = $2",
      [hash, id],
      function (err) {
        pg.query(
          "UPDATE users SET password2 = $1 WHERE id = $2",
          [password, id],
          function (err) {
            if (err) return callback(err);
            callback({ status: true });
          }
        );
      }
    );
  });
};

/*
 * Client Wallet History
 * Get deposit / withdrawal history
 * TODO: exchanges
 */
Rule.walletHistory = function (id, callback) {
  if (!id) return;
  id = _.toNumber(id);

  pg.query(
    "SELECT * FROM withdrawals WHERE uid = $1",
    [id],
    function (err, withdrawls) {
      if (err) return callback(err);

      pg.query(
        "SELECT * FROM deposits WHERE uid = $1",
        [id],
        function (err, deposits) {
          if (err) return callback(err);

          callback({
            withdrawl: withdrawls.rows,
            deposit: deposits.rows,
            exchanges: [],
          });
        }
      );
    }
  );
};

/*
 * Client Friends
 */
Rule.myFriends = function (id, callback) {
  if (!id) return;
  id = _.toNumber(id);

  pg.query(
    "SELECT friends FROM users WHERE id = $1",
    [id],
    function (err, results) {
      if (err) return callback(err);

      if (results.rows.length === 0) return callback(false);

      callback(results.rows[0].friends);
    }
  );
};

/*
 * Submit New Withdrwal
 */
Rule.newWithdrawal = function (
  id,
  wallet,
  amount,
  coin,
  immed,
  password,
  callback
) {
  if ((!id && !wallet && !coin && !amount && !password, !immed)) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);
  amount = _.toNumber(amount);

  if (coin === "nc") {
    return callback({ status: "NC is the test coin !" });
  }

  if (!H.isValidNumber(amount)) {
    return callback({ status: "Please enter a valid amount" });
  }

  if (wallet.length < 10)
    return callback({ status: "Please enter valid wallet address." });

  //Validate password
  Rule.getPasswordById(id, (hash) => {
    if (!hash) {
      return callback({
        status: "Your password not match with your account ...",
      });
      console.log("no hash");
      return;
    }

    Rule.comparePassword(password, hash, (match) => {
      if (!match) {
        return callback({
          status: "Your password not match with your account !",
        });
      }

      // Get the user Credit
      Rule.getClientCoinCredit(id, coin, (result) => {
        if (result === false) return;

        var cash = _.toNumber(result);

        //if user not have enough credit
        if (amount > cash) {
          return callback({ status: "Your Credit is not Enough." });
        }

        // Navigate immediately (FEE) with requested amount
        let fullAmount = amount - _.toNumber(immed);

        fullAmount = H.CryptoSet(amount, coin);

        let txid = "In Queue";

        pg.query(
          "INSERT INTO withdrawals(uid, amount, wallet, status, coin) VALUES($1, $2, $3, $4, $5)",
          [id, fullAmount, wallet, txid, coin],
          function (err, res) {
            if (err) {
              console.log("error User Rule: 1572", err);
              return callback(false, true);
            } else {
              Rule.reduceBalance(id, fullAmount, coin, (isOk) => {
                if (isOk) {
                  Notify.send("Request Withdrawal Added");
                  return callback({
                    status: "Your Withdrawal Request was sended.",
                  });
                } else {
                  return callback({ status: "Something is Wrong !" });
                }
              });
            }
          }
        );
      });
    });
  });
};

Rule.swapCoins = function (id, amount, amount1, coin, coin1, immed, callback) {
  if ((!id && !coin && !amount && !coin1 && !amount1, !immed)) {
    return callback({ status: "Invalid parameters provided." });
  }

  id = _.toNumber(id);
  coin = _.lowerCase(coin);
  coin1 = _.lowerCase(coin1);
  amount = _.toNumber(amount);
  amount1 = _.toNumber(amount1);

  // Check if the test coin is being used
  if (coin === "nc" || coin1 === "nc") {
    return callback({ status: "NC is the test coin !" });
  }

  // Validate the input amounts
  if (!H.isValidNumber(amount) || !H.isValidNumber(amount1)) {
    return callback({ status: "Please enter valid amounts" });
  }

  // Get the user's credit for the initial coin
  Rule.getClientCoinCredit(id, coin, (result) => {
    if (result === false) {
      return callback({ status: "Failed to get user's credit." });
    }

    var cash = _.toNumber(result);

    // If user doesn't have enough credit for the initial coin
    if (amount > cash) {
      return callback({ status: "Your Credit is not Enough." });
    }

    // Deduct the initial coin amount from the user's balance
    Rule.reduceBalance(id, amount, coin, (isOk) => {
      if (!isOk) {
        return callback({ status: "Failed to reduce user's balance." });
      }

      // Add the target coin amount to the user's balance
      Rule.addBalance(id, amount1, coin1, (isOk) => {
        if (!isOk) {
          return callback({ status: "Failed to add to user's balance." });
        }

        // Notify the user that the swap was successful
        Notify.send("Swap Coins Successful");
        return callback({ status: "Coins swapped successfully." });
      });
    });
  });
};

/*
 * Get Client Credit by Coin
 */
Rule.creditCoin = function (id, coin, callback) {
  if (!id && !coin) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);

  Rule.getClientCoinCredit(id, coin, (result) => {
    if (result === false) return;
    callback({ credit: H.CryptoSet(result, coin), uid: id });
  });
};

/*
 * Get Client Wallet Address
 */
Rule.getWalletAddress = function (id, coin, callback) {
  if (!id && !coin) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);

  pg.query(
    "SELECT wallet FROM users WHERE id = $1",
    [id],
    function (err, results) {
      if (err) {
        console.log("error on UserRule: 1310", err);
        return callback(false, true);
      }
      try {
        if (!_.isUndefined(results.rows[0])) {
          if (
            results.rows[0].wallet[coin] !== null &&
            !_.isUndefined(results.rows[0].wallet[coin])
          ) {
            callback(
              { uid: id, wallet: coin, address: results.rows[0].wallet[coin] },
              false
            );
          } else {
            cryptoMakeWallet(_.toNumber(id), _.lowerCase(coin), (result) => {
              callback({ uid: id, wallet: coin, address: result });
            });
          }
        } else {
          cryptoMakeWallet(_.toNumber(id), _.lowerCase(coin), (result) => {
            callback({ uid: id, wallet: coin, address: result });
          });
        }
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Update Client Wallet Address
 */
Rule.updateUserWallet = function (id, coin, address, callback) {
  if (!id && !coin && !address) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);

  Rule.getUserInfo(id, (result, er) => {
    if (er) return;

    const wallet = result.wallet;

    let update = Object.assign({}, wallet, { [coin]: address });
    update = JSON.stringify(update);

    pg.query(
      `UPDATE users SET wallet = $1 WHERE id = $2`,
      [update, id],
      function (err, ok) {
        if (ok) {
          callback(true, false);
        } else return callback(false, true);
      }
    );
  });
};

/*
 * Top Winners
 */
Rule.topWinners = function (callback) {
  let cache = myCache.get("topWinners");

  if (!_.isUndefined(cache)) {
    callback(cache);
  } else {
    pg.query(
      "SELECT name, profit, avatar, id FROM users ORDER BY profit_high->>'btc' DESC LIMIT 8",
      function (err, results) {
        if (err) {
          console.log("error on User Rule: 1094");
          return callback(false);
        }
        try {
          myCache.mset([
            { key: "topWinners", val: results.rows, ttl: 60 * 60 },
          ]);
          return callback(results.rows);
        } catch (e) {
          console.log(e);
        }
      }
    );
  }
};

/*
 * Last Bets for Home
 */
Rule.lastBets = function (callback) {
  pg.query(
    "SELECT * FROM bets WHERE game != $1 ORDER BY created DESC LIMIT $2",
    ["crash", 10],
    function (err, results) {
      if (err) {
        console.log("error on User Rule lastBets", err);
        return callback(false, true);
      }
      try {
        callback(_.reverse(results.rows), false);
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Get BankRoll Amount
 */
Rule.getBankRoll = function (game, coin, callback) {
  if (!coin) return;

  coin = _.lowerCase(coin);

  if (coin === "nc") return callback(0, false);

  pg.query(
    `SELECT ${coin} FROM bank WHERE id = $1`,
    [1],
    function (err, results) {
      if (err) {
        console.log("error on User Rule: 1969", err);
        return callback(false, true);
      }
      try {
        let amount = results.rows[0][coin];
        if (_.toNumber(amount) < 0) {
          amount = amount.toString().replace("-", "");
          if (amount > 1.0) {
            amount = amount % 2;
          }
        }
        callback(amount, false);
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Get  Full BankRoll Amount
 */
Rule.getFullBankRoll = function (callback) {
  pg.query(`SELECT * FROM bank WHERE id = $1`, [1], function (err, results) {
    if (err) {
      console.log("error on User Rule: 2024", err);
      return callback(false, true);
    }
    try {
      callback(results.rows[0], false);
    } catch (e) {
      console.log(e);
    }
  });
};

/*
 * Get Client Profit
 */
Rule.getUserProfit = function (id, coin, callback) {
  if (!id && !coin) return;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);

  pg.query(
    "SELECT profit FROM users WHERE id = $1",
    [id],
    function (err, results) {
      if (err) {
        console.log("error user role: 841");
        return callback(false, true);
      }

      try {
        if (!_.isUndefined(results.rows[0])) {
          let bc = results.rows[0].profit[coin];

          if (_.isUndefined(bc)) bc = "0.00000000";

          callback(bc, false);
        }
      } catch (e) {
        console.log("error user role: 841");
        callback(false, true);
      }
    }
  );
};

/*
 * Get the user play count
 */
Rule.playingCount = function (id, callback) {
  let cache = myCache.get("playingCount" + id);

  if (!_.isUndefined(cache)) {
    callback(cache);
  } else {
    pg.query(
      "SELECT COUNT(*) from bets where uid = $1",
      [id],
      function (err, res) {
        if (err) return callback(0);
        else {
          myCache.mset([
            { key: "playingCount" + id, val: res.rows[0].count, ttl: 5 * 60 },
          ]);
          callback(res.rows[0].count);
        }
      }
    );
  }
};

/*
 * Get the user win count
 */
Rule.winCount = function (id, callback) {
  let cache = myCache.get("winCount" + id);

  if (!_.isUndefined(cache)) {
    callback(cache);
  } else {
    pg.query(
      "SELECT COUNT(*) from bets where uid = $1 AND profit != 0.00000000;",
      [id],
      function (err, res) {
        if (err) return callback(0);
        else {
          myCache.mset([
            { key: "winCount" + id, val: res.rows[0].count, ttl: 5 * 60 },
          ]);
          try {
            callback(res.rows[0].count);
          } catch (e) {}
        }
      }
    );
  }
};

/*
 * Get and Calculate User Medal
 */
Rule.getUserMedal = function (id, callback) {
  let cache = myCache.get("getUserMedal" + id);

  if (!_.isUndefined(cache)) {
    callback(cache);
  } else {
    pg.query(
      "SELECT SUM( amount ) FROM deposits WHERE uid = $1 AND coin = $2",
      [id, "btc"],
      function (err, results) {
        if (err) return callback(0);
        else {
          if (results.rows.length === 0) return callback(0);

          let medal = 0;

          let deposited = _.toNumber(results.rows[0].sum);

          //100 USD
          let medal_A = 0.0031;

          //500 USD
          let medal_B = 0.015;

          //2000 USD
          let medal_C = 0.061;

          if (deposited >= medal_C) medal = 3;
          else if (deposited >= medal_B) medal = 2;
          else if (deposited >= medal_A) medal = 1;

          myCache.mset([{ key: "getUserMedal" + id, val: medal, ttl: 6200 }]);

          return callback(medal);
        }
      }
    );
  }
};

/*
 * Get The User favorite games
 */
Rule.getUserFavoriteGames = function (id, callback) {
  let cache = myCache.get("getUserFavoriteGames" + id);

  if (!_.isUndefined(cache)) {
    callback(cache);
  } else {
    pg.query(
      "SELECT game FROM bets WHERE uid = $1",
      [id],
      function (err, results) {
        if (err) return callback([]);
        else {
          if (results.rows.length === 0) return callback([]);

          let games = results.rows;

          if (_.isUndefined(games)) return callback([]);

          myCache.mset([
            { key: "getUserFavoriteGames" + id, val: games, ttl: 3600 },
          ]);

          return callback(games);
        }
      }
    );
  }
};

/*
 * Get the user last profit status
 */
Rule.getUserLastProfit = function (id, callback) {
  if (!id && _.isUndefined(id) && _.isNull(id)) return callback(false);

  id = _.toNumber(id);
  pg.query(
    "SELECT profit FROM bets WHERE uid = $1 ORDER by created DESC LIMIT 2",
    [id],
    function (err, results) {
      if (err) return callback(null);
      else {
        try {
          if (!_.isUndefined(results.rows)) {
            callback(_.last(results.rows));
          } else callback(false);
        } catch (e) {
          callback(false);
        }
      }
    }
  );
};

/*
 * Get the full User info
 */
Rule.userInfo = function (id, coin, first, rateNULL, callback) {
  if (!id && !coin) return;

  coin = _.lowerCase(coin);

  var rate = Market.rate(); // Get Market Rate

  Rule.getUserInfoById(id, function (details, err) {
    if (err || !details.id) {
      return callback({ status: false });
    }

    const coins = [
      "btc",
      "eth",
      "bch",
      "usdt",
      "ltc",
      "trx",
      "bnb",
      "doge",
      "xrp",
      "ada",
      "usdp",
      "usdc",
      "mkr",
      "nexo",
      "tusd",
      "busd",
      "matic",
      "shib",
      "inr",
    ];

    var highArr = [];
    var bestCoin = coin;
    var plusProfit = 0;

    // Get default user coin and amounts by high net profit
    if (first === true) {
      coins.forEach((coinName, i) => {
        var np =
          details.profit[coinName] !== undefined ? details.profit[coinName] : 0;
        np = _.replace(_.toString(np), "-", "");

        np = _.toNumber(np);
        np = H.CryptoSet(np);

        if (np !== 0 && np !== NaN) {
          var convertToUsd =
            np * H.CryptoSet(_.toNumber(rate[_.upperCase(coinName)]));
          if (convertToUsd !== NaN && !_.isNaN(convertToUsd))
            plusProfit += convertToUsd;
        }
      });
    }

    let avatar =
      details.avatar !== null ? details.avatar : UPLOAD_URL + "avatar.png";
    let opts = { format: "%v", code: "USD", maxFraction: 5, minFraction: 5 };
    let netProfit = formatCurrency(plusProfit, opts);

    Rule.playingCount(details.id, (count) => {
      Rule.winCount(details.id, (win) => {
        Rule.getUserMedal(details.id, (medal) => {
          Rule.getUserFavoriteGames(details.id, (games) => {
            callback({
              status: true,
              id: id,
              coin: _.upperCase(bestCoin),
              played: count,
              wined: win,
              medal: medal,
              name: details.name,
              avatar: avatar,
              level: details.level,
              created: details.created,
              friend: details.friends,
              profit: netProfit,
              favorite_games: games,
            });
          });
        });
      });
    });
  });
};

// Define the threshold amounts and corresponding variables to increment

Rule.rakeback = function (plusProfit, id, rakeback) {
  const profitThresholds = [
    { amount: 0, reward: 0 },
    { amount: 100, reward: 0.2 },
    { amount: 500, reward: 1 },
    { amount: 2000, reward: 4 },
    { amount: 5000, reward: 10 },
    { amount: 9000, reward: 18 },
    { amount: 15000, reward: 30 },
    { amount: 22000, reward: 44 },
    { amount: 30000, reward: 60 },
    { amount: 50000, reward: 100 },
    { amount: 80000, reward: 160 },
    { amount: 150000, reward: 300 },
  ];

  console.log(rakeback, "rakeback");
  console.log(plusProfit, "plusProfit");

  profitThresholds.forEach((threshold, index) => {
    if (
      plusProfit <= threshold.amount &&
      (index === 0 || plusProfit > profitThresholds[index - 1].amount)
    ) {
      if (rakeback < index) {
        let amount = threshold.reward;

        pg.query(
          `UPDATE credits SET usdt = usdt + $2 WHERE uid = $1`,
          [id, amount],
          function (err, res) {
            if (!err) {
              if (!_.isUndefined(res.rows[0])) console.log(res.rows[0][usdt]);
            } else {
              console.log("err user rule -1336", err);
              // callback(false, true);
            }
          }
        );

        pg.query(
          `UPDATE users SET rakeback = $2 WHERE id = $1`,
          [id, index],
          function (err, res) {
            if (!err) {
              if (!_.isUndefined(res.rows[0]))
                console.log(res.rows[0][rakeback]);
            } else {
              console.log("err user rule -1336", err);
              // callback(false, true);
            }
          }
        );

        // rakeback += 1;
      }
    }
  });
};

Rule.rakeAmount = function (uid, amount, coin) {
  var rate = Market.rate(); // Get Market Rate

  amount = (amount * 2) / 1000;

  amount = amount * _.toNumber(rate[_.upperCase(coin)]);

  if (amount)
    pg.query(
      `UPDATE users SET rakeamount = rakeamount + $2 WHERE id = $1`,
      [uid, amount],
      function (err, res) {
        if (!err) {
          if (!_.isUndefined(res.rows[0])) console.log(res.rows[0][rakeback]);
        }
      }
    );
};

Rule.checkProfit = function (id) {
  if (!id) return;

  // coin = _.lowerCase(coin);

  var rate = Market.rate(); // Get Market Rate

  Rule.getUserInfoById(id, function (details, err) {
    if (err || !details.id) {
      console.log(err, details.id, "err in getUserInfoById 2629");
      return;
    }

    console.log(id, details.id);

    id = _.toNumber(id);

    const coins = [
      "btc",
      "eth",
      "bch",
      "usdt",
      "ltc",
      "trx",
      "bnb",
      "doge",
      "xrp",
      "ada",
      "usdp",
      "usdc",
      "mkr",
      "nexo",
      "tusd",
      "busd",
      "matic",
      "shib",
      "inr",
    ];

    var highArr = [];
    // var bestCoin = coin;
    var plusProfit = 0;
    // var rakeback = Rule.rakeback(id, plusProfit);

    // Get default user coin and amounts by high net profit

    coins.forEach((coinName, i) => {
      var np =
        details.profit[coinName] !== undefined ? details.profit[coinName] : 0;
      np = _.replace(_.toString(np), "-", "");

      np = _.toNumber(np);
      np = H.CryptoSet(np);

      if (np !== 0 && np !== NaN) {
        var convertToUsd =
          np * H.CryptoSet(_.toNumber(rate[_.upperCase(coinName)]));
        if (convertToUsd !== NaN && !_.isNaN(convertToUsd))
          plusProfit += convertToUsd;
      }
    });

    pg.query(
      "SELECT rakeback from users WHERE id = $1",
      [id],
      function (err, res) {
        if (!err) {
          var _rakeback = res.rows[0].rakeback;

          Rule.rakeback(plusProfit, id, _rakeback);
        } else {
          console.log("err -166", err);
          // callback(null);
        }
      }
    );
  });
};

Rule.checkRakeAmount = function (id, callback) {
  pg.query(
    "SELECT rakeamount from users WHERE id = $1",
    [id],
    function (err, res) {
      if (!err) {
        var _rakeback = res.rows[0].rakeamount;
        callback({ amount: _rakeback });
      } else {
        console.log("err -166", err);
        // callback(null);
      }
    }
  );
};

Rule.addRakeback = function (id, callback) {
  return new Promise((resolve, reject) => {
    pg.query(
      "SELECT rakeamount from users WHERE id = $1",
      [id],
      function (err, res) {
        if (err) {
          console.log("Error selecting rakeamount: ", err);
          reject(err);
        } else {
          const _rakeback = res.rows[0].rakeamount;
          const rakebackNum = _.toNumber(_rakeback);
          if (rakebackNum) {
            pg.query(
              `UPDATE credits SET usdt = usdt + $2 WHERE uid = $1`,
              [id, rakebackNum],
              function (err, res) {
                if (err) {
                  console.log("Error updating credits: ", err);
                  reject(err);
                } else {
                  pg.query(
                    `UPDATE users SET rakeamount = $2 WHERE id = $1`,
                    [id, 0],
                    function (err, res) {
                      if (err) {
                        console.log("Error updating rakeamount: ", err);
                        reject(err);
                      } else {
                        callback({ status: "Rakeback claimed successfully." });
                        resolve();
                      }
                    }
                  );
                }
              }
            );
          } else {
            resolve();
          }
        }
      }
    );
  });
};

/*
 * Get Client Chart
 */
Rule.userChart = function (id, game, coin, callback) {
  if (!id && !game) return;

  let cache = myCache.get("userChart_" + id);

  if (!_.isUndefined(cache)) {
    callback({ data: cache }, false);
  } else {
    pg.query(
      "SELECT * FROM bets WHERE uid = $1 AND game = $2 ORDER BY created DESC LIMIT 20",
      [id, game],
      function (err, results) {
        if (err) {
          console.log("error on User Rule: 917", err);
          return callback(false, true);
        }
        if (results.rows.length === 0) {
          return callback([]);
        }
        try {
          myCache.mset([
            { key: "userChart_" + id, val: results.rows, ttl: 3 * 60 },
          ]);
          callback({ data: results.rows }, false);
        } catch (e) {
          console.log(e);
        }
      }
    );
  }
};

/*
 * Get Site notifications
 */
Rule.notifications = function (callback) {
  let cache = myCache.get("notifications");

  if (!_.isUndefined(cache)) {
    callback(cache);
  } else {
    pg.query(
      "SELECT * FROM notifications ORDER BY date DESC LIMIT 4",
      function (err, results) {
        myCache.mset([{ key: "notifications", val: results.rows, ttl: 5000 }]);
        callback(results.rows);
      }
    );
  }
};

/*
 * Get Game Details
 */
Rule.gameDetails = function (id, callback) {
  if (!id) return;
  id = _.toString(id);

  let cache = myCache.get("gameDetails_" + id);

  if (!_.isUndefined(cache)) {
    callback(cache);
  } else {
    pg.query(
      "SELECT * FROM bets WHERE gid = $1",
      [id],
      function (err, results) {
        if (err) {
          console.log("error on User Rule: 946", err);
          return callback(false, true);
        } else {
          pg.query(
            "SELECT * FROM crashs WHERE gid = $1",
            [id],
            function (err, res) {
              if (err) {
                console.log("error on User Rule: 946", err);
                return callback(false, true);
              }

              let bets = [];

              if (!_.isUndefined(results.rows)) {
                bets = results.rows;
              }

              let info = [];

              if (!_.isUndefined(res.rows[0])) {
                info = res.rows[0];
              }

              var data = {
                id: id,
                data: bets,
                info: info,
              };

              callback(data);

              myCache.mset([
                { key: "gameDetails_" + id, val: data, ttl: 3600 },
              ]);
            }
          );
        }
      }
    );
  }
};

/*
 * Client Private Messages
 */
Rule.messages = function (token, id, name, callback) {
  if (!id && !name) return;

  Rule.getIdByName(name, function (target) {
    if (!target) {
      console.log("target not found USER ROLE: 1304");
      return false;
    }

    let key = _.toNumber(id) + _.toNumber(target);

    pg.query(
      "SELECT * FROM messages WHERE room_key = $1",
      [key],
      function (err, message) {
        if (err) return callback(err);

        let rows = [];

        if (!_.isUndefined(message.rows)) {
          rows = message.rows;
        }

        callback({ message: rows });
      }
    );
  });
};

/*
 * Send Private Messages
 *
 * Can be Optimize
 */
Rule.addMessages = function (token, id, to, message, callback) {
  if (!id && !message && !to) return;

  id = _.toNumber(id);
  let time = H.getCurrentTime(new Date());

  Rule.getIdByName(to, function (targetID) {
    if (!targetID) return;

    Rule.getNameById(id, function (sender) {
      if (!sender) return;

      var key = _.toNumber(id) + _.toNumber(targetID);

      pg.query(
        "INSERT INTO messages (from_uid, to_uid, message, time, from_name, to_name, room_key) VALUES($1, $2, $3, $4, $5, $6, $7)",
        [id, targetID, message, time, sender, to, key],
        function (err, result) {
          if (err) return callback(err);

          callback({
            uid: id,
            target: targetID,
            message: message,
            time: time,
            date: H.getCurrentDate(new Date()),
            from_name: sender,
            to_name: to,
          });
        }
      );
    });
  });
};

/*
 * Get User Bets
 * used in games page in "my bets" tab
 */
Rule.getClientBets = function (uid, game, callback) {
  if (!uid && !game) return;

  uid = _.toNumber(uid);
  game = _.lowerCase(game);
  game = _.replace(game, "-", "_");
  game = _.replace(game, " ", "_");

  pg.query(
    "SELECT * FROM bets WHERE uid = $1 AND game = $2 ORDER by created DESC LIMIT 10",
    [uid, game],
    function (err, results) {
      if (err) {
        console.log("error on UserRule: 42", err);
        return callback(false, true);
      }
      try {
        callback(results.rows, false);
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Get The Last Game Bets
 * used in games page in "all bets" tab
 */
Rule.lastGameBets = function (game, callback) {
  if (!game) return;

  game = _.lowerCase(game);
  game = _.replace(game, "-", "_");
  game = _.replace(game, " ", "_");

  pg.query(
    "SELECT * FROM bets WHERE game = $1 ORDER BY created DESC LIMIT 10",
    [game],
    function (err, results) {
      if (err) {
        console.log("Error User Role: 1028", err);
        return callback(false);
      }
      try {
        callback(results.rows, false);
      } catch (e) {
        console.log(e);
      }
    }
  );
};

/*
 * Get The Game Bots
 */
Rule.getGameBots = function (game, callback) {
  if (!game) return;

  pg.query(
    "SELECT * FROM bots WHERE status = $2 AND game = $1 LIMIT 15",
    [game, "active"],
    function (err, results) {
      if (err) {
        console.log("Error User Role: 1381", err);
        return callback(false);
      }

      let rows = [];

      if (!_.isUndefined(results.rows)) {
        rows = results.rows;
      }

      callback(rows);
    }
  );
};

/*
 * dummy
 */
Rule.getdummyBots = function (callback) {
  pg.query("SELECT * FROM bots LIMIT 15", function (err, results) {
    if (err) {
      console.log("Error User Role: 1381", err);
      return callback(false);
    }

    let rows = [];

    if (!_.isUndefined(results.rows)) {
      rows = results.rows;
    }

    callback(rows);
  });
};

/*
 * Get The Random Bots by time
 */
Rule.getRandomBots = function (callback) {
  const time = new Date().getHours();

  let cache = myCache.get("random_bots_" + time);

  if (!_.isUndefined(cache)) {
    callback(cache);
  } else {
    pg.query(
      "SELECT * FROM bots WHERE time = $1 AND status = $3 AND game != $4 AND game != $5 ORDER BY RANDOM() LIMIT $2",
      [time, BotsLimit, "active", "crash", "keno"],
      function (err, results) {
        if (err) {
          console.log("Error User Role: 1753", err);
          return callback(false);
        }

        let rows = [];

        if (!_.isUndefined(results.rows)) {
          rows = results.rows;
        }

        if (rows.length === 0) return callback(false);

        myCache.mset([{ key: "random_bots_" + time, val: rows, ttl: 3600 }]);

        callback(rows);
      }
    );
  }
};

/*
 * Get The Game Bots By Game and Time
 */
Rule.getBots = function (game, time, callback) {
  if (!game) return;

  const timer = new Date().getHours();

  let cache = myCache.get("bots_" + timer + game);

  if (!_.isUndefined(cache)) {
    callback(cache);
  } else {
    pg.query(
      "SELECT * FROM bots WHERE status = $3 AND game = $1 AND time = $2 LIMIT $4",
      [game, timer, "active", BotsLimit],
      function (err, results) {
        if (err) {
          console.log("Error User Role: 1781", err);
          return callback(false);
        }

        let rows = [];

        if (!_.isUndefined(results.rows)) {
          rows = results.rows;
        }

        if (rows.length === 0) return callback(false);

        myCache.mset([{ key: "bots_" + timer + game, val: rows, ttl: 3600 }]);

        callback(rows);
      }
    );
  }
};

/*
 * Get The Bots for Chats by Time
 */
Rule.getBotsByTime = function (callback) {
  const timer = new Date().getHours();

  pg.query(
    "SELECT * FROM bots WHERE status = $2 AND time = $1 ORDER BY RANDOM() LIMIT $3",
    [timer, "active", 10],
    function (err, results) {
      if (err) {
        console.log("Error User Role: 1761", err);
        return callback(false);
      }

      let rows = [];

      if (!_.isUndefined(results.rows)) {
        rows = results.rows;
      }

      if (rows.length === 0) return callback(false);

      callback(rows);
    }
  );
};

/*
 * Get The Game Bots for Chats
 */
Rule.getBotsForRapidChats = function (callback) {
  pg.query(
    "SELECT * FROM bots WHERE status = $1 ORDER BY RANDOM() LIMIT $2",
    ["active", 1],
    function (err, results) {
      if (err) {
        console.log("Error User Role: 1913", err);
        return callback(false);
      }

      let rows = [];

      if (!_.isUndefined(results.rows)) {
        rows = results.rows;
      }

      if (rows.length === 0) return callback(false);

      callback(rows);
    }
  );
};

/*
 * Get The Random Bots
 */
Rule.getTheRandomBots = function (c, callback) {
  pg.query(
    "SELECT * FROM bots WHERE status = $1 ORDER BY RANDOM() LIMIT $2",
    ["active", c],
    function (err, results) {
      if (err) {
        console.log("Error User Role: 1913", err);
        return callback(false);
      }

      let rows = [];

      if (!_.isUndefined(results.rows)) {
        rows = results.rows;
      }

      if (rows.length === 0) return callback(false);

      callback(rows);
    }
  );
};

/*
 * Get The Bot from Users by name
 */
Rule.getBotByName = function (name, callback) {
  pg.query(
    "SELECT * FROM users WHERE name = $1",
    [name],
    function (err, results) {
      if (err) {
        console.log("Error User Role: 1913", err);
        return callback(false);
      }

      let rows = [];

      if (!_.isUndefined(results.rows)) {
        rows = results.rows[0];
      }

      if (rows.length === 0) return callback(false);

      callback(rows);
    }
  );
};

/*
 * Upload User Avatar
 */
Rule.uploadAvatar = function (client, event, uploader, callback) {
  if (!client && !event) return;

  let type = H.baseName(event.file.name).trim();

  if (type !== "jpg" || type !== "png") {
    // uploader.abort(event.file.id, client);
  }

  let token = event.file.meta.token;
  if (!token) return;

  let avatar = UPLOAD_URL + event.file.name;

  Rule.getIdByName(token.name, function (uid) {
    if (!uid) return;
    pg.query(
      "UPDATE users SET avatar = $1 WHERE id = $2",
      [avatar, uid],
      function (err, res) {
        if (err) {
          console.log("error user role: 1258", err);
          return callback(false);
        }
        return callback({ status: true, file: UPLOAD_URL + event.file.name });
      }
    );
  });
};

/*
 * Bulk Change Database
 */
Rule.bulkChange = function (username, id, callback) {
  if (!username && !id) return;
  let uid = _.toNumber(id);
  let rooms = ["chat_global", "chat_brazil"];

  //Change on Chat rooms
  rooms.forEach((name, i) => {
    pg.query(
      "UPDATE " + name + " SET name = $1 WHERE uid = $2",
      [username, uid],
      function (err, res) {
        if (err) {
          return callback(false);
        }
      }
    );
  });
  //Change on Bets
  pg.query(
    "UPDATE bets SET name = $1 WHERE uid = $2",
    [username, uid],
    function (err, res) {
      if (err) {
        console.warn(err);
        return callback(false);
      }
    }
  );
  //Change on Private Messages
  pg.query(
    "UPDATE messages SET from_name = $1 WHERE from_uid = $2",
    [username, uid],
    function (err, res) {
      if (err) {
        console.warn(err);
        return callback(false);
      }
    }
  );
  pg.query(
    "UPDATE messages SET to_name = $1 WHERE to_uid = $2",
    [username, uid],
    function (err, res) {
      if (err) {
        console.warn(err);
        return callback(false);
      }
    }
  );

  callback(true);
};

/*
 * Make Rain
 */
Rule.makeRain = function (token, id, amount, coin, players, room, callback) {
  if (!coin && !amount && !id && !players && !room) return false;

  id = _.toNumber(id);
  coin = _.lowerCase(coin);
  amount = _.toNumber(amount);

  if (coin === "nc") {
    return callback(
      false,
      { status: false, message: "NC is the test coin !" },
      true
    );
  }

  if (amount <= 0)
    return callback(
      false,
      { status: false, message: "Please Enter Currect Amount !" },
      true
    );

  assert(amount >= 0);

  if (_.isNaN(amount))
    return callback(
      false,
      { status: false, message: "Please Enter Currect Amount !" },
      true
    );

  let table = "chat_" + _.lowerCase(room);
  let locked = false;
  let list = [];
  let ids = [];

  //Check if User Have Credit
  Rule.getClientCoinCredit(id, coin, (bc) => {
    let credit = _.toNumber(bc);
    let allAmount = amount * _.toNumber(players);
    allAmount = _.toNumber(allAmount);

    if (_.isUndefined(credit)) {
      console.log("ERROR USER CREDIT ! 1", credit);
      return callback(
        false,
        { status: false, message: "Your Credit is Not enough!" },
        true
      );
    }

    if (credit === 0) {
      console.log("ERROR USER CREDIT ! 2", credit);
      return callback(
        false,
        { status: false, message: "Your Credit is Not enough!" },
        true
      );
    }

    if (credit < allAmount) {
      console.log("ERROR USER CREDIT ! 3", credit);
      return callback(
        false,
        { status: false, message: "Your Credit is Not enough!" },
        true
      );
    }

    assert(credit > allAmount);

    pg.query(
      `SELECT uid, name FROM ${table} ORDER BY sorter DESC LIMIT $1`,
      [200],
      function (err, results) {
        if (err) {
          console.log("Error RAIN USER Role: 2210", err);
          return callback(
            false,
            { status: false, message: "Please try again later" },
            true
          );
        }

        results.rows.forEach((user, i) => {
          Rule.getNameById(id, (senderName) => {
            if (user.name !== "SystemBot") {
              if (user.name !== senderName) {
                if (_.toNumber(user.uid) !== id) {
                  let r = "@" + user.name;
                  if (list.indexOf(r) === -1) {
                    if (list.length < _.toNumber(players)) {
                      list.push(r);
                      ids.push(user.uid);
                    }
                  }
                }
              }
            }
          });
        }); // End Foreach

        var SENDED = false;

        //Added
        H.wait(1000).then(() => {
          if (list.length === _.toNumber(players)) {
            //Reduce Client Amount
            Rule.reduceBalance(id, allAmount, coin, (result, er) => {
              if (er) {
                H.log("info", `ERROR on reduceBalance: 2446`);
                return callback(
                  false,
                  { status: false, message: "Please try again later" },
                  true
                );
              } else {
                //Add Balance to Users
                for (var i = 0; i < ids.length; i++) {
                  Rule.addBalance(
                    _.toNumber(ids[i]),
                    amount,
                    coin,
                    (rr, errr) => {
                      //Emit On Chat
                      Rule.getNameById(id, (senderName) => {
                        if (!senderName) return;

                        let RoboID = 7153444;
                        let RoboName = "SystemBot";
                        let avatar = UPLOAD_URL + "1241124121414121412.png";
                        let sorter = _.toNumber(new Date().getTime());
                        let time = H.getCurrentTime(new Date());

                        let message = `@${senderName} Has Been Rained ${H.CryptoSet(
                          amount * _.toNumber(players),
                          coin
                        )} ${_.upperCase(coin)} on: ${list}`;

                        //Add Message then Boardcast to Clients
                        pg.query(
                          "INSERT INTO " +
                            table +
                            "(name, uid, avatar, message, time, sorter) VALUES($1, $2, $3, $4, $5, $6)",
                          [RoboName, RoboID, avatar, message, time, sorter],
                          function (err, result) {
                            if (err) {
                              H.log("info", `error on insert to table in rain`);
                              return callback(
                                false,
                                {
                                  status: false,
                                  message: "Please try again later",
                                },
                                true
                              );
                            } else {
                              if (!SENDED) {
                                SENDED = true;

                                return callback(
                                  {
                                    country: _.lowerCase(room),
                                    message: message,
                                    name: RoboName,
                                    avatar: avatar,
                                    level: 1,
                                    time: time,
                                    date: H.getCurrentDate(new Date()),
                                    sorter: sorter,
                                  },
                                  {
                                    message: "Rained Done !",
                                    token: token,
                                  },
                                  false
                                );
                              }
                            }
                          }
                        );
                      });
                    }
                  );
                }
              }
            });
          }
        });
      }
    );
  });
};

/*
 * Save Wallet
 */
function saveWallet(id, coin, address, callback) {
  if (!id && !coin && !address) return;

  coin = _.lowerCase(coin);

  id = _.toNumber(id);
  pg.query(
    "INSERT INTO wallets(address, uid, coin) VALUES($1, $2, $3)",
    [address, id, coin],
    function (err, result) {
      if (err) {
        console.log("WALLET ERROR : 11", err);
        return callback(false);
      }
      callback(true);
    }
  );
}

/*
 * Make BTC Wallet
 */
function cryptoMakeBtcWallet(uid, callback) {
  if (!uid) return;

  let coin = "btc";
  let myCallBack = encodeURIComponent(`${API_URL}/crypto_callbacks?uid=${uid}`);

  const tokenAPI = `http://api.cryptapi.io/${coin}/create/?callback=${myCallBack}&address=${BTC_ADDRESS}&pending=1`;

  axios({  headers:{
    "Access-Control-Allow-Origin": "*",
    "Access-Control-Allow-Methods":"*"
  },
    url: tokenAPI,
    method:'GET'
  }).then(function (req) {
    var response = req.data;

    Rule.updateUserWallet(uid, coin, response.address_in, function (result) {
      if (result) {
        saveWallet(uid, coin, response.address_in, () => {
          callback(response.address_in);
        });
      }
    });
  });
}

/*
 * Make ETH Wallet
 */
function cryptoMakeEthWallet(uid, callback) {
  if (!uid) return;

  let coin = "eth";
  let myAddress = ETH_ADDRESS;

  let myCallBack = `${API_URL}/crypto_callbacks?uid=${uid}`;
  myCallBack = encodeURIComponent(myCallBack);

  const tokenAPI = `http://api.cryptapi.io/${coin}/create/?callback=${myCallBack}&address=${myAddress}&pending=1`;

  axios({  headers:{
    "Access-Control-Allow-Origin": "*",
    "Access-Control-Allow-Methods":"*"
  },
    url: tokenAPI,
    method:'GET'
  }).then(function (req) {
    var response = req.data;
    Rule.updateUserWallet(uid, coin, response.address_in, function (result) {
      if (result) {
        saveWallet(uid, coin, response.address_in, () => {
          callback(response.address_in);
        });
      }
    });
  });
}

/*
 * Make BCH Wallet
 */
function cryptoMakeBchWallet(uid, callback) {
  if (!uid) return;

  let coin = "bch";
  let myAddress = BCH_ADDRESS;

  let myCallBack = `${API_URL}/crypto_callbacks?uid=${uid}`;
  myCallBack = encodeURIComponent(myCallBack);

  const tokenAPI = `http://api.cryptapi.io/${coin}/create/?callback=${myCallBack}&address=${myAddress}&pending=1`;

  axios({  headers:{
    "Access-Control-Allow-Origin": "*",
    "Access-Control-Allow-Methods":"*"
  },
    url: tokenAPI,
    method:'GET'
  }).then(function (req) {
    var response = req.data;
    Rule.updateUserWallet(uid, coin, response.address_in, function (result) {
      if (result) {
        saveWallet(uid, coin, response.address_in, () => {
          callback(response.address_in);
        });
      }
    });
  });
}

/*
 * Make LTC Wallet
 */
function cryptoMakeLtcWallet(uid, callback) {
  if (!uid) return;

  let coin = "ltc";
  let myCallBack = `${API_URL}/crypto_callbacks?uid=${uid}`;
  myCallBack = encodeURIComponent(myCallBack);

  const tokenAPI = `http://api.cryptapi.io/${coin}/create/?callback=${myCallBack}&address=${LTC_ADDRESS}&pending=1`;

  axios({  headers:{
    "Access-Control-Allow-Origin": "*",
    "Access-Control-Allow-Methods":"*"
  },
    url: tokenAPI,
    method:'GET'
  }).then(function (req) {
    var response = req.data;
    Rule.updateUserWallet(uid, coin, response.address_in, function (result) {
      if (result) {
        saveWallet(uid, coin, response.address_in, () => {
          callback(response.address_in);
        });
      }
    });
  });
}

/*
 * Make TRX Wallet
 */
function cryptoMakeTrxWallet(uid, callback) {
  if (!uid) return;

  let coin = "trx";
  let myCallBack = `${API_URL}/crypto_callbacks?uid=${uid}`;
  myCallBack = encodeURIComponent(myCallBack);

  const tokenAPI = `http://api.cryptapi.io/${coin}/create/?callback=${myCallBack}&address=${TRX_ADDRESS}&pending=1`;

  axios({  headers:{
    "Access-Control-Allow-Origin": "*",
    "Access-Control-Allow-Methods":"*"
  },
    url: tokenAPI,
    method:'GET'
  }).then(function (req) {
    var response = req.data;
    Rule.updateUserWallet(uid, coin, response.address_in, function (result) {
      if (result) {
        saveWallet(uid, coin, response.address_in, () => {
          callback(response.address_in);
        });
      }
    });
  });
}

/*
 * Make BNB Wallet
 */
function cryptoMakeBnbWallet(uid, callback) {
  if (!uid) return;

  let coin = "bnb";
  let myCallBack = `${API_URL}/crypto_callbacks?uid=${uid}`;
  myCallBack = encodeURIComponent(myCallBack);

  const tokenAPI = `http://api.cryptapi.io/bep20/${coin}/create/?callback=${myCallBack}&address=${ETH_ADDRESS}&pending=1&multi_token=1`;

  axios({  headers:{
    "Access-Control-Allow-Origin": "*",
    "Access-Control-Allow-Methods":"*"
  },
    url: tokenAPI,
    method:'GET'
  }).then(function (req) {
    var response = req.data;
    Rule.updateUserWallet(uid, coin, response.address_in, function (result) {
      if (result) {
        saveWallet(uid, coin, response.address_in, () => {
          callback(response.address_in);
        });
      }
    });
  });
}

/*
 * Make XRP Wallet
 */
function cryptoMakeBinanceWallet(uid, coin, callback) {
  if (!uid) return;

  let myCallBack = `${API_URL}/crypto_callbacks?uid=${uid}`;
  myCallBack = encodeURIComponent(myCallBack);

  const tokenAPI = `http://api.cryptapi.io/bep20/${coin}/create/?callback=${myCallBack}&address=${BINANCE_WALLET_ADDRESS}&pending=1&multi_token=1`;

  // console.log("bnb make", tokenAPI);
  axios({  headers:{
    "Access-Control-Allow-Origin": "*",
    "Access-Control-Allow-Methods":"*"
  },
    url: tokenAPI,
    method:'GET'
  }).then(function (req) {
    var response = req.data;
    Rule.updateUserWallet(uid, coin, response.address_in, function (result) {
      if (result) {
        saveWallet(uid, coin, response.address_in, () => {
          callback(response.address_in);
        });
      }
    });
  });
}

/* make brx wallet */

function cryptoMakeBrxWallet(uid, coin, callback) {
  const address = "0x3ad0ECeB70962C48938e0B4fE3EF9DcA690fa03A";
  Rule.updateUserWallet(uid, coin, address, function (result) {
    if (result) {
      saveWallet(uid, coin, address, () => {
        callback(address);
      });
    }
  });
}

/*
 * Make BSC Wallet
 */
function cryptoMakeBSCWallet(uid, coin, callback) {
  if (!uid && !coin) return;

  let myCallBack = `${API_URL}/crypto_callbacks?uid=${uid}`;
  myCallBack = encodeURIComponent(myCallBack);

  const tokenAPI = `http://api.cryptapi.io/bep20/${coin}/create/?callback=${myCallBack}&address=${BSC_ADDRESS}&pending=1`;

  axios({  headers:{
    "Access-Control-Allow-Origin": "*",
    "Access-Control-Allow-Methods":"*"
  },
    url: tokenAPI,
    method:'GET'
  }).then(function (req) {
    var response = req.data;
    Rule.updateUserWallet(uid, coin, response.address_in, function (result) {
      if (result) {
        saveWallet(uid, coin, response.address_in, () => {
          callback(response.address_in);
        });
      }
    });
  });
}

function cryptoMakeShibWallet(uid, coin, callback) {
  if (!uid && !coin) return;

  let myCallBack = `${API_URL}/crypto_callbacks?uid=${uid}`;
  myCallBack = encodeURIComponent(myCallBack);

  const tokenAPI = `http://api.cryptapi.io/bep20/${coin}/create/?callback=${myCallBack}&address=${BSC_ADDRESS}&pending=1`;

  axios({  headers:{
    "Access-Control-Allow-Origin": "*",
    "Access-Control-Allow-Methods":"*"
  },
    url: tokenAPI,
    method:'GET'
  }).then(function (req) {
    var response = req.data;
    Rule.updateUserWallet(uid, coin, response.address_in, function (result) {
      if (result) {
        saveWallet(uid, coin, response.address_in, () => {
          callback(response.address_in);
        });
      }
    });
  });
}

function cryptoMakeMaticWallet(uid, coin, callback) {
  if (!uid && !coin) return;

  let myCallBack = `${API_URL}/crypto_callbacks?uid=${uid}`;
  myCallBack = encodeURIComponent(myCallBack);

  const tokenAPI = `http://api.cryptapi.io/polygon/${coin}/create/?callback=${myCallBack}&address=${BSC_ADDRESS}&pending=1&multi_token=1`;

  axios({  headers:{
    "Access-Control-Allow-Origin": "*",
    "Access-Control-Allow-Methods":"*"
  },
    url: tokenAPI,
    method:'GET'
  }).then(function (req) {
    var response = req.data;
    Rule.updateUserWallet(uid, coin, response.address_in, function (result) {
      if (result) {
        saveWallet(uid, coin, response.address_in, () => {
          callback(response.address_in);
        });
      }
    });
  });
}

/*
 * Generate Wallet Using cryptoapi.io
 */
function cryptoMakeWallet(uid, coin, callback) {
  if (!coin && !uid) return;

  uid = _.toNumber(uid);
  coin = _.lowerCase(coin);

  console.log("cryptoMakeWallet " + coin, uid);

  if (coin === "bch") {
    return cryptoMakeBchWallet(uid, callback);
  }

  if (coin === "eth") {
    return cryptoMakeEthWallet(uid, callback);
  }

  if (coin === "btc") {
    return cryptoMakeBtcWallet(uid, callback);
  }

  if (coin === "ltc") {
    return cryptoMakeLtcWallet(uid, callback);
  }

  if (coin === "trx") {
    return cryptoMakeTrxWallet(uid, callback);
  }

  if (coin === "bnb") {
    return cryptoMakeBnbWallet(uid, callback);
  }
  if (coin === "usdt") {
    return cryptoMakeTrxWallet(uid, callback);
  }
  if (
    coin === "xrp" ||
    coin === "ada" ||
    coin === "doge" ||
    coin === "shib" ||
    coin === "usdc"
  ) {
    return cryptoMakeBinanceWallet(uid, coin, callback);
  }

  if (coin === "bnb") {
    return cryptoMakeBSCWallet(uid, coin, callback);
  }

  if (coin === "brx") {
    return cryptoMakeBrxWallet(uid, coin, callback);
  }

  // if (coin === "shib") {
  //   return cryptoMakeShibWallet(uid, coin, callback);
  // }

  if (coin === "matic") {
    return cryptoMakeMaticWallet(uid, coin, callback);
  }

  //Default erc20 address
  let myAddress = ETH_ADDRESS;

  let myCallBack = `${API_URL}/crypto_callbacks?uid=${uid}`;
  myCallBack = encodeURIComponent(myCallBack);

  const tokenAPI = `http://api.cryptapi.io/erc20/${coin}/create/?callback=${myCallBack}&address=${myAddress}&pending=1&multi_token=1`;

  axios({  headers:{
    "Access-Control-Allow-Origin": "*",
    "Access-Control-Allow-Methods":"*"
  },
    url: tokenAPI,
    method:'GET'
  }).then(function (req) {
    var response = req.data;
    Rule.updateUserWallet(uid, coin, response.address_in, function (result) {
      if (result) {
        saveWallet(uid, coin, response.address_in, () => {
          callback(response.address_in);
        });
      }
    });
  });
}

//Export Users Rule Module
module.exports = Rule;
