// src/contexts/BettingContext.jsx
import React, { createContext, useContext, useState, useEffect } from 'react'
import { useExposure } from '../hooks/useExposure'

const BettingContext = createContext()
const API_ROOT    = 'https://api.boss707.com/sportsbetting'

// helper: pull uid from cookies
const readUid = () => {
  const m = document.cookie.match(/(?:^|; )uid=([^;]+)/)
  return m ? decodeURIComponent(m[1]) : null
}

// normalize strings
export const norm = s => (s||'').replace(/\s+/g,' ').trim().toLowerCase()

// your existing calcBook logic (unchanged)
const calcBook = bets => {
  const book = { MO:{}, BM:{}, FAN:{} }
  bets.forEach(b => {
    const st = +b.stake||0, od = +b.odds||0
    if (!st||!od) return

    if (b.market==='MO') {
      const [a, bn] = b.matchTitle.split(' v ')
      const t1 = norm(a), t2 = norm(bn)
      book.MO[t1] = book.MO[t1] ?? 0
      book.MO[t2] = book.MO[t2] ?? 0
      book.MO['the draw'] = book.MO['the draw'] ?? 0
      const profit = b.type==='back'
        ? st*(od-1)
        : -st*(od-1)

      if (norm(b.selectionName)===t1) {
        book.MO[t1] += profit
        book.MO[t2] += b.type==='back' ? -st : +st
        book.MO['the draw'] += b.type==='back' ? -st : +st
      }
      else if (norm(b.selectionName)===t2) {
        book.MO[t2] += profit
        book.MO[t1] += b.type==='back' ? -st : +st
        book.MO['the draw'] += b.type==='back' ? -st : +st
      }
      else {
        book.MO['the draw'] += profit
        book.MO[t1] += b.type==='back' ? -st : +st
        book.MO[t2] += b.type==='back' ? -st : +st
      }
    }
    else if (b.market==='BM') {
      const k = norm(b.selectionName)
      book.BM[k] = book.BM[k] ?? 0
      book.BM[k] += b.type==='back'
        ? st*(od-1)
        : -st*(od-1)
    }
    else if (b.market==='FAN') {
      const k = norm(b.selectionName)
      book.FAN[k] = book.FAN[k] ?? 0
      const d = st*od/100
      book.FAN[k] += b.type==='back' ? d : -d
    }
  })

  // round to 2 decimals
  ;['MO','BM','FAN'].forEach(m =>
    Object.keys(book[m]).forEach(k =>
      book[m][k] = +book[m][k].toFixed(2)
    )
  )
  return book
}

export const BettingProvider = ({ children }) => {
  const [selectedBets, setSelectedBets] = useState([])
  const [placedBets,   setPlacedBets]   = useState([])
  const [plByMarket,   setPlByMarket]   = useState({ MO:{}, BM:{}, FAN:{} })

  const { patchMatchExposure, patchBalance, refreshAll, exposures } = useExposure()

  // recalc P/L whenever bets change
  useEffect(() => {
    setPlByMarket(calcBook([...placedBets, ...selectedBets]))
  }, [placedBets, selectedBets])

  const getPL = (m, k) => plByMarket[m]?.[norm(k)] ?? 0

  // manage selected bets
  const addBet         = b => setSelectedBets(p => [...p, { ...b, id: Date.now() }])
  const updateBetStake = (id, v) => setSelectedBets(p => p.map(b => b.id===id?{...b,stake:v}:b))
  const updateBetOdds  = (id, v) => setSelectedBets(p => p.map(b => b.id===id?{...b,odds:v}:b))
  const removeBet      = id => setSelectedBets(p => p.filter(b => b.id!==id))

  // place bets on server
  const placeBets = async rows => {
const uid = readUid();
    console.log('🔐 ExposureProvider: Initializing with UID:', uid);
    if (!uid) {
      alert('No user id found – are you logged in?')
      return { confirmed: [], errors: ['No user id found'] }
    }

    const confirmed = []
    const errors = []
    for (const b of rows) {
      // ensure market is present (default to MO if not provided)
      const safeBet = { market: 'MO', ...b }
      try {
        const token = localStorage.getItem('token');
        if (!token) {
          errors.push({ betId: safeBet.id, message: 'Missing auth token' })
          continue
        }
        const res = await fetch(`${API_ROOT}/place`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token}`,
          },
          body: JSON.stringify({
            user_id:         uid,
            game_type:       safeBet.market,
            match_id:        safeBet.matchId,
            match_title:     safeBet.matchTitle,
            selection_name:  safeBet.selectionName,
            bet_type:        safeBet.type,
            odds:            +safeBet.odds,
            stake_amount:    +safeBet.stake,
            team_one:        safeBet.matchTitle.split(' v ')[0]?.trim() || null,
            team_two:        safeBet.matchTitle.split(' v ')[1]?.trim() || null,
            category:        1,
            original_currency: 'INR',
            original_amount: +safeBet.stake,
            usd_amount:      +(safeBet.stake/83).toFixed(2),
            match_start_time: new Date().toISOString()
          })
        })

        let j = null
        try {
          j = await res.json()
        } catch (_) {}
        if (!res.ok || (j && j.success === false)) {
          const msg = (j && (j.message || j.error)) || `HTTP ${res.status}`
          errors.push({ betId: safeBet.id, message: msg })
          continue
        }

        // update exposures & internal balance
        if (j && j.exposure && j.exposure[safeBet.matchId]) {
          patchMatchExposure(safeBet.matchId, j.exposure[safeBet.matchId].teams)
        }
        if (j && typeof j.balanceDelta === 'number') {
          patchBalance(j.balanceDelta)

          // —— BALANCE STORAGE & BROADCAST ——
          const prev    = Number(localStorage.getItem('credit') || 0)
          const updated = prev + j.balanceDelta
          localStorage.setItem('credit', updated.toString())
          window.dispatchEvent(
            new CustomEvent('credit_update', { detail: updated })
          )
        }
        // ——————————————————————————

        confirmed.push({ ...safeBet, serverBetId: j ? j.bet_id : undefined })
      } catch (e) {
        console.error(e)
        errors.push({ betId: safeBet.id, message: e?.message || 'Network error' })
      }
    }

    const confirmedIds = confirmed.map(b => b.id)
    if (confirmed.length > 0) {
      setPlacedBets(p => [...p, ...confirmed])
    }
    setSelectedBets(prev => prev.filter(b => !confirmedIds.includes(b.id)))

    await refreshAll(uid)

    return { confirmed, errors }
  }

  return (
    <BettingContext.Provider value={{
      selectedBets,
      placedBets,
      setPlacedBets,
      addBet,
      removeBet,
      updateBetStake,
      updateBetOdds,
      placeBets,
      getPL,
      exposures
    }}>
      {children}
    </BettingContext.Provider>
  )
}

export const useBetting = () => useContext(BettingContext)
