import React, { useEffect, useState } from 'react';
import { useGameLaunch } from './gameLaunchContext';
import styled from 'styled-components';
import { MoveLeftIcon } from 'lucide-react';

const LoaderContainer = styled.div`
  position: fixed;
  top: 0;
  left: 0;
  width: 100vw;
  height: 100vh;
  background: #1a1a1a;
  display: flex;
  justify-content: center;
  align-items: center;
  z-index: 10000;
`;

const Loader = styled.div`
  width: 60px;
  height: 60px;
  border: 5px solid #FFB606;
  border-top: 5px solid transparent;
  border-radius: 50%;
  animation: spin 1s linear infinite;

  @keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
  }
`;

const GameIframe = () => {
  const { gameUrl, isGameOpen, setIsGameOpen } = useGameLaunch();
  const [loading, setIsLoading] = useState(true); // Controls the loader
  const [iframeLoaded, setIframeLoaded] = useState(false); // Tracks iframe load status
  const [isMobile, setIsMobile] = useState(window.innerWidth <= 768);

  // Reset states when isGameOpen changes
  useEffect(() => {
    if (isGameOpen) {
      setIsLoading(true); // Show loader when game opens
      setIframeLoaded(false); // Reset iframe loaded state
      const timer = setTimeout(() => {
        setIsLoading(false); // Hide loader after 5 seconds (minimum display time)
      }, 3000);
      return () => clearTimeout(timer); // Cleanup timer
    } else {
      setIsLoading(true); // Reset loader when game closes
      setIframeLoaded(false); // Reset iframe state when game closes
    }
  }, [isGameOpen]);

  // Track screen size changes
  useEffect(() => {
    const handleResize = () => {
      setIsMobile(window.innerWidth <= 768);
    };

    window.addEventListener('resize', handleResize);
    return () => window.removeEventListener('resize', handleResize);
  }, []);

  // Handle iframe load completion
  const handleIframeLoad = () => {
    setIframeLoaded(true); // Set iframe as loaded when onLoad fires
  };

  if (!isGameOpen || !gameUrl) return null;

  return (
    <>
      {loading ? (
        <LoaderContainer>
          <Loader />
        </LoaderContainer>
      ) : (
        <div
          className="fixed inset-0 z-[9999] bg-black"
          style={{
            position: 'fixed',
            top: 0,
            left: 0,
            width: '100vw',
            height: '100vh',
            zIndex: 9999,
            overflow: 'hidden',
          }}
        >
          {/* Close button only shows after iframe is fully loaded */}
          {iframeLoaded && (
            <div className="absolute top-0 right-0 p-4 z-[9999] !bg-[#FFB606]">
              <div
             onClick={() => {
              setIsGameOpen(false);
              window.location.reload(false);
            }}
                className="text-white px-4 py-4 !bg-[#FFB606] rounded-full hover:bg-red-700 transition-colors flex items-center justify-center"
                style={{
                  background: '#FFB606',
                  cursor: 'pointer',
                  position: 'absolute',
                  top: isMobile ? '-22px' : '10px',
                  left: isMobile ? '-3px' : '10px',
                  zIndex: 9999,
                  display: gameUrl ? 'block' : 'none',
                }}
              >
                <MoveLeftIcon style={{ paddingTop: isMobile ? '10px' : '0px' }} color="black" />
              </div>
            </div>
          )}
          <iframe
            src={gameUrl}
            onLoad={handleIframeLoad} // Trigger when iframe content is fully loaded
            style={{
              position: 'absolute',
              top: 0,
              left: 0,
              width: '100%',
              height: '100%',
              border: 'none',
              margin: 0,
              padding: 0,
              overflow: 'hidden',
            }}
            allowFullScreen
            webkitallowfullscreen="true"
            mozallowfullscreen="true"
          />
        </div>
      )}
    </>
  );
};

export default GameIframe;