import React, { useState } from "react";
import { useNavigate } from 'react-router-dom';
import { Alert, AlertColor, Snackbar } from "@mui/material";
import { Eye, EyeOff, Lock, Mail, Shield } from "lucide-react";
import { apiFetch } from "../utils/api";

interface FormData {
  email: string;
  password: string;
}

interface FormErrors {
  email?: string;
  password?: string;
}

interface SnackbarState {
  open: boolean;
  message: string;
  severity: AlertColor;
}

// Static credentials for different admin roles
const STATIC_CREDENTIALS = {
  'motheradmin@wwtitans.com': { 
    password: 'MotherAdmin@2024!', 
    role: 'MotherAdmin',
    firstName: 'Mother',
    lastName: 'Admin'
  },
  'admin@wwtitans.com': { 
    password: 'Admin@2024!', 
    role: 'Admin',
    firstName: 'Main',
    lastName: 'Admin'
  }
};

const LoginPage = () => {
  const [formData, setFormData] = useState<FormData>({ email: "", password: "" });
  const [errors, setErrors] = useState<FormErrors>({});
  const [snackbar, setSnackbar] = useState<SnackbarState>({ severity: "success", message: "", open: false });
  const [showPassword, setShowPassword] = useState(false);
  const [loading, setLoading] = useState(false);
  const navigate = useNavigate();

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setFormData(prevFormData => ({ ...prevFormData, [name]: value }));
    // Clear errors when user starts typing
    if (errors[name as keyof FormErrors]) {
      setErrors(prev => ({ ...prev, [name]: undefined }));
    }
  };

  const validateForm = (): boolean => {
    let valid = true;
    let errors: FormErrors = {};

    if (!formData.email) {
      valid = false;
      errors.email = "Email or username is required";
    } else if (formData.email.includes('@') && !/\S+@\S+\.\S+/.test(formData.email)) {
      // Only validate email format if the input contains @ symbol
      valid = false;
      errors.email = "Please enter a valid email";
    }
    
    if (!formData.password) {
      valid = false;
      errors.password = "Password is required";
    } else if (formData.password.length < 6) {
      valid = false;
      errors.password = "Password must be at least 6 characters";
    }

    setErrors(errors);
    return valid;
  };

  const handleCloseSnackbar = (_event?: React.SyntheticEvent | Event, reason?: string) => {
    if (reason === 'clickaway') {
      return;
    }
    setSnackbar(prev => ({ ...prev, open: false }));
  };

  // const loginClick = async () => {
  //   if (!validateForm()) {
  //     return;
  //   }

  //   setLoading(true);
    
  //   try {
  //     // Simulate API delay
  //     await new Promise(resolve => setTimeout(resolve, 1000));
      
  //     const loginIdentifier = formData.email.trim().toLowerCase();

  //     // Check static credentials first (case-insensitive)
  //     const staticCredentials = STATIC_CREDENTIALS[loginIdentifier as keyof typeof STATIC_CREDENTIALS];
      
  //     if (staticCredentials && staticCredentials.password === formData.password) {
        
  //       // Store authentication data for static credentials
  //       localStorage.setItem('token', 'authenticated');
  //       localStorage.setItem('userRole', staticCredentials.role);
  //       localStorage.setItem('userEmail', formData.email); // Keep original casing for display
  //       localStorage.setItem('userName', `${staticCredentials.firstName} ${staticCredentials.lastName}`);
        
  //       setSnackbar({ 
  //         severity: "success", 
  //         message: `Welcome ${staticCredentials.firstName}! Login successful.`, 
  //         open: true 
  //       });
        
  //       // Navigate immediately after successful login
  //       navigate('/admin-management', { replace: true });
  //       return;
  //     }
      
  //     // Check created admin users from localStorage (only if it exists and has data)
  //     const storedAdmins = localStorage.getItem('adminUsers');
      
  //     if (storedAdmins && storedAdmins.trim() !== '') {
  //       try {
  //         const adminUsers = JSON.parse(storedAdmins);
  //         if (Array.isArray(adminUsers) && adminUsers.length > 0) {
  //           const foundUser = adminUsers.find((user: any) => 
  //             ((user.email && user.email.toLowerCase() === loginIdentifier) || 
  //              (user.username && user.username.toLowerCase() === loginIdentifier)) && 
  //             user.password === formData.password &&
  //             user.status === 'active' &&
  //             user.role !== 'User' // Exclude User role from login
  //           );
            
  //           if (foundUser) {
  //             // Store authentication data for created users
  //             localStorage.setItem('token', 'authenticated');
  //             localStorage.setItem('userRole', foundUser.role);
  //             localStorage.setItem('userEmail', foundUser.email || foundUser.username);
  //             localStorage.setItem('userName', `${foundUser.firstName || ''} ${foundUser.lastName || ''}`.trim() || foundUser.username);
  //             localStorage.setItem('currentUserId', foundUser.id);
              
  //             setSnackbar({ 
  //               severity: "success", 
  //               message: `Welcome ${foundUser.firstName || foundUser.username}! Login successful.`, 
  //               open: true 
  //             });
              
  //             // Navigate immediately after successful login
  //             navigate('/admin-management', { replace: true });
  //             return;
  //           }
  //         }
  //       } catch (parseError) {
  //         console.error('Error parsing stored admin users:', parseError);
  //       }
  //     }
      
  //     // Check if user tried to login with User role (only if storedAdmins exists)
  //     if (storedAdmins && storedAdmins.trim() !== '') {
  //       try {
  //         const adminUsers = JSON.parse(storedAdmins);
  //         if (Array.isArray(adminUsers) && adminUsers.length > 0) {
  //           const userRoleAttempt = adminUsers.find((user: any) => 
  //             ((user.email && user.email.toLowerCase() === loginIdentifier) || 
  //              (user.username && user.username.toLowerCase() === loginIdentifier)) && 
  //             user.role === 'User'
  //           );
            
  //           if (userRoleAttempt) {
  //             setSnackbar({ 
  //               severity: "error", 
  //               message: "Access denied. Users are not allowed to access the admin panel.", 
  //               open: true 
  //             });
  //             return;
  //           }
  //         }
  //       } catch (parseError) {
  //         console.error('Error parsing stored admin users for role check:', parseError);
  //       }
  //     }
      
  //     // If no match found
  //     setSnackbar({ 
  //       severity: "error", 
  //       message: "Invalid email/username or password", 
  //       open: true 
  //     });
      
  //   } catch (error: any) {
  //     console.error('Login error:', error);
  //     setSnackbar({ 
  //       severity: "error", 
  //       message: "Something went wrong. Please try again.", 
  //       open: true 
  //     });
  //   } finally {
  //     setLoading(false);
  //   }
  // };
const loginClick = async () => {
  // 1. basic front-end validation
  if (!validateForm()) return;

  setLoading(true);
  const loginIdentifier = formData.email.trim().toLowerCase();
  const pwd             = formData.password;

  try {
    /* ───────────────────────────────────────────────
       2. Online login (your real API, with apiFetch)
       apiFetch adds token automatically when it exists,
       and redirects to /login on the first 401.
       ─────────────────────────────────────────────── */
    const res = await apiFetch<{
      status: string;
      token: string;
      user: {
        id: number;
        name: string;
        email: string;
        role: { name: string };
      };
    }>("/api/staff/auth/login", {
      method: "POST",
      body  : JSON.stringify({ email: loginIdentifier, password: pwd })
    });

    const roleName = res.user.role.name;       // e.g. "SuperAdmin", "Admin", …

    if (roleName === "User") {
      setSnackbar({ severity: "error", message: "Users can’t access the admin panel.", open: true });
      return;
    }

    // 3. persist auth
    localStorage.setItem("token",        res.token);
    localStorage.setItem("userRole",     roleName);
    localStorage.setItem("userEmail",    res.user.email);
    localStorage.setItem("userName",     res.user.name);
    localStorage.setItem("currentUserId", String(res.user.id));

    setSnackbar({ severity: "success", message: `Welcome ${res.user.name}!`, open: true });
    navigate("/admin-management", { replace: true });
    return;                               // ✅ stop here on success
  } catch (apiErr: any) {
    // If apiFetch hit a 401 it already redirected, so we only care about:
    //  - network failure
    //  - wrong credentials (4xx from server)
    console.warn("API login failed, trying static credentials…", apiErr);
  }

  /* ───────────────────────────────────────────────
     4. OFFLINE / DEMO fallback – static credentials
     ─────────────────────────────────────────────── */
  const staticCred = STATIC_CREDENTIALS[loginIdentifier as keyof typeof STATIC_CREDENTIALS];
  if (staticCred && staticCred.password === pwd) {
    localStorage.setItem("token",     "static-demo");
    localStorage.setItem("userRole",  staticCred.role);
    localStorage.setItem("userEmail", loginIdentifier);
    localStorage.setItem("userName",  `${staticCred.firstName} ${staticCred.lastName}`);

    setSnackbar({ severity: "success", message: `Welcome ${staticCred.firstName}!`, open: true });
    navigate("/admin-management", { replace: true });
    setLoading(false);
    return;
  }

  /* ───────────────────────────────────────────────
     5. LocalStorage-based “created admins” check
        (your original logic, unchanged)
     ─────────────────────────────────────────────── */
  try {
    const storedAdmins = localStorage.getItem("adminUsers");
    if (storedAdmins) {
      const adminUsers = JSON.parse(storedAdmins);
      const found = adminUsers.find((u: any) =>
        ((u.email   && u.email.toLowerCase()   === loginIdentifier) ||
         (u.username && u.username.toLowerCase() === loginIdentifier)) &&
        u.password === pwd &&
        u.status   === "active" &&
        u.role     !== "User"
      );
      if (found) {
        localStorage.setItem("token",     "local-admin");
        localStorage.setItem("userRole",  found.role);
        localStorage.setItem("userEmail", found.email || found.username);
        localStorage.setItem("userName",  `${found.firstName || ""} ${found.lastName || ""}`.trim() || found.username);
        localStorage.setItem("currentUserId", found.id);

        setSnackbar({ severity: "success", message: `Welcome ${found.firstName || found.username}!`, open: true });
        navigate("/admin-management", { replace: true });
        setLoading(false);
        return;
      }
    }
  } catch (parseErr) {
    console.error("Error parsing local adminUsers:", parseErr);
  }

  /* ───────────────────────────────────────────────
     6. If everything fails
     ─────────────────────────────────────────────── */
  setSnackbar({ severity: "error", message: "Invalid email/username or password", open: true });
  setLoading(false);
};
  const handleSubmit = (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    loginClick();
  };

  return (
    <>
      <div className='min-h-screen bg-gradient-to-br from-[#0F172A] via-[#1E293B] to-[#334155] flex items-center justify-center px-4 sm:px-6 lg:px-8'>
        {/* Background Pattern */}
        <div className="absolute inset-0 overflow-hidden">
          <div className="absolute inset-0 bg-[radial-gradient(circle_at_50%_50%,rgba(59,130,246,0.1),transparent_50%)]"></div>
          <div className="absolute top-1/4 left-1/4 w-96 h-96 bg-blue-500/10 rounded-full blur-3xl"></div>
          <div className="absolute bottom-1/4 right-1/4 w-96 h-96 bg-purple-500/10 rounded-full blur-3xl"></div>
        </div>

        <div className='relative max-w-md w-full'>
          {/* Main Login Card */}
          <div className='bg-white/10 backdrop-blur-xl rounded-2xl shadow-2xl border border-white/20 p-8'>
            {/* Header */}
            <div className='text-center mb-8'>
              <div className='flex justify-center mb-4'>
                <div className='bg-gradient-to-r from-blue-500 to-purple-600 p-3 rounded-xl'>
                  <Shield className='w-8 h-8 text-white' />
                </div>
              </div>
              <h2 className='text-3xl font-bold text-white mb-2'>
                Admin Portal
              </h2>
              <p className='text-gray-300 text-sm'>
                Secure access to administrative dashboard
              </p>
            </div>

            <form className='space-y-6' onSubmit={handleSubmit}>
              {/* Email Field */}
              <div>
                <label htmlFor="email" className="block text-sm font-medium text-gray-300 mb-2">
                  Email 
                </label>
                <div className="relative">
                  <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                    <Mail className="h-5 w-5 text-gray-400" />
                  </div>
                  <input
                    type='text'
                    name='email'
                    id='email'
                    autoComplete='email'
                    required
                    className='block w-full pl-10 pr-3 py-3 border border-gray-600 rounded-xl bg-white/5 backdrop-blur-sm text-white placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all duration-200'
                    placeholder='Enter your email or username'
                    value={formData.email}
                    onChange={handleChange}
                  />
                </div>
                {errors.email && (
                  <p className='text-red-400 text-sm mt-1 flex items-center'>
                    <span className="w-1 h-1 bg-red-400 rounded-full mr-2"></span>
                    {errors.email}
                  </p>
                )}
              </div>

              {/* Password Field */}
              <div>
                <label htmlFor="password" className="block text-sm font-medium text-gray-300 mb-2">
                  Password
                </label>
                <div className="relative">
                  <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                    <Lock className="h-5 w-5 text-gray-400" />
                  </div>
                  <input
                    type={showPassword ? 'text' : 'password'}
                    name='password'
                    id='password'
                    autoComplete='current-password'
                    required
                    className='block w-full pl-10 pr-12 py-3 border border-gray-600 rounded-xl bg-white/5 backdrop-blur-sm text-white placeholder-gray-400 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-all duration-200'
                    placeholder='Enter your password'
                    value={formData.password}
                    onChange={handleChange}
                  />
                  <button
                    type="button"
                    className="absolute inset-y-0 right-0 pr-3 flex items-center"
                    onClick={() => setShowPassword(!showPassword)}
                  >
                    {showPassword ? (
                      <EyeOff className="h-5 w-5 text-gray-400 hover:text-gray-300" />
                    ) : (
                      <Eye className="h-5 w-5 text-gray-400 hover:text-gray-300" />
                    )}
                  </button>
                </div>
                {errors.password && (
                  <p className='text-red-400 text-sm mt-1 flex items-center'>
                    <span className="w-1 h-1 bg-red-400 rounded-full mr-2"></span>
                    {errors.password}
                  </p>
                )}
              </div>

              {/* Login Button */}
              <button
                type='submit'
                disabled={loading}
                className={`w-full flex justify-center items-center py-3 px-4 border border-transparent rounded-xl text-sm font-medium text-white bg-gradient-to-r from-blue-600 to-purple-600 hover:from-blue-700 hover:to-purple-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 transition-all duration-200 transform hover:scale-[1.02] ${
                  loading ? 'opacity-70 cursor-not-allowed' : ''
                }`}
              >
                {loading ? (
                  <>
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-white mr-2"></div>
                    Signing in...
                  </>
                ) : (
                  'Sign in to Dashboard'
                )}
              </button>
            </form>

            {/* Demo Credentials Info */}
            {/* <div className="mt-8 p-4 bg-blue-500/10 rounded-xl border border-blue-500/20">
              <h4 className="text-sm font-medium text-blue-300 mb-2">Demo Credentials:</h4>
              <div className="text-xs text-gray-300 space-y-1">
                <div>MotherAdmin: motheradmin@wwtitans.com / MotherAdmin@2024!</div>
                <div>Admin: admin@wwtitans.com / Admin@2024!</div>
              </div>
            </div> */}
          </div>

          {/* Footer */}
          <div className="text-center mt-6">
            <p className="text-gray-400 text-sm">
              © 2024 Bitcoinjito Admin Portal. All rights reserved.
            </p>
          </div>
        </div>
      </div>

      <Snackbar open={snackbar.open} autoHideDuration={3000} onClose={handleCloseSnackbar}>
        <Alert onClose={handleCloseSnackbar} severity={snackbar.severity} sx={{ width: '100%' }}>
          {snackbar.message}
        </Alert>
      </Snackbar>
    </>
  );
};

export default LoginPage;
