import React, { useEffect, useState } from "react";
import { apiFetch } from "../utils/api";
import { BOOLEAN_KEYS } from "../constants/booleanKeys";

type BoolKey = typeof BOOLEAN_KEYS[number];
type Config = Record<string, any>;
type Draft = Partial<Record<BoolKey, boolean>>;

// Grouped keys for better organization
const GROUPED_KEYS = {
  "Site Features": [
    "sports", "casino", "lotto", "vipclub", "clubmembership", "bonus", "affiliate",
    "crash", "originals", "livegames", "slotsgames", "alllivegames", "allslotsgames",
    "lotterygames", "indiangames", "cards", "instantgames", "esports", "provablyfair"
  ],
  "Cryptocurrencies": [
    "btc", "eth", "ltc", "bch", "usdt", "trx", "doge", "ada", "xrp", "bnb", 
    "usdp", "nexo", "mkr", "tusd", "usdc", "busd", "nc", "npr", "shib", "matic", "sc"
  ],
  "Game Providers": [
    "spribe", "evolution", "pragmaticslots", "pragmaticlive", "ideal",
    "microgaming", "pgsoft", "hacksawgaming", "jili", "netent"
  ]
};

type BoolSwitchProps = {
  label: string;
  checked: boolean;
  onChange: React.ChangeEventHandler<HTMLInputElement>;
};

const BoolSwitch = ({ label, checked, onChange }: BoolSwitchProps) => (
  <div className="flex items-center justify-between py-2">
    <span className="text-gray-700 capitalize">{label}</span>
    <label className="relative inline-flex items-center cursor-pointer">
      <input
        type="checkbox"
        checked={checked}
        onChange={onChange}
        className="sr-only peer"
      />
      <div className="w-11 h-6 bg-gray-200 rounded-full peer peer-checked:bg-blue-600 transition-colors">
        <div className="w-5 h-5 bg-white rounded-full shadow transform transition-transform peer-checked:translate-x-5" />
      </div>
    </label>
  </div>
);

export default function SiteConfig() {
  const [cfg, setCfg] = useState<Config | null>(null);
  const [draft, setDraft] = useState<Draft>({});
  const [busy, setBusy] = useState(false);

  useEffect(() => {
    apiFetch<Config>("/api/admin/config/global").then(setCfg);
  }, []);

  const toggle = (k: BoolKey) => {
    if (!cfg) return;
    setCfg({ ...cfg, [k]: !cfg[k] });
    setDraft({ ...draft, [k]: !cfg[k] });
  };

  const save = async () => {
    if (!Object.keys(draft).length) return;
    setBusy(true);
    try {
      const upd = await apiFetch<Config>("/api/admin/config/global", {
        method: "PUT",
        body: JSON.stringify(draft)
      });
      setCfg(upd);
      setDraft({});
    } finally {
      setBusy(false);
    }
  };

  if (!cfg) return (
    <div className="flex items-center justify-center min-h-screen">
      <div className="text-gray-600 text-lg animate-pulse">Loading...</div>
    </div>
  );

  return (
    	<div className='ml-0 lg:ml-[256px]'>
    <div className="max-w-7xl mx-auto p-6 space-y-8 bg-gray-50 min-h-screen">
      <div className="flex justify-between items-center">
        <h1 className="text-3xl font-bold text-gray-900">Global Configuration</h1>
        <button
          onClick={save}
          disabled={!Object.keys(draft).length || busy}
          className="px-6 py-2 bg-blue-600 text-white rounded-lg font-semibold 
            hover:bg-blue-700 transition-colors disabled:opacity-50 
            disabled:cursor-not-allowed flex items-center space-x-2"
        >
          {busy && (
            <svg className="animate-spin h-5 w-5 text-white" viewBox="0 0 24 24">
              <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" />
              <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4z" />
            </svg>
          )}
          <span>{busy ? "Saving..." : "Save Changes"}</span>
        </button>
      </div>

      {Object.entries(GROUPED_KEYS).map(([group, keys]) => (
        <div key={group} className="bg-white rounded-lg shadow-md p-6">
          <h2 className="text-xl font-semibold text-gray-800 mb-4">{group}</h2>
          <div className="grid gap-2 sm:grid-cols-2 lg:grid-cols-3">
            {keys.map(k => (
              <BoolSwitch
                key={k}
                label={k}
                checked={!!cfg[k]}
                onChange={() => toggle(k as BoolKey)}
              />
            ))}
          </div>
        </div>
      ))}
    </div>
    </div>
  );
}