/*  BankingTab.tsx  (drop-in replacement with search by USERNAME & ROLE)  */
import React, { useEffect, useMemo, useState } from "react";
import { Snackbar, Alert, AlertColor } from "@mui/material";
import { apiFetch } from "../../utils/api";

/* ─────────────────────────────────────────── CONSTANTS */
export const ROLE_HIERARCHY = {
  SuperAdmin: { level: 0, color: "bg-black",      description: "Supreme admin" },
  Admin     : { level: 1, color: "bg-red-500",    description: "Admin"         },
  SubAdmin  : { level: 2, color: "bg-orange-500", description: "Sub-Admin"     },
  Master    : { level: 3, color: "bg-blue-500",   description: "Master"        },
  Agent     : { level: 4, color: "bg-green-500",  description: "Agent"         },
  SubAgent  : { level: 5, color: "bg-purple-500", description: "Sub-Agent"     },
  User      : { level: 6, color: "bg-gray-500",   description: "Player"        }
} as const;



/* ─────────────────────────────────────────── TYPES */
export interface Row {
  id        : number;
  name      : string;                          // username (player) or full name
  role      : keyof typeof ROLE_HIERARCHY;
  balance   : number;
  parent_id : number | null;
  phone?    : string | null;
  country?  : string | null;
}

interface SnackbarState {
  open     : boolean;
  message  : string;
  severity : AlertColor;
}

/* ─────────────────────────────────────────── HELPERS */
const toArray = (x: any) =>
  Array.isArray(x)     ? x :
  Array.isArray(x.data) ? x.data : [];

const BalanceCard: React.FC<{ title: string; balance: number; currency?: string }> = ({
  title, balance, currency = "INR"
}) => (
  <div className="bg-white p-3 rounded-lg text-gray-800 text-center border border-gray-200 shadow-sm flex flex-col justify-center">
    <h4 className="text-sm md:text-md font-semibold mb-2 text-gray-600">{title}</h4>
    <div className="bg-blue-600 rounded-full px-2 py-2">
      <p className="text-white font-bold text-sm md:text-lg truncate">{`${currency} (${balance.toFixed(2)})`}</p>
    </div>
  </div>
);

/* ─────────────────────────────────────────── COMPONENT */
export default function BankingTab() {
  /* who am I? */
  const myRole = (localStorage.getItem("userRole") as keyof typeof ROLE_HIERARCHY) || "SuperAdmin";
  const myId   = Number(localStorage.getItem("currentUserId") || 1);
  const myLvl  = ROLE_HIERARCHY[myRole].level;

  /* state */
  const [rows,    setRows]    = useState<Row[]>([]);
  const [rollup,  setRollup]  = useState<Record<number, any>>({});
  const [myBal,   setMyBal]   = useState(0);
  const [depVal,  setDepVal]  = useState<Record<number, string>>({});
  const [wdVal,   setWdVal]   = useState<Record<number, string>>({});
  const [snack,   setSnack]   = useState<SnackbarState>({ open: false, message: "", severity: "success" });
  const [loading, setLoading] = useState(true);
  const [plData, setPlData] = useState<Record<number, number>>({});
  const [plLoading, setPlLoading] = useState<Record<number, boolean>>({});

  // search state (new)
  const [q, setQ] = useState("");             // free-text search for USERNAME
  const [roleFilter, setRoleFilter] = useState<string>("All"); // role dropdown

  /* fetch -------------------------------------------------------------------------------- */
  const refresh = async () => {
    try {
      setLoading(true);
      const [list, rollAll, self] = await Promise.all([
        apiFetch<Row[]>("/api/staff"),
        apiFetch<any[]>(`/api/staff/rollupStaff/${myId}`),
        apiFetch<any>(`/api/staff/rollup/${myId}`)
      ]);

      setRows(toArray(list));
      const rows1=toArray(list);
      setMyBal(Number(self.data?.own_balance || 0));
      const arr = toArray(rollAll);             // make sure it’s an array
      const map = Object.fromEntries(
        arr.map((o: any) => [Number(o.staff_id), o])
      );
      setRollup(map);
      // Fetch P/L for each row
      // Skip P/L fetching if rows is empty
      if (rows1.length === 0) {
        console.warn("[refresh] No rows to fetch P/L for");
        setPlData({});
        setPlLoading({});
        return;
      }

      // Fetch P/L for each row
      const plPromises = await rows1.map(async (row: { id: any; role: string; }) => {
        setPlLoading((prev) => ({ ...prev, [Number(row.id)]: true }));
        const endpoint = row.role === "User" ? "/api/deposit/user-pl" : "/api/deposit/staff-pl";
        const body = row.role === "User" ? { userId: row.id } : { staffId: row.id };
        console.log(`[refresh] Fetching P/L for ${row.role} ${row.id} at ${endpoint} with body:`, body); // Debug: Log API call
        try {
          const response = await apiFetch(endpoint, {
            method: "POST",
            body: JSON.stringify(body)
          });
          console.log(`[refresh] P/L response for ${row.role} ${row.id}:`, response); // Debug: Log response
          const profitLoss = Number(response?.data?.profitLoss ?? 0);
          if (isNaN(profitLoss)) {
            console.warn(`[refresh] Invalid profitLoss for ${row.role} ${row.id}:`, response);
            return { id: Number(row.id), profitLoss: 0 };
          }
          return { id: Number(row.id), profitLoss };
        } catch (err: any) {
          console.error(`[refresh] Failed to fetch P/L for ${row.role} ${row.id}:`, err.message || err); // Debug: Log error
          return { id: Number(row.id), profitLoss: 0 };
        } finally {
          setPlLoading((prev) => ({ ...prev, [Number(row.id)]: false }));
        }
      });

      const plResults = await Promise.all(plPromises);
      const plMap = Object.fromEntries(
        plResults.map(({ id, profitLoss }) => [Number(id), Number(profitLoss)])
      );
      console.log("[refresh] P/L map:", plMap); // Debug: Log final P/L map
      setPlData(plMap);
    } catch (err: any) {
      setSnack({ open: true, severity: "error", message: err.message || "Server error" });
    } finally {
      setLoading(false);
    }
  };
  useEffect(() => { refresh(); }, []);

  /* derived ------------------------------------------------------------------------------ */
  const roleTotals = useMemo(() => {
    const tot: Record<string, number> = {};
    Object.keys(ROLE_HIERARCHY).forEach(r => (tot[r] = 0));
    rows.forEach(r => { tot[r.role] += r.balance; });
    return tot;
  }, [rows]);

  const myChildren  = useMemo(() => rows.filter(r => r.parent_id === myId), [rows, myId]);
  const manageable  = useMemo(() => rows.filter(r => ROLE_HIERARCHY[r.role].level > myLvl), [rows, myLvl]);
  const haveFunds = (amt: number) => {
    const balNum   = Number(myBal);        // make 100 % sure we compare numbers
    const amtNum   = Number(amt);
    console.log("[guard] bal =", balNum, "amt =", amtNum);   // ①

    if (Number.isNaN(balNum) || Number.isNaN(amtNum)) {
      console.warn("[guard] one of the values is NaN!");
      return false;                       // treat NaN as “no funds”
    }
    return balNum >= amtNum;
  };

  /* transfer ----------------------------------------------------------------------------- */
  const doTransfer = async (opts: {
    toId: number; toRole: Row["role"]; amount: number; direction: "deposit" | "withdraw";
  }) => {
    const { toId, toRole, amount, direction } = opts;
    if (!amount || amount <= 0) return setSnack({ open: true, severity: "error", message: "Enter valid amount" });
    console.log(direction, toRole, amount, toId);
    console.log("[action] direction =", direction);            // ②
    if (direction === "deposit" && !haveFunds(amount)) {
      console.log("[action] blocked – insufficient");          // ③
      setSnack({
        open     : true,
        severity : "error",
        message  : "Insufficient balance"
      });
      return;                         // ← hard stop
    }
    if (direction === "withdraw") {
      const child = rows.find(r => r.id === toId);
      if (!child) return;
      if (amount > child.balance)
        return setSnack({ open: true, severity: "error", message: `${child.name} has only ₹${child.balance.toFixed(2)}` });
    }

    const body = {
      toType   : toRole === "User" ? "user" : "staff",
      toId,
      amount,
      direction
    };

    try {
      await apiFetch("/api/staff/transfer", { method: "POST", body: JSON.stringify(body) });
      setSnack({ open: true, severity: "success", message: `${direction === "deposit" ? "Sent" : "Received"} ₹${amount}` });
      setDepVal(v => ({ ...v, [toId]: "" }));
      setWdVal(v => ({ ...v, [toId]: "" }));
      refresh();
      window.dispatchEvent(new Event("balance-changed"));
    } catch (err: any) {
      setSnack({ open: true, severity: "error", message: err.message || "Transfer failed" });
    }
  };

  /* loading guard */
  if (loading) return <div className="text-center py-16 text-gray-400">Loading…</div>;

  /* --- SEARCH/ FILTER LOGIC (applies both for SubAdmin and others) --- */
  const normalize = (s: string) => (s || "").toString().toLowerCase().trim();
  const matchesFilter = (r: Row) => {
    const qn = normalize(q);
    const roleN = normalize(roleFilter);
    const nameMatch = qn === "" || normalize(r.name).includes(qn);
    const roleMatch = roleN === "all" || roleFilter === "All" || normalize(r.role).includes(roleN);
    return nameMatch && roleMatch;
  };

  /* view for Sub-Admin only -------------------------------------------------------------- */
  if (myRole === "SubAdmin") {
    const childBal = myChildren.reduce((s, c) => s + c.balance, 0);
    const filteredChildren = myChildren.filter(matchesFilter);

    return (
      <Section>
        {/* search box */}
        <div className="flex items-center gap-2 mb-4">
          <input
            className="border rounded px-2 py-1 w-60"
            placeholder="Search by username..."
            value={q}
            onChange={e => setQ(e.target.value)}
          />
          <select className="border rounded px-2 py-1" value={roleFilter} onChange={e => setRoleFilter(e.target.value)}>
            <option>All</option>
            {Object.keys(ROLE_HIERARCHY).map(r => <option key={r}>{r}</option>)}
          </select>
          <button className="px-3 py-1 bg-gray-200 rounded" onClick={() => { setQ(""); setRoleFilter("All"); }}>Clear</button>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
          <BalanceCard title="Your Balance"             balance={myBal} />
          <BalanceCard title="Total Child Balance"      balance={childBal} />
          <BalanceCard title="Available Operations"     balance={myBal} />
        </div>

        <BankTable
          rows        ={filteredChildren}
          rollup      ={rollup}
          depVal      ={depVal}
          wdVal       ={wdVal}
          setDepVal   ={setDepVal}
          setWdVal    ={setWdVal}
          onTransfer  ={doTransfer}
          plData      ={plData}
          plLoading   ={plLoading}
        />

        <Snack {...snack} onClose={() => setSnack(s => ({ ...s, open: false }))} />
      </Section>
    );
  }

  /* everyone else ----------------------------------------------------------------------- */
  const visible = myRole === "SuperAdmin" ? rows.filter(r => r.role !== "SuperAdmin") : manageable;
  const filteredVisible = visible.filter(matchesFilter);
  const isAdmin = myRole === "Admin";

  return (
    <Section>
      {/* search box */}
      <div className="flex items-center gap-2 mb-4">
        <input
          className="border rounded px-2 py-1 w-60"
          placeholder="Search by username..."
          value={q}
          onChange={e => setQ(e.target.value)}
        />
        <select className="border rounded px-2 py-1" value={roleFilter} onChange={e => setRoleFilter(e.target.value)}>
          <option>All</option>
          {Object.keys(ROLE_HIERARCHY).map(r => <option key={r}>{r}</option>)}
        </select>
        <button className="px-3 py-1 bg-gray-200 rounded" onClick={() => { setQ(""); setRoleFilter("All"); }}>Clear</button>
      </div>

      {/* summary cards */}
      {isAdmin ? (
        <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-5 gap-4 mb-6">
          <BalanceCard title="Total Balance"       balance={roleTotals.User} />
          <BalanceCard title="Remaining Balance"   balance={roleTotals.Admin} />
          <BalanceCard title="Total Agent Balance" balance={roleTotals.Agent} />
          <BalanceCard title="Total Exposure"      balance={0} />
          <BalanceCard title="Total Admin"         balance={roleTotals.Admin} />
        </div>
      ) : (
        <>
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-5 gap-4 mb-4">
            <BalanceCard title="Total User Balance"      balance={roleTotals.User} />
            <BalanceCard title="Total Sub-Agent Balance" balance={roleTotals.SubAgent} />
            <BalanceCard title="Total Agent Balance"     balance={roleTotals.Agent} />
            <BalanceCard title="Total Master Balance"    balance={roleTotals.Master} />
            <BalanceCard title="Total Sub-Admin Balance" balance={roleTotals.SubAdmin} />
          </div>
          <div className="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-5 gap-4 mb-6">
            <BalanceCard title="Total Admin Balance" balance={roleTotals.Admin} />
          </div>
        </>
      )}

      <BankTable
        rows        ={filteredVisible}
        rollup      ={rollup}
        depVal      ={depVal}
        wdVal       ={wdVal}
        setDepVal   ={setDepVal}
        setWdVal    ={setWdVal}
        onTransfer  ={doTransfer}
        plData      ={plData}
        plLoading   ={plLoading}
      />

      <Snack {...snack} onClose={() => setSnack(s => ({ ...s, open: false }))} />
    </Section>
  );
}

/* ────────────────────────── PRESENTATIONAL SUB-COMPONENTS ───────────── */

const Section: React.FC<{ children: React.ReactNode }> = ({ children }) => (
  <div className="p-2 max-w-7xl mx-auto">{children}</div>
);

const Snack: React.FC<SnackbarState & { onClose: () => void }> = ({ open, severity, message, onClose }) => (
  <Snackbar open={open} autoHideDuration={3000} onClose={onClose}>
    <Alert severity={severity} onClose={onClose}>{message}</Alert>
  </Snackbar>
);

/* table (layout unchanged) ------------------------------------------- */
interface TableProps {
  rows: Row[];
  rollup: Record<number, any>;
  depVal: Record<number, string>;
  wdVal:  Record<number, string>;
  setDepVal: React.Dispatch<React.SetStateAction<Record<number, string>>>;
  setWdVal:  React.Dispatch<React.SetStateAction<Record<number, string>>>;
  onTransfer: (o: { toId: number; toRole: Row["role"]; amount: number; direction: "deposit" | "withdraw" }) => void;
  plData: Record<number, number>;
  plLoading?: Record<number, boolean>;
}

const BankTable: React.FC<TableProps> = ({
  rows, rollup, depVal, wdVal, setDepVal, setWdVal, onTransfer,plData, plLoading = {}
}) => (
  <div className="bg-white rounded-xl shadow-sm border overflow-x-auto">
    <table className="w-full">
      <thead className="bg-gray-50">
        {["USERNAME","ROLE","AVAILABLE BALANCE","BALANCE IN DL","EXPOSURE IN DL",
          "DEPOSIT","WITHDRAW","CREDIT REF.","REF P/L","REMARK","ALL"]
          .map((h, i) => (
            <th key={i} className="border px-2 py-2 text-xs font-bold text-gray-700 uppercase">{h}</th>
          ))}
      </thead>
      <tbody className="divide-y">
        {rows.map(r => {
          const dep = Number(depVal[r.id] || 0);
          const wdr = Number(wdVal[r.id] || 0);

          return (
            <tr key={r.id} className="hover:bg-gray-50 text-xs">
              <td className="border px-2 py-2 underline text-blue-700">{r.name}</td>
              <td className="border px-2 py-2 font-semibold">{r.role}</td>
              <td className="border px-2 py-2 font-mono">{r.balance.toFixed(2)}</td>
              <td className="border px-2 py-2">{(rollup[r.id]?.downline_balance ?? 0).toFixed(2)}</td>
              <td className="border px-2 py-2"></td>

              {/* deposit */}
              <td className="border px-2 py-2">
                <input
                  type="number"
                  min={0}
                  value={depVal[r.id] || ""}
                  onChange={e => setDepVal(v => ({ ...v, [r.id]: e.target.value }))}
                  className="w-16 border rounded px-1 mr-1"
                />
                <button
                  className="px-2 py-1 rounded text-white bg-blue-600 hover:bg-blue-700"
                  disabled={!dep}
                  onClick={() => onTransfer({ toId: r.id, toRole: r.role, amount: dep, direction: "deposit" })}
                >
                  Deposit
                </button>
              </td>

              {/* withdraw */}
              <td className="border px-2 py-2">
                <input
                  type="number"
                  min={0}
                  value={wdVal[r.id] || ""}
                  onChange={e => setWdVal(v => ({ ...v, [r.id]: e.target.value }))}
                  className="w-16 border rounded px-1 mr-1"
                />
                <button
                  className="px-2 py-1 rounded text-white bg-red-600 hover:bg-red-700"
                  disabled={!wdr}
                  onClick={() => onTransfer({ toId: r.id, toRole: r.role, amount: wdr, direction: "withdraw" })}
                >
                  Withdraw
                </button>
              </td>

              {/* untouched cols */}
              <td className="border px-2 py-2" />
             <td className="border px-2 py-2 font-mono">{plLoading[r.id] ? "Loading..." : plData[r.id] !== undefined ? plData[r.id].toFixed(2) : "N/A"}</td>
              <td className="border px-2 py-2" />
              <td className="border px-2 py-2" />
            </tr>
          );
        })}
      </tbody>
    </table>
  </div>
);
