// src/pages/DashboardTab.tsx
import React from "react";
import { 
  Box, 
  Card, 
  CardContent, 
  Typography, 
  Skeleton, 
  Stack,
  Avatar,
  Chip,
  Paper
} from "@mui/material";
import TrendingUpIcon from "@mui/icons-material/TrendingUp";
import PeopleIcon from "@mui/icons-material/People";
import AccountBalanceIcon from "@mui/icons-material/AccountBalance";
import ArrowUpwardIcon from "@mui/icons-material/ArrowUpward";
import {
  BarChart as ReBarChart,
  Bar, 
  XAxis, 
  YAxis, 
  Tooltip, 
  ResponsiveContainer,
  Area,
  AreaChart
} from "recharts";
import { useTree } from "../../hooks/useTree";

export default function DashboardTab() {
  const { metrics, analytics, loading } = useTree();

  if (loading || !metrics || !analytics) {
    return (
      <Box sx={{ p: 3 }}>
        <Stack spacing={3}>
          <Box sx={{ display: 'grid', gridTemplateColumns: { xs: '1fr', sm: 'repeat(3, 1fr)' }, gap: 2 }}>
            {[1, 2, 3].map((i) => (
              <Skeleton key={i} variant="rounded" height={120} />
            ))}
          </Box>
          <Box sx={{ display: 'grid', gridTemplateColumns: { xs: '1fr', lg: '1fr 1fr' }, gap: 3 }}>
            <Skeleton variant="rounded" height={300} />
            <Skeleton variant="rounded" height={300} />
          </Box>
        </Stack>
      </Box>
    );
  }

  return (
    <Box sx={{ 
      p: 3, 
      backgroundColor: '#f8fafc',
      minHeight: '100vh'
    }}>
      {/* Stats Grid */}
      <Box sx={{
        display: 'grid',
        gridTemplateColumns: { xs: '1fr', sm: 'repeat(3, 1fr)' },
        gap: 2.5,
        mb: 4
      }}>
        <StatCard
          icon={<PeopleIcon />}
          label="Members"
          value={metrics.staff_cnt}
          trend="+12%"
          trendUp={true}
          color="#6366f1"
          bgColor="#f0f0ff"
        />
        <StatCard
          icon={<TrendingUpIcon />}
          label="Active Players"
          value={metrics.player_cnt}
          trend="+8%"
          trendUp={true}
          color="#10b981"
          bgColor="#f0fdf4"
        />
        <StatCard
          icon={<AccountBalanceIcon />}
          label="Total Balance"
          value={`₹${Number(metrics.bank).toLocaleString()}`}
          trend="+24%"
          trendUp={true}
          color="#f59e0b"
          bgColor="#fffbeb"
        />
      </Box>

      {/* Charts Section */}
      <Box sx={{
        display: 'grid',
        gridTemplateColumns: { xs: '1fr', lg: '1fr 1fr' },
        gap: 3
      }}>
        <ChartCard
          title="Members Growth"
          subtitle="Last 7 days performance"
          data={analytics.staffTrend}
          color="#6366f1"
          type="area"
        />
        <ChartCard
          title="Player Activity"
          subtitle="Daily active players"
          data={analytics.playerTrend}
          color="#10b981"
          type="bar"
        />
      </Box>
    </Box>
  );
}

/* ═════════════════════ Components ═════════════════════ */

function StatCard({ 
  icon, 
  label, 
  value, 
  trend, 
  trendUp, 
  color, 
  bgColor 
}: {
  icon: React.ReactNode;
  label: string;
  value: string | number;
  trend: string;
  trendUp: boolean;
  color: string;
  bgColor: string;
}) {
  return (
    <Card
      elevation={0}
      sx={{
        p: 3,
        borderRadius: 3,
        border: '1px solid #e2e8f0',
        background: 'linear-gradient(135deg, #ffffff 0%, #f8fafc 100%)',
        transition: 'all 0.3s ease',
        cursor: 'pointer',
        '&:hover': {
          transform: 'translateY(-4px)',
          boxShadow: '0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04)',
          border: `1px solid ${color}20`
        }
      }}
    >
      <Box sx={{ display: 'flex', alignItems: 'flex-start', justifyContent: 'space-between', mb: 2 }}>
        <Avatar
          sx={{
            width: 56,
            height: 56,
            backgroundColor: bgColor,
            color: color,
            border: `1px solid ${color}20`
          }}
        >
          {icon}
        </Avatar>
        <Chip
          icon={<ArrowUpwardIcon sx={{ fontSize: 14 }} />}
          label={trend}
          size="small"
          sx={{
            backgroundColor: trendUp ? '#dcfce7' : '#fee2e2',
            color: trendUp ? '#16a34a' : '#dc2626',
            border: 'none',
            fontWeight: 600,
            fontSize: 12
          }}
        />
      </Box>

      <Typography
        variant="h4"
        sx={{
          fontWeight: 700,
          color: '#0f172a',
          mb: 0.5,
          fontSize: { xs: '1.75rem', sm: '2rem' }
        }}
      >
        {value}
      </Typography>

      <Typography
        variant="body2"
        sx={{
          color: '#64748b',
          fontWeight: 500,
          fontSize: 14
        }}
      >
        {label}
      </Typography>
    </Card>
  );
}

function ChartCard({ 
  title, 
  subtitle, 
  data, 
  color, 
  type = 'bar' 
}: {
  title: string;
  subtitle: string;
  data: { day: string; count: number }[];
  color: string;
  type?: 'bar' | 'area';
}) {
  return (
    <Paper
      elevation={0}
      sx={{
        p: 3,
        borderRadius: 3,
        border: '1px solid #e2e8f0',
        background: '#ffffff',
        height: 'fit-content'
      }}
    >
      <Box sx={{ mb: 3 }}>
        <Typography
          variant="h6"
          sx={{
            fontWeight: 700,
            color: '#0f172a',
            fontSize: 18,
            mb: 0.5
          }}
        >
          {title}
        </Typography>
        <Typography
          variant="body2"
          sx={{
            color: '#64748b',
            fontSize: 14
          }}
        >
          {subtitle}
        </Typography>
      </Box>

      <Box sx={{ height: 280 }}>
        <ResponsiveContainer width="100%" height="100%">
          {type === 'area' ? (
            <AreaChart data={data}>
              <defs>
                <linearGradient id={`gradient-${color}`} x1="0" y1="0" x2="0" y2="1">
                  <stop offset="5%" stopColor={color} stopOpacity={0.3} />
                  <stop offset="95%" stopColor={color} stopOpacity={0.05} />
                </linearGradient>
              </defs>
              <XAxis 
                dataKey="day" 
                stroke="#94a3b8"
                fontSize={12}
                tickLine={false}
                axisLine={false}
              />
              <YAxis 
                stroke="#94a3b8"
                fontSize={12}
                tickLine={false}
                axisLine={false}
                allowDecimals={false}
              />
              <Tooltip
                contentStyle={{
                  backgroundColor: '#ffffff',
                  border: '1px solid #e2e8f0',
                  borderRadius: 12,
                  boxShadow: '0 10px 15px -3px rgba(0, 0, 0, 0.1)',
                  color: '#0f172a'
                }}
                labelStyle={{ color: '#64748b', fontWeight: 500 }}
              />
              <Area
                type="monotone"
                dataKey="count"
                stroke={color}
                strokeWidth={3}
                fill={`url(#gradient-${color})`}
              />
            </AreaChart>
          ) : (
            <ReBarChart data={data}>
              <XAxis 
                dataKey="day" 
                stroke="#94a3b8"
                fontSize={12}
                tickLine={false}
                axisLine={false}
              />
              <YAxis 
                stroke="#94a3b8"
                fontSize={12}
                tickLine={false}
                axisLine={false}
                allowDecimals={false}
              />
              <Tooltip
                contentStyle={{
                  backgroundColor: '#ffffff',
                  border: '1px solid #e2e8f0',
                  borderRadius: 12,
                  boxShadow: '0 10px 15px -3px rgba(0, 0, 0, 0.1)',
                  color: '#0f172a'
                }}
                labelStyle={{ color: '#64748b', fontWeight: 500 }}
              />
              <Bar 
                dataKey="count" 
                fill={color}
                radius={[6, 6, 0, 0]}
              />
            </ReBarChart>
          )}
        </ResponsiveContainer>
      </Box>
    </Paper>
  );
}